<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Page;
use Illuminate\Support\Facades\File;

class AddMissingStructuredDataSeeder extends Seeder
{
    public function run(): void
    {
        echo "Adding Structured Data to Missing Pages...\n";
        echo "==========================================\n\n";

        // Get the structured data templates
        $servicesData = $this->getServicesStructuredData();
        $aboutData = $this->getAboutStructuredData();
        $writersData = $this->getWritersStructuredData();
        $guaranteesData = $this->getGuaranteesStructuredData();

        // Services index pages
        $serviceSlugs = ['services', 'services-1', 'services-2', 'services-3', 'services-4'];
        foreach ($serviceSlugs as $slug) {
            $page = Page::where('slug', $slug)->first();
            if ($page && $page->seo) {
                $page->seo->update(['structured_data' => $servicesData]);
                echo "✅ Updated: {$slug}\n";
            }
        }

        // About page
        $about = Page::where('slug', 'about')->first();
        if ($about && $about->seo) {
            $about->seo->update(['structured_data' => $aboutData]);
            echo "✅ Updated: about\n";
        }

        // Writers page
        $writers = Page::where('slug', 'writers')->first();
        if ($writers && $writers->seo) {
            $writers->seo->update(['structured_data' => $writersData]);
            echo "✅ Updated: writers\n";
        }

        // Guarantees page
        $guarantees = Page::where('slug', 'guarantees')->first();
        if ($guarantees && $guarantees->seo) {
            $guarantees->seo->update(['structured_data' => $guaranteesData]);
            echo "✅ Updated: guarantees\n";
        }

        // Clear caches
        \Illuminate\Support\Facades\Cache::flush();

        echo "\n";
        echo "Summary:\n";
        echo "========\n";
        $totalPages = Page::count();
        $withStructuredData = Page::whereHas('seo', function($q) {
            $q->whereNotNull('structured_data');
        })->count();
        
        echo "Total Pages: {$totalPages}\n";
        echo "Pages with Structured Data: {$withStructuredData}\n";
        echo "Coverage: " . round(($withStructuredData / $totalPages) * 100, 1) . "%\n";
        
        if ($totalPages === $withStructuredData) {
            echo "\n✅ ALL PAGES NOW HAVE STRUCTURED DATA!\n";
        }
    }

    private function getServicesStructuredData()
    {
        return [
            '@context' => 'https://schema.org',
            '@graph' => [
                [
                    '@type' => 'WebPage',
                    'name' => 'Academic Writing Services',
                    'url' => 'https://academicscribe.com/services',
                    'description' => 'Explore our comprehensive academic writing services including essays, research papers, dissertations, case studies, and more'
                ],
                [
                    '@type' => 'Service',
                    'name' => 'Academic Writing Services',
                    'serviceType' => 'Educational Services',
                    'provider' => [
                        '@type' => 'Organization',
                        'name' => 'Academic Scribe',
                        'url' => 'https://academicscribe.com'
                    ],
                    'areaServed' => 'Worldwide',
                    'hasOfferCatalog' => [
                        '@type' => 'OfferCatalog',
                        'name' => 'Academic Services',
                        'itemListElement' => [
                            [
                                '@type' => 'Offer',
                                'itemOffered' => [
                                    '@type' => 'Service',
                                    'name' => 'Essay Writing Service',
                                    'url' => 'https://academicscribe.com/services/essay-writing'
                                ]
                            ],
                            [
                                '@type' => 'Offer',
                                'itemOffered' => [
                                    '@type' => 'Service',
                                    'name' => 'Research Paper Writing',
                                    'url' => 'https://academicscribe.com/services/research-papers'
                                ]
                            ],
                            [
                                '@type' => 'Offer',
                                'itemOffered' => [
                                    '@type' => 'Service',
                                    'name' => 'Dissertation Writing',
                                    'url' => 'https://academicscribe.com/services/dissertation-writing'
                                ]
                            ]
                        ]
                    ]
                ],
                [
                    '@type' => 'BreadcrumbList',
                    'itemListElement' => [
                        [
                            '@type' => 'ListItem',
                            'position' => 1,
                            'name' => 'Home',
                            'item' => 'https://academicscribe.com'
                        ],
                        [
                            '@type' => 'ListItem',
                            'position' => 2,
                            'name' => 'Services',
                            'item' => 'https://academicscribe.com/services'
                        ]
                    ]
                ]
            ]
        ];
    }

    private function getAboutStructuredData()
    {
        return [
            '@context' => 'https://schema.org',
            '@graph' => [
                [
                    '@type' => 'AboutPage',
                    'name' => 'About Academic Scribe',
                    'url' => 'https://academicscribe.com/about',
                    'description' => 'Learn about our professional academic writing services, expert writers, and commitment to academic excellence',
                    'mainEntity' => [
                        '@type' => 'Organization',
                        'name' => 'Academic Scribe',
                        'url' => 'https://academicscribe.com',
                        'description' => 'Professional academic writing services with expert writers providing high-quality essays, research papers, dissertations, and academic assistance',
                        'foundingDate' => '2020',
                        'numberOfEmployees' => [
                            '@type' => 'QuantitativeValue',
                            'value' => '100+'
                        ],
                        'knowsAbout' => [
                            'Academic Writing',
                            'Essay Writing',
                            'Research Papers',
                            'Dissertation Writing',
                            'Academic Research'
                        ]
                    ]
                ],
                [
                    '@type' => 'BreadcrumbList',
                    'itemListElement' => [
                        [
                            '@type' => 'ListItem',
                            'position' => 1,
                            'name' => 'Home',
                            'item' => 'https://academicscribe.com'
                        ],
                        [
                            '@type' => 'ListItem',
                            'position' => 2,
                            'name' => 'About',
                            'item' => 'https://academicscribe.com/about'
                        ]
                    ]
                ]
            ]
        ];
    }

    private function getWritersStructuredData()
    {
        return [
            '@context' => 'https://schema.org',
            '@graph' => [
                [
                    '@type' => 'WebPage',
                    'name' => 'Our Expert Academic Writers',
                    'url' => 'https://academicscribe.com/writers',
                    'description' => 'Meet our team of professional academic writers with advanced degrees and expertise in all academic disciplines',
                    'about' => [
                        '@type' => 'EducationalOrganization',
                        'name' => 'Academic Scribe Writing Team',
                        'description' => 'Team of PhD and Masters level academic writers specializing in essays, research papers, and dissertations'
                    ]
                ],
                [
                    '@type' => 'BreadcrumbList',
                    'itemListElement' => [
                        [
                            '@type' => 'ListItem',
                            'position' => 1,
                            'name' => 'Home',
                            'item' => 'https://academicscribe.com'
                        ],
                        [
                            '@type' => 'ListItem',
                            'position' => 2,
                            'name' => 'Writers',
                            'item' => 'https://academicscribe.com/writers'
                        ]
                    ]
                ]
            ]
        ];
    }

    private function getGuaranteesStructuredData()
    {
        return [
            '@context' => 'https://schema.org',
            '@graph' => [
                [
                    '@type' => 'WebPage',
                    'name' => 'Our Guarantees - Quality & Satisfaction',
                    'url' => 'https://academicscribe.com/guarantees',
                    'description' => 'Our quality guarantees: 100% original content, on-time delivery, money-back guarantee, unlimited revisions, and 24/7 support'
                ],
                [
                    '@type' => 'Service',
                    'name' => 'Academic Writing Service Guarantees',
                    'provider' => [
                        '@type' => 'Organization',
                        'name' => 'Academic Scribe',
                        'url' => 'https://academicscribe.com'
                    ],
                    'termsOfService' => 'https://academicscribe.com/terms-of-service'
                ],
                [
                    '@type' => 'BreadcrumbList',
                    'itemListElement' => [
                        [
                            '@type' => 'ListItem',
                            'position' => 1,
                            'name' => 'Home',
                            'item' => 'https://academicscribe.com'
                        ],
                        [
                            '@type' => 'ListItem',
                            'position' => 2,
                            'name' => 'Guarantees',
                            'item' => 'https://academicscribe.com/guarantees'
                        ]
                    ]
                ]
            ]
        ];
    }
}

