<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Page;
use App\Models\PageSeo;
use App\Models\PageContent;

class SeedReviewsPageComplete extends Seeder
{
    public function run()
    {
        // Delete existing reviews page if it exists
        $existingPage = Page::where('slug', 'reviews')->first();
        if ($existingPage) {
            $existingPage->contentBlocks()->delete();
            $existingPage->seo()->delete();
            $existingPage->delete();
        }

        // Create the Reviews page
        $page = Page::create([
            'title' => 'Customer Reviews & Testimonials - Academic Scribe',
            'slug' => 'reviews',
            'page_type' => 'landing',
            'status' => 'published',
            'template' => 'default',
            'seo_priority' => 88,
            'published_at' => now(),
        ]);

        // Create SEO data (EXACT from original)
        PageSeo::create([
            'page_id' => $page->id,
            'meta_title' => 'Customer Reviews & Testimonials | 2800+ Verified Student Reviews | Academic Scribe',
            'meta_description' => 'Read 2800+ verified customer reviews and student testimonials for our essay writing, research paper, dissertation, and assignment help services. 4.9/5 star rating from satisfied students worldwide.',
            'meta_keywords' => 'customer reviews, student testimonials, essay writing reviews, research paper reviews, dissertation reviews, assignment help reviews, academic writing testimonials, verified customer feedback, 5-star reviews, satisfied students, quality reviews, service ratings, customer satisfaction, genuine reviews, authentic testimonials',
            'canonical_url' => 'https://academicscribe.com/reviews',
            'robots' => 'index, follow',
            'og_title' => '2800+ Verified Customer Reviews | Academic Writing Service Reviews',
            'og_description' => 'Read authentic student testimonials and customer reviews. 4.9/5 star rating from satisfied customers. Verified reviews for essay writing, research papers, dissertations, and more.',
            'og_image' => '/images/customer-reviews-og.jpg',
            'og_type' => 'website',
            'twitter_card' => 'summary_large_image',
            'twitter_title' => '2800+ Verified Customer Reviews | Student Testimonials',
            'twitter_description' => 'Read authentic student testimonials and customer reviews. 4.9/5 star rating from satisfied customers worldwide.',
            'twitter_image' => '/images/customer-reviews-twitter.jpg',
            'structured_data' => [
                'organization' => [
                    '@context' => 'https://schema.org',
                    '@type' => 'Organization',
                    'name' => 'Academic Scribe',
                    'aggregateRating' => [
                        '@type' => 'AggregateRating',
                        'ratingValue' => 4.9,
                        'reviewCount' => 2847,
                        'bestRating' => 5,
                        'worstRating' => 1
                    ]
                ]
            ]
        ]);

        // Block 1: Hero with Stats
        PageContent::create([
            'page_id' => $page->id,
            'block_type' => 'reviews_hero',
            'block_key' => 'reviews_hero',
            'order' => 1,
            'is_active' => true,
            'content' => [
                'title' => 'Customer Reviews & Student Testimonials',
                'description' => 'Read authentic reviews from 2,800+ satisfied students who trusted us with their academic success. See why we maintain a 4.9/5 star rating across all our academic writing services.',
                'stats' => [
                    ['value' => '2,847', 'label' => 'Total Reviews'],
                    ['value' => '4.9', 'label' => 'Average Rating'],
                    ['value' => '98%', 'label' => 'Satisfaction Rate'],
                    ['value' => '100%', 'label' => 'Verified Reviews'],
                ]
            ]
        ]);

        // Block 2: Rating Overview
        PageContent::create([
            'page_id' => $page->id,
            'block_type' => 'rating_overview',
            'block_key' => 'rating_overview',
            'order' => 2,
            'is_active' => true,
            'content' => [
                'title' => 'Verified Customer Reviews & Ratings',
                'description' => 'Our customer reviews speak for themselves. With over <strong>2,800 verified student testimonials</strong> and a <strong>4.9/5 star rating</strong>, we\'re proud to be the most trusted academic writing service.',
                'stats' => [
                    'totalReviews' => 2847,
                    'averageRating' => 4.9,
                    'fiveStars' => 2654,
                    'fourStars' => 156,
                    'threeStars' => 28,
                    'twoStars' => 6,
                    'oneStars' => 3,
                ],
                'features' => [
                    [
                        'title' => '100% Verified Reviews',
                        'description' => 'All reviews are from real customers who completed orders',
                    ],
                    [
                        'title' => 'Moderated for Quality',
                        'description' => 'We ensure all feedback is constructive and helpful',
                    ],
                    [
                        'title' => 'Regular Updates',
                        'description' => 'Fresh reviews added weekly from recent orders',
                    ],
                    [
                        'title' => '24/7 Customer Support',
                        'description' => 'Round-the-clock assistance',
                    ],
                ]
            ]
        ]);

        // Block 3: Featured Testimonials
        PageContent::create([
            'page_id' => $page->id,
            'block_type' => 'featured_testimonials',
            'block_key' => 'featured_testimonials',
            'order' => 3,
            'is_active' => true,
            'content' => [
                'title' => 'Featured Student Testimonials',
                'description' => 'Hear directly from students who achieved <strong>academic success</strong> with our help',
                'testimonials' => [
                    [
                        'name' => 'Jennifer Adams',
                        'service' => 'Essay Writing',
                        'serviceSlug' => 'essay-writing',
                        'quote' => 'The essay writing service transformed my academic performance. Professional writers, original content, and excellent customer support. My grades improved significantly after using their services.',
                        'rating' => 5,
                        'university' => 'Yale University',
                        'course' => 'Political Science',
                    ],
                    [
                        'name' => 'Thomas Lee',
                        'service' => 'Research Papers',
                        'serviceSlug' => 'research-papers',
                        'quote' => 'Exceptional research paper quality! The writer had PhD-level expertise and delivered comprehensive analysis with credible sources. The methodology was rigorous and findings were well-presented.',
                        'rating' => 5,
                        'university' => 'Princeton University',
                        'course' => 'Economics',
                    ],
                    [
                        'name' => 'Amanda Foster',
                        'service' => 'Dissertation Writing',
                        'serviceSlug' => 'dissertation-writing',
                        'quote' => 'My dissertation was handled with utmost professionalism. The writer provided expert guidance through every chapter. The final work exceeded my committee\'s expectations. Highly recommended!',
                        'rating' => 5,
                        'university' => 'Oxford University',
                        'course' => 'Literature',
                    ],
                ]
            ]
        ]);

        // Block 4: Service Filters
        PageContent::create([
            'page_id' => $page->id,
            'block_type' => 'service_filters',
            'block_key' => 'service_filters',
            'order' => 4,
            'is_active' => true,
            'content' => [
                'filters' => [
                    ['value' => 'all', 'label' => 'All Services', 'count' => 2847],
                    ['value' => 'essay-writing', 'label' => 'Essay Writing Reviews', 'count' => 1245],
                    ['value' => 'research-papers', 'label' => 'Research Paper Reviews', 'count' => 687],
                    ['value' => 'dissertation-writing', 'label' => 'Dissertation Reviews', 'count' => 423],
                    ['value' => 'assignments', 'label' => 'Assignment Help Reviews', 'count' => 298],
                    ['value' => 'case-studies', 'label' => 'Case Study Reviews', 'count' => 134],
                    ['value' => 'literature-reviews', 'label' => 'Literature Review Reviews', 'count' => 60],
                ]
            ]
        ]);

        // Block 5: All Customer Reviews (12 reviews - EXACT from original)
        PageContent::create([
            'page_id' => $page->id,
            'block_type' => 'customer_reviews',
            'block_key' => 'customer_reviews',
            'order' => 5,
            'is_active' => true,
            'content' => [
                'title' => 'All Customer Reviews by Service',
                'description' => 'Browse <strong>verified reviews</strong> for specific academic writing services',
                'reviews' => [
                    [
                        'id' => 1,
                        'name' => 'Sarah Johnson',
                        'service' => 'Essay Writing',
                        'serviceSlug' => 'essay-writing',
                        'rating' => 5,
                        'title' => 'Outstanding Essay Writing Service - Highly Recommended!',
                        'review' => 'Exceptional essay writing service! The quality exceeded my expectations. My writer delivered a perfectly structured argumentative essay with impeccable grammar and compelling arguments. The plagiarism report showed 100% originality. Customer support was responsive throughout the process. Definitely using this service again for future essays.',
                        'date' => '2024-01-15',
                        'verified' => true,
                        'helpful' => 47,
                        'course' => 'English Literature',
                        'university' => 'Harvard University',
                        'grade' => 'A+',
                        'orderType' => 'Argumentative Essay',
                    ],
                    [
                        'id' => 2,
                        'name' => 'Michael Chen',
                        'service' => 'Research Papers',
                        'serviceSlug' => 'research-papers',
                        'rating' => 5,
                        'title' => 'Excellent Research Paper Quality - Professional Writers',
                        'review' => 'Amazing research paper service! The writer demonstrated deep understanding of my topic and provided comprehensive analysis with credible sources. The methodology was sound and the findings were well-presented. Received my psychology research paper on time with free revisions. The quality was PhD-level work. Highly satisfied with the results.',
                        'date' => '2024-01-12',
                        'verified' => true,
                        'helpful' => 52,
                        'course' => 'Psychology',
                        'university' => 'Stanford University',
                        'grade' => 'A',
                        'orderType' => 'Research Paper',
                    ],
                    [
                        'id' => 3,
                        'name' => 'Emma Rodriguez',
                        'service' => 'Dissertation Writing',
                        'serviceSlug' => 'dissertation-writing',
                        'rating' => 5,
                        'title' => 'Professional Dissertation Help - Exceeded Expectations',
                        'review' => 'Outstanding dissertation writing service! My PhD dissertation was handled with utmost professionalism. The writer had expertise in my field and provided excellent literature review, methodology, and analysis chapters. The work was original, well-researched, and properly formatted. Customer service was excellent throughout the lengthy process. Worth every penny!',
                        'date' => '2024-01-10',
                        'verified' => true,
                        'helpful' => 89,
                        'course' => 'Business Administration',
                        'university' => 'MIT',
                        'grade' => 'A+',
                        'orderType' => 'PhD Dissertation',
                    ],
                    [
                        'id' => 4,
                        'name' => 'David Thompson',
                        'service' => 'Assignment Help',
                        'serviceSlug' => 'assignments',
                        'rating' => 5,
                        'title' => 'Reliable Assignment Help - Always On Time',
                        'review' => 'Fantastic assignment help service! I\'ve used them for multiple assignments and they never disappoint. The writers understand requirements perfectly and deliver high-quality work on time. My recent marketing assignment received excellent feedback from my professor. The customer support team is available 24/7 and very helpful. Highly recommend for urgent assignments.',
                        'date' => '2024-01-08',
                        'verified' => true,
                        'helpful' => 34,
                        'course' => 'Marketing',
                        'university' => 'UCLA',
                        'grade' => 'A-',
                        'orderType' => 'Marketing Assignment',
                    ],
                    [
                        'id' => 5,
                        'name' => 'Lisa Wang',
                        'service' => 'Case Studies',
                        'serviceSlug' => 'case-studies',
                        'rating' => 5,
                        'title' => 'Impressive Case Study Analysis - Expert Writers',
                        'review' => 'Excellent case study writing service! The writer provided thorough analysis with practical recommendations. The case study was well-structured with clear problem identification, analysis, and solutions. Used real-world examples and industry insights. My business professor was impressed with the quality. Great value for money and professional service throughout.',
                        'date' => '2024-01-05',
                        'verified' => true,
                        'helpful' => 28,
                        'course' => 'Business Strategy',
                        'university' => 'Wharton',
                        'grade' => 'A',
                        'orderType' => 'Business Case Study',
                    ],
                    [
                        'id' => 6,
                        'name' => 'James Wilson',
                        'service' => 'Literature Reviews',
                        'serviceSlug' => 'literature-reviews',
                        'rating' => 5,
                        'title' => 'Comprehensive Literature Review - Scholarly Quality',
                        'review' => 'Outstanding literature review service! The writer conducted comprehensive research and provided critical analysis of current literature in my field. The review was well-organized with proper citations and identified research gaps effectively. The quality was publication-ready. Excellent communication throughout the process. Highly recommend for graduate students.',
                        'date' => '2024-01-03',
                        'verified' => true,
                        'helpful' => 41,
                        'course' => 'Education',
                        'university' => 'Columbia University',
                        'grade' => 'A+',
                        'orderType' => 'Systematic Literature Review',
                    ],
                    [
                        'id' => 7,
                        'name' => 'Rachel Green',
                        'service' => 'Essay Writing',
                        'serviceSlug' => 'essay-writing',
                        'rating' => 5,
                        'title' => 'Perfect Essay Writing - Exceeded My Expectations',
                        'review' => 'Amazing essay writing experience! The writer crafted a compelling narrative essay with excellent flow and engaging content. The introduction was captivating and the conclusion was powerful. Received detailed feedback and free revisions. The final essay was polished and professional. My English professor gave excellent marks. Will definitely order again!',
                        'date' => '2024-01-01',
                        'verified' => true,
                        'helpful' => 36,
                        'course' => 'Creative Writing',
                        'university' => 'NYU',
                        'grade' => 'A+',
                        'orderType' => 'Narrative Essay',
                    ],
                    [
                        'id' => 8,
                        'name' => 'Alex Kumar',
                        'service' => 'Research Papers',
                        'serviceSlug' => 'research-papers',
                        'rating' => 5,
                        'title' => 'Top-Quality Research Paper - Highly Professional',
                        'review' => 'Exceptional research paper service! The writer demonstrated expertise in computer science and delivered a technically sound paper with innovative solutions. The literature review was comprehensive and the methodology was rigorous. All requirements were met perfectly. The paper contributed valuable insights to my field. Excellent customer service and timely delivery.',
                        'date' => '2023-12-28',
                        'verified' => true,
                        'helpful' => 43,
                        'course' => 'Computer Science',
                        'university' => 'Carnegie Mellon',
                        'grade' => 'A',
                        'orderType' => 'Technical Research Paper',
                    ],
                    [
                        'id' => 9,
                        'name' => 'Sophie Martinez',
                        'service' => 'Dissertation Writing',
                        'serviceSlug' => 'dissertation-writing',
                        'rating' => 5,
                        'title' => 'Excellent Dissertation Support - Professional Service',
                        'review' => 'Outstanding dissertation writing service! The writer helped me through every chapter with expert guidance. The research was thorough, analysis was insightful, and writing was scholarly. The dissertation committee was impressed with the quality. Received continuous support and revisions until perfect. This service made my PhD journey much smoother. Highly recommended!',
                        'date' => '2023-12-25',
                        'verified' => true,
                        'helpful' => 67,
                        'course' => 'Sociology',
                        'university' => 'University of Chicago',
                        'grade' => 'A+',
                        'orderType' => 'PhD Dissertation',
                    ],
                    [
                        'id' => 10,
                        'name' => 'Ryan O\'Connor',
                        'service' => 'Assignment Help',
                        'serviceSlug' => 'assignments',
                        'rating' => 5,
                        'title' => 'Reliable Assignment Writing - Always Delivers',
                        'review' => 'Fantastic assignment help service! I\'ve been using them for my MBA coursework and they consistently deliver high-quality work. The writers understand business concepts well and provide practical insights. My recent strategic management assignment received top marks. Fast turnaround, excellent quality, and great customer support. My go-to service for assignments.',
                        'date' => '2023-12-22',
                        'verified' => true,
                        'helpful' => 29,
                        'course' => 'Strategic Management',
                        'university' => 'Kellogg',
                        'grade' => 'A',
                        'orderType' => 'MBA Assignment',
                    ],
                    [
                        'id' => 11,
                        'name' => 'Maria Gonzalez',
                        'service' => 'Case Studies',
                        'serviceSlug' => 'case-studies',
                        'rating' => 5,
                        'title' => 'Professional Case Study Writing - Excellent Analysis',
                        'review' => 'Excellent case study writing service! The writer provided comprehensive analysis with actionable recommendations. The case study was well-researched with relevant data and industry insights. My healthcare management professor praised the practical approach and professional presentation. Great attention to detail and timely delivery. Highly satisfied with the quality and service.',
                        'date' => '2023-12-20',
                        'verified' => true,
                        'helpful' => 31,
                        'course' => 'Healthcare Management',
                        'university' => 'Johns Hopkins',
                        'grade' => 'A-',
                        'orderType' => 'Healthcare Case Study',
                    ],
                    [
                        'id' => 12,
                        'name' => 'Kevin Park',
                        'service' => 'Literature Reviews',
                        'serviceSlug' => 'literature-reviews',
                        'rating' => 5,
                        'title' => 'Comprehensive Literature Review - Scholarly Excellence',
                        'review' => 'Outstanding literature review service! The writer conducted extensive research and provided critical synthesis of current literature. The review identified key themes, research gaps, and future directions effectively. The quality was suitable for publication. Excellent organization and proper academic formatting. My advisor was impressed with the scholarly quality. Highly recommend!',
                        'date' => '2023-12-18',
                        'verified' => true,
                        'helpful' => 38,
                        'course' => 'Environmental Science',
                        'university' => 'UC Berkeley',
                        'grade' => 'A+',
                        'orderType' => 'Systematic Review',
                    ],
                ]
            ]
        ]);

        // Block 6: Reviews by Service
        PageContent::create([
            'page_id' => $page->id,
            'block_type' => 'reviews_by_service',
            'block_key' => 'reviews_by_service',
            'order' => 6,
            'is_active' => true,
            'content' => [
                'title' => 'Reviews by Academic Writing Service',
                'description' => 'See what students say about each of our <strong>specialized services</strong>',
                'services' => [
                    [
                        'icon' => '📝',
                        'label' => 'Essay Writing Reviews',
                        'value' => 'essay-writing',
                        'rating' => 4.9,
                        'averageRating' => '4.9',
                        'count' => 1245,
                        'features' => [
                            'Original, plagiarism-free content',
                            'Expert writers with advanced degrees',
                            'On-time delivery guarantee',
                            'Free unlimited revisions',
                        ],
                        'link' => '/services/essay-writing',
                        'linkText' => 'View Service & Order',
                    ],
                    [
                        'icon' => '🔬',
                        'label' => 'Research Paper Reviews',
                        'value' => 'research-papers',
                        'rating' => 4.9,
                        'averageRating' => '4.9',
                        'count' => 687,
                        'features' => [
                            'Original, plagiarism-free content',
                            'Expert writers with advanced degrees',
                            'On-time delivery guarantee',
                            'Free unlimited revisions',
                        ],
                        'link' => '/services/research-papers',
                        'linkText' => 'View Service & Order',
                    ],
                    [
                        'icon' => '🎓',
                        'label' => 'Dissertation Reviews',
                        'value' => 'dissertation-writing',
                        'rating' => 4.9,
                        'averageRating' => '4.9',
                        'count' => 423,
                        'features' => [
                            'Original, plagiarism-free content',
                            'Expert writers with advanced degrees',
                            'On-time delivery guarantee',
                            'Free unlimited revisions',
                        ],
                        'link' => '/services/dissertation-writing',
                        'linkText' => 'View Service & Order',
                    ],
                    [
                        'icon' => '📚',
                        'label' => 'Assignment Help Reviews',
                        'value' => 'assignments',
                        'rating' => 4.9,
                        'averageRating' => '4.9',
                        'count' => 298,
                        'features' => [
                            'Original, plagiarism-free content',
                            'Expert writers with advanced degrees',
                            'On-time delivery guarantee',
                            'Free unlimited revisions',
                        ],
                        'link' => '/services/assignments',
                        'linkText' => 'View Service & Order',
                    ],
                    [
                        'icon' => '💼',
                        'label' => 'Case Study Reviews',
                        'value' => 'case-studies',
                        'rating' => 4.9,
                        'averageRating' => '4.9',
                        'count' => 134,
                        'features' => [
                            'Original, plagiarism-free content',
                            'Expert writers with advanced degrees',
                            'On-time delivery guarantee',
                            'Free unlimited revisions',
                        ],
                        'link' => '/services/case-studies',
                        'linkText' => 'View Service & Order',
                    ],
                    [
                        'icon' => '📖',
                        'label' => 'Literature Review Reviews',
                        'value' => 'literature-reviews',
                        'rating' => 4.9,
                        'averageRating' => '4.9',
                        'count' => 60,
                        'features' => [
                            'Original, plagiarism-free content',
                            'Expert writers with advanced degrees',
                            'On-time delivery guarantee',
                            'Free unlimited revisions',
                        ],
                        'link' => '/services/literature-reviews',
                        'linkText' => 'View Service & Order',
                    ],
                ]
            ]
        ]);

        // Block 7: Trust Indicators
        PageContent::create([
            'page_id' => $page->id,
            'block_type' => 'review_trust',
            'block_key' => 'review_trust',
            'order' => 7,
            'is_active' => true,
            'content' => [
                'title' => 'Why Our Customer Reviews Are Trustworthy',
                'description' => 'We maintain the highest standards for <strong>review authenticity and verification</strong>',
                'indicators' => [
                    [
                        'title' => 'Verified Customer Reviews',
                        'description' => 'All reviews are from real customers who completed orders. We verify each review against our order database to ensure authenticity.',
                    ],
                    [
                        'title' => 'Moderated for Quality',
                        'description' => 'Reviews are moderated to ensure they\'re helpful and constructive. We don\'t edit content but ensure reviews follow community guidelines.',
                    ],
                    [
                        'title' => 'Complete Transparency',
                        'description' => 'We display all verified reviews, including critical feedback. Our rating reflects genuine customer satisfaction across all services.',
                    ],
                    [
                        'title' => 'No Fake Reviews',
                        'description' => 'We never purchase fake reviews or incentivize positive feedback. Every review represents authentic student experiences.',
                    ],
                ]
            ]
        ]);

        // Block 8: FAQs
        PageContent::create([
            'page_id' => $page->id,
            'block_type' => 'reviews_faqs',
            'block_key' => 'reviews_faqs',
            'order' => 8,
            'is_active' => true,
            'content' => [
                'title' => 'Frequently Asked Questions About Our Reviews',
                'description' => 'Common questions about our <strong>customer review system and verification process</strong>',
                'faqs' => [
                    [
                        'question' => 'How do you verify customer reviews?',
                        'answer' => 'Every review is verified through our order system. Only customers who have completed an order can leave a review. We cross-reference each review with actual order data to ensure authenticity. All reviews include order verification badges and are linked to real transactions.',
                    ],
                    [
                        'question' => 'Can I trust the 4.9/5 star rating?',
                        'answer' => 'Absolutely! Our 4.9/5 star rating is based on 2,800+ genuine customer reviews. We never purchase fake reviews or incentivize positive feedback. The rating reflects the actual satisfaction of students who have used our academic writing services across all categories.',
                    ],
                    [
                        'question' => 'Do you remove negative reviews?',
                        'answer' => 'We maintain complete transparency in our review system. Negative reviews are not removed unless they violate our content policy (contain inappropriate language, personal attacks, or false information). We believe authentic feedback, both positive and constructive, helps us improve our services.',
                    ],
                    [
                        'question' => 'How recent are these customer reviews?',
                        'answer' => 'Our reviews span from recent orders to historical feedback over several years. You can filter reviews by date to see the most recent customer experiences. We continuously receive new reviews as we complete more orders for students worldwide.',
                    ],
                    [
                        'question' => 'Can I leave a review after my order?',
                        'answer' => 'Yes! After your order is completed, you\'ll receive an email invitation to leave a review. You can rate your experience and provide detailed feedback about the quality, writer performance, customer service, and overall satisfaction. Your honest feedback helps other students make informed decisions.',
                    ],
                    [
                        'question' => 'Are reviews specific to different services?',
                        'answer' => 'Yes, reviews are categorized by service type (essay writing, research papers, dissertations, etc.). This allows you to read feedback specific to the type of academic writing help you need. Each service maintains its own rating based on relevant customer experiences.',
                    ],
                ]
            ]
        ]);

        // Block 9: Final CTA
        PageContent::create([
            'page_id' => $page->id,
            'block_type' => 'reviews_cta',
            'block_key' => 'reviews_cta',
            'order' => 9,
            'is_active' => true,
            'content' => [
                'title' => 'Join 2,800+ Satisfied Students Today',
                'description' => 'Experience the quality that 2,800+ students have praised. Order your academic paper today and see why we maintain a 4.9/5 star rating across all services.',
                'primaryButtonText' => 'Order Your Paper Now',
                'primaryButtonLink' => '/place-order',
                'secondaryButtonText' => 'View Pricing',
                'secondaryButtonLink' => '/pricing',
                'trustBadges' => [
                    '4.9/5 Star Rating',
                    '2,800+ Happy Students',
                    '100% Verified Reviews',
                    'Money-Back Guarantee',
                ]
            ]
        ]);

        // Block 10: Service Links Footer
        PageContent::create([
            'page_id' => $page->id,
            'block_type' => 'review_service_links',
            'block_key' => 'review_service_links',
            'order' => 10,
            'is_active' => true,
            'content' => [
                'title' => 'Read Reviews for All Our Academic Writing Services',
                'description' => 'Every service maintains <strong>excellent customer satisfaction ratings</strong>',
                'services' => [
                    ['icon' => '📝', 'name' => 'Essay Writing', 'link' => '/services/essay-writing', 'reviewCount' => '1,245'],
                    ['icon' => '🔬', 'name' => 'Research Papers', 'link' => '/services/research-papers', 'reviewCount' => '687'],
                    ['icon' => '🎓', 'name' => 'Dissertations', 'link' => '/services/dissertation-writing', 'reviewCount' => '423'],
                    ['icon' => '📚', 'name' => 'Assignments', 'link' => '/services/assignments', 'reviewCount' => '298'],
                    ['icon' => '💼', 'name' => 'Case Studies', 'link' => '/services/case-studies', 'reviewCount' => '134'],
                    ['icon' => '📖', 'name' => 'Literature Reviews', 'link' => '/services/literature-reviews', 'reviewCount' => '60'],
                    ['icon' => '➕', 'name' => 'All Services', 'link' => '/services', 'reviewCount' => 'View all'],
                ],
            ]
        ]);

        $this->command->info('✅ Reviews page created successfully with all content blocks!');
        $this->command->info('   - Hero with stats');
        $this->command->info('   - Rating overview with bar chart data');
        $this->command->info('   - 3 featured testimonials');
        $this->command->info('   - 12 detailed customer reviews');
        $this->command->info('   - 6 service review categories');
        $this->command->info('   - 4 trust indicators');
        $this->command->info('   - 6 FAQs');
        $this->command->info('   - CTA and service links');
    }
}




