<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use App\Models\Page;
use App\Models\PageSeo;
use App\Models\PageContent;

class SeedServiceDissertations extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Upsert the page for the Dissertation Writing service
        $page = Page::updateOrCreate(
            ['slug' => 'dissertation-writing'],
            [
                'title' => 'Dissertation Writing Service',
                'page_type' => 'service',
                'status' => 'published',
                'template' => 'Services/Dissertations',
                'seo_priority' => 80,
                'published_at' => now(),
            ]
        );

        // SEO data mapped from resources/js/data/services/dissertations.js -> meta + structuredData
        $structuredData = [
            'service' => [
                '@context' => 'https://schema.org',
                '@type' => 'Service',
                'name' => 'Professional Dissertation Writing Service',
                'description' => 'Expert dissertation writing service providing PhD dissertation help with professional writers, advanced research methodology, comprehensive analysis, and perfect academic formatting for doctoral and master\'s level students.',
                'provider' => [
                    '@type' => 'Organization',
                    'name' => 'Academic Scribe',
                    'url' => 'https://academicscribe.com',
                    'logo' => 'https://academicscribe.com/images/logo.png',
                    'contactPoint' => [
                        '@type' => 'ContactPoint',
                        'telephone' => '+1-555-123-4567',
                        'contactType' => 'customer service',
                        'availableLanguage' => ['English'],
                        'areaServed' => 'Worldwide',
                    ],
                    'address' => [
                        '@type' => 'PostalAddress',
                        'addressCountry' => 'US',
                        'addressRegion' => 'Global',
                    ],
                    'sameAs' => [
                        'https://facebook.com/academicscribe',
                        'https://twitter.com/academicscribe',
                        'https://linkedin.com/company/academicscribe',
                    ],
                ],
                'serviceType' => 'Academic Writing Service',
                'areaServed' => 'Worldwide',
                'availableChannel' => [
                    '@type' => 'ServiceChannel',
                    'serviceUrl' => 'https://academicscribe.com/services/dissertation-writing',
                    'servicePhone' => '+1-555-123-4567',
                    'availableLanguage' => 'English',
                ],
            ],
            'breadcrumb' => [
                '@context' => 'https://schema.org',
                '@type' => 'BreadcrumbList',
                'itemListElement' => [
                    [
                        '@type' => 'ListItem',
                        'position' => 1,
                        'name' => 'Home',
                        'item' => 'https://academicscribe.com',
                    ],
                    [
                        '@type' => 'ListItem',
                        'position' => 2,
                        'name' => 'Services',
                        'item' => 'https://academicscribe.com/services',
                    ],
                    [
                        '@type' => 'ListItem',
                        'position' => 3,
                        'name' => 'Dissertation Writing Service',
                        'item' => 'https://academicscribe.com/services/dissertation-writing',
                    ],
                ],
            ],
            'organization' => [
                '@context' => 'https://schema.org',
                '@type' => 'Organization',
                'name' => 'Academic Scribe',
                'description' => 'Professional academic writing services with expert dissertation writers providing PhD dissertation help and comprehensive doctoral thesis assistance for students worldwide.',
                'url' => 'https://academicscribe.com',
                'logo' => 'https://academicscribe.com/images/logo.png',
                'image' => 'https://academicscribe.com/images/dissertation-writing-service.jpg',
                'contactPoint' => [
                    '@type' => 'ContactPoint',
                    'telephone' => '+1-555-123-4567',
                    'contactType' => 'customer service',
                    'availableLanguage' => ['English'],
                    'areaServed' => 'Worldwide',
                ],
                'address' => [
                    '@type' => 'PostalAddress',
                    'addressCountry' => 'US',
                    'addressRegion' => 'Global',
                ],
                'sameAs' => [
                    'https://facebook.com/academicscribe',
                    'https://twitter.com/academicscribe',
                    'https://linkedin.com/company/academicscribe',
                ],
            ],
            'website' => [
                '@context' => 'https://schema.org',
                '@type' => 'WebSite',
                'name' => 'Academic Scribe - Dissertation Writing Service',
                'description' => 'Professional dissertation writers providing PhD dissertation help and comprehensive doctoral thesis writing service for students at all academic levels.',
                'url' => 'https://academicscribe.com/services/dissertation-writing',
                'potentialAction' => [
                    '@type' => 'SearchAction',
                    'target' => 'https://academicscribe.com/search?q={search_term_string}',
                    'query-input' => 'required name=search_term_string',
                ],
            ],
            'webpage' => [
                '@context' => 'https://schema.org',
                '@type' => 'WebPage',
                'name' => 'Professional Dissertation Writing Service | PhD Dissertation Help',
                'description' => 'Expert dissertation writing service with PhD-qualified writers. Get professional doctoral dissertation help with advanced research methodology, comprehensive analysis, and guaranteed results.',
                'url' => 'https://academicscribe.com/services/dissertation-writing',
                'isPartOf' => [
                    '@type' => 'WebSite',
                    'name' => 'Academic Scribe',
                    'url' => 'https://academicscribe.com',
                ],
                'about' => [
                    '@type' => 'Service',
                    'name' => 'Dissertation Writing Service',
                ],
                'mainEntity' => [
                    '@type' => 'Service',
                    'name' => 'Professional Dissertation Writing Service',
                ],
            ],
        ];

        PageSeo::updateOrCreate(
            ['page_id' => $page->id],
            [
                'meta_title' => 'PhD Dissertation Writing Service | Expert PhD Writers | 24/7 Support | Academic Scribe',
                'meta_description' => 'Struggling with PhD dissertation writing? Get expert help from PhD-qualified writers. 100% original, plagiarism-free dissertations with methodology support. Money-back guarantee. Order now for stress-free PhD completion!',
                'meta_keywords' => 'dissertation writing service, professional dissertation writing, PhD dissertation help, doctoral dissertation service, dissertation writers, dissertation writing assistance, doctoral thesis help, PhD thesis writing, dissertation writing for students, professional academic writing service, dissertation help online, custom dissertations, academic dissertation writing, dissertation writing company, dissertation writing experts',
                'canonical_url' => 'https://academicscribe.com/services/dissertation-writing',
                'og_url' => 'https://academicscribe.com/services/dissertation-writing',
                'robots' => 'index, follow, max-snippet:-1, max-image-preview:large, max-video-preview:-1',
                'language' => 'en-US',
                'structured_data' => $structuredData,
                'twitter_card' => 'summary_large_image',
                'twitter_title' => 'Professional Dissertation Writing Service | PhD Dissertation Help | Expert Writers',
                'twitter_description' => 'Expert dissertation writing service with PhD-qualified writers. Get professional doctoral dissertation help with advanced research methodology and guaranteed results.',
                'twitter_image' => 'https://academicscribe.com/images/dissertation-writing-service-twitter.jpg',
                'og_title' => 'Professional Dissertation Writing Service | PhD Dissertation Help | Expert Writers',
                'og_description' => 'Expert dissertation writing service with PhD-qualified writers. Get professional doctoral dissertation help with advanced research methodology and comprehensive analysis. Trusted by 1,000+ PhD students worldwide.',
                'og_image' => 'https://academicscribe.com/images/dissertation-writing-service-og.jpg',
                'og_type' => 'website',
            ]
        );

        // Content blocks (keys mirror Essays implementation, prefixed with dissertation_*)
        $blocks = [
            [
                'block_type' => 'service_hero',
                'block_key' => 'dissertation_hero',
                'order' => 1,
                'content' => [
                    'title' => 'Struggling with PhD Dissertation Writing? Get Expert Help from PhD Writers',
                    'description' => 'Running out of time for your doctoral thesis? Need expert guidance for complex research? Our PhD-qualified dissertation writers provide professional dissertation help with original research, advanced methodology, and comprehensive analysis. Trusted by 1,000+ PhD students worldwide with 24/7 support and guaranteed results.',
                    'keyPoints' => [
                        'Struggling with PhD dissertation writing? Our expert PhD writers solve your research crisis',
                        'Need professional dissertation help with tight deadlines? Get 100% original, plagiarism-free content',
                        'Running out of time for doctoral thesis? Our dissertation service covers all disciplines with advanced methodology',
                        'Worried about academic standards? Get dissertation writing assistance with perfect formatting and citations',
                    ],
                    'ctaText' => 'Solve Your PhD Dissertation Crisis Now',
                    'ctaLink' => '/contact',
                    'trustSignals' => [
                        '1,000+ successful dissertations',
                        '4.9/5 average rating',
                        '24/7 expert support',
                        'Money-back guarantee',
                        'Limited time: 15% OFF for new PhD students',
                    ],
                ],
            ],
            [
                'block_type' => 'service_overview',
                'block_key' => 'dissertation_overview',
                'order' => 2,
                'content' => [
                    'title' => 'Our Comprehensive Professional PhD Dissertation Writing Service',
                    'description' => 'We provide affordable PhD dissertation writing help online across all academic disciplines with expert research methodology and doctoral-level excellence. Our professional dissertation writing service with PhD writers includes original research, advanced analysis, and comprehensive writing support for doctoral and master\'s students at all universities.',
                    'features' => [
                        [
                            'title' => 'PhD-Qualified Dissertation Writers with Doctoral Expertise',
                            'description' => 'Expert dissertation writers with doctoral degrees (PhD & advanced qualifications) who understand dissertation standards, research methodology, and discipline-specific requirements for quality professional dissertation writing service with PhD writers',
                            'icon' => 'Award',
                        ],
                        [
                            'title' => 'Advanced Research & Methodology for Professional PhD Dissertation Help',
                            'description' => 'Professional dissertation help with original research, advanced methodology, comprehensive literature review, and statistical analysis tailored to your specific doctoral requirements and research objectives for reliable doctoral dissertation writing service',
                            'icon' => 'Search',
                        ],
                        [
                            'title' => 'All Dissertation Types Covered by Expert PhD Writers',
                            'description' => 'Doctoral dissertation writing service covering PhD dissertations, master\'s theses, EdD projects, DBA dissertations, and professional doctorate work with subject-matter experts and research specialists for best professional writers for PhD dissertation',
                            'icon' => 'BookOpen',
                        ],
                        [
                            'title' => 'Perfect Academic Standards for PhD Dissertation Writing Service',
                            'description' => 'Professional dissertation writing with proper citations, references, and formatting in APA, MLA, Chicago, Harvard, and institutional styles with attention to university-specific guidelines for PhD dissertation help professional',
                            'icon' => 'FileText',
                        ],
                    ],
                ],
            ],
            [
                'block_type' => 'service_process',
                'block_key' => 'dissertation_process',
                'order' => 3,
                'content' => [
                    'title' => 'Our Professional Dissertation Writing Process',
                    'description' => 'We follow a rigorous, doctoral-level approach to deliver exceptional dissertation writing service with advanced research methodology, original analysis, and academic excellence that meets your university\'s highest standards.',
                    'steps' => [
                        [ 'number' => 1, 'title' => 'Dissertation Topic Analysis & Research Planning', 'description' => 'Comprehensive analysis of your dissertation requirements, research objectives, university guidelines, and dissertation scope to establish clear direction for your PhD dissertation help project with detailed timeline planning.', 'icon' => 'Target' ],
                        [ 'number' => 2, 'title' => 'Advanced Research Methodology Development', 'description' => 'Develop sophisticated research methodology with relevant academic sources, database selection, research design, and analytical frameworks for professional dissertation writing service excellence.', 'icon' => 'Search' ],
                        [ 'number' => 3, 'title' => 'Literature Review & Theoretical Framework', 'description' => 'Conduct comprehensive literature review and develop theoretical framework with critical analysis, gap identification, and scholarly synthesis for your doctoral dissertation service with academic rigor.', 'icon' => 'BookOpen' ],
                        [ 'number' => 4, 'title' => 'Data Collection & Statistical Analysis', 'description' => 'Implement data collection strategies and perform advanced statistical analysis, qualitative research, or mixed methods approaches for your dissertation writing assistance with methodological excellence.', 'icon' => 'BarChart' ],
                        [ 'number' => 5, 'title' => 'Professional Dissertation Writing', 'description' => 'PhD-qualified dissertation writers craft your dissertation with clear academic structure, compelling arguments, logical flow, and adherence to doctoral writing standards for professional dissertation help excellence.', 'icon' => 'Edit' ],
                        [ 'number' => 6, 'title' => 'Citation & Academic Formatting', 'description' => 'Apply proper academic citations, reference lists, and formatting according to your university\'s required style guide with attention to institutional guidelines for dissertation writing service compliance.', 'icon' => 'FileText' ],
                        [ 'number' => 7, 'title' => 'Quality Review & Doctoral-Level Editing', 'description' => 'Comprehensive review, editing, and proofreading to ensure your dissertation meets doctoral standards, university requirements, and excellence expectations for professional dissertation writing service delivery.', 'icon' => 'CheckCircle' ],
                    ],
                ],
            ],
            [
                'block_type' => 'service_benefits',
                'block_key' => 'dissertation_benefits',
                'order' => 4,
                'content' => [
                    'title' => 'Why Choose Our Professional Dissertation Writing Service',
                    'description' => 'Discover why over 1,000 doctoral students choose our expert dissertation writing assistance for their academic success and PhD completion',
                    'benefits' => [
                        [ 'title' => 'PhD-Qualified Dissertation Specialists', 'description' => 'Professional dissertation writers with doctoral degrees (PhD & advanced qualifications) in your specific field who understand dissertation requirements, research methodology, and academic standards for exceptional results', 'icon' => 'Award' ],
                        [ 'title' => '100% Original Research & Custom Dissertations', 'description' => 'PhD dissertation help with completely original research, unique analysis, plagiarism-free content, and comprehensive originality reports tailored to your specific doctoral requirements and research objectives', 'icon' => 'Shield' ],
                        [ 'title' => 'Advanced Research Methodology', 'description' => 'Doctoral dissertation service with sophisticated research design, advanced data collection techniques, statistical analysis methods, and evidence-based conclusions that meet rigorous academic standards', 'icon' => 'Search' ],
                        [ 'title' => 'Perfect Doctoral Standards Compliance', 'description' => 'Dissertation writing assistance that meets all doctoral standards with proper structure, perfect citations, university formatting requirements, and adherence to academic integrity guidelines', 'icon' => 'CheckCircle' ],
                        [ 'title' => 'Guaranteed Completion & Expert Support', 'description' => 'Professional dissertation writing service with reliable completion schedules, 24/7 expert support, unlimited revisions, and money-back guarantee for your complete satisfaction and PhD success', 'icon' => 'Clock' ],
                    ],
                    'imageAlt' => 'Professional Dissertation Writing Service Benefits - PhD Dissertation Help with Expert Writers and Original Research',
                    'ctaText' => 'Start Your PhD Success Story Today',
                    'ctaLink' => '/place-order',
                    'guarantees' => [
                        '100% Original Research',
                        'On-Time Completion',
                        '24/7 Expert Support',
                        'Money-Back Guarantee',
                        'Unlimited Revisions',
                        'Fast Delivery: 7-14 days',
                    ],
                ],
            ],
            [
                'block_type' => 'service_testimonials',
                'block_key' => 'dissertation_testimonials',
                'order' => 5,
                'content' => [
                    'title' => 'What Our Dissertation Writing Service Clients Say',
                    'description' => 'Read authentic testimonials from doctoral students who\'ve achieved PhD success with our professional dissertation writing assistance',
                    'testimonials' => [
                        [ 'text' => 'The dissertation writing service I received was exceptional. The writer conducted advanced research methodology and provided comprehensive analysis that impressed my dissertation committee. The PhD dissertation help included perfect citations and met all my university\'s doctoral requirements. I successfully defended my dissertation!', 'author' => 'Dr. Amanda Chen', 'role' => 'PhD Graduate, Educational Psychology', 'rating' => 5, 'verified' => true, 'university' => 'Stanford University' ],
                        [ 'text' => 'I needed professional dissertation writing for my doctoral thesis, and the quality exceeded my expectations. The doctoral dissertation service provided original research with sophisticated methodology and clear presentation of findings. The writer was highly knowledgeable and responsive throughout the entire process.', 'author' => 'Dr. Robert Martinez', 'role' => 'PhD Graduate, Business Administration', 'rating' => 5, 'verified' => true, 'university' => 'Wharton School, University of Pennsylvania' ],
                        [ 'text' => 'As an international doctoral student, I struggled with advanced academic writing standards. The dissertation writing assistance helped me understand proper methodology and citation styles while delivering a high-quality dissertation that earned committee approval. The expert support was available 24/7 and incredibly helpful.', 'author' => 'Dr. Priya Sharma', 'role' => 'PhD Graduate, Computer Science', 'rating' => 5, 'verified' => true, 'university' => 'MIT' ],
                    ],
                    'trustMetrics' => [
                        'totalReviews' => '847',
                        'averageRating' => '4.9',
                        'satisfactionRate' => '99%',
                        'repeatCustomers' => '85%',
                    ],
                ],
            ],
            [
                'block_type' => 'service_faqs',
                'block_key' => 'dissertation_faqs',
                'order' => 6,
                'content' => [
                    'title' => 'Dissertation Writing Service FAQs',
                    'description' => 'Get answers to common questions about our professional dissertation writing assistance and PhD dissertation help services',
                    'faqs' => [
                        [ 'question' => 'What types of dissertations does your dissertation writing service handle?', 'answer' => 'Our professional dissertation writing service handles all types of dissertations including PhD dissertations, doctoral theses, master\'s dissertations, EdD dissertations, DBA dissertations, professional doctorate projects, and thesis proposals. Our PhD dissertation help covers all academic disciplines including sciences, humanities, business, engineering, social sciences, medical sciences, law, education, and specialized fields across all universities and institutions worldwide. We also handle specific formats like APA dissertations, MLA dissertations, Chicago style dissertations, and university-specific requirements.' ],
                        [ 'question' => 'What qualifications do your dissertation writers have?', 'answer' => 'Our dissertation writers hold advanced degrees (PhD and doctoral qualifications) in their respective fields with extensive dissertation writing and research experience. Each professional dissertation writer undergoes rigorous screening including academic credential verification, dissertation portfolio review, research methodology expertise testing, and advanced writing assessment. Our doctoral dissertation service team includes former professors, published researchers, dissertation supervisors, and academic specialists with proven track records in dissertation completion and PhD supervision.' ],
                        [ 'question' => 'How do you ensure original research in your dissertation writing service?', 'answer' => 'Our dissertation writing service guarantees 100% original research through comprehensive database searches, primary source analysis, custom research methodology, and original data collection. Our PhD dissertation writers conduct fresh research for each project, avoiding pre-written content and template-based approaches. We use advanced plagiarism detection tools (Turnitin, iThenticate, Grammarly Premium) and provide detailed plagiarism reports with every doctoral dissertation help order. Additionally, we maintain strict confidentiality and delete all client work after completion to ensure complete originality.' ],
                        [ 'question' => 'Can you help with dissertation methodology and research design?', 'answer' => 'Yes, our doctoral dissertation service includes comprehensive research methodology assistance and consultation. Our PhD dissertation writers help design appropriate research methods, select suitable data collection techniques, choose proper analytical approaches, develop research instruments, and justify methodological choices. Whether you need quantitative research methods (statistical analysis, surveys, experiments), qualitative research methods (interviews, case studies, ethnography), or mixed methods approaches, our dissertation writing assistance covers all methodological aspects with expert guidance.' ],
                        [ 'question' => 'Do you provide chapter-by-chapter dissertation writing help?', 'answer' => 'Yes, our professional dissertation writing service offers chapter-by-chapter assistance including introduction chapters, literature review chapters, methodology chapters, results/findings chapters, discussion chapters, and conclusion chapters. Our PhD dissertation help includes comprehensive support for each section with proper academic structure, citations, and formatting. We also provide dissertation proposal writing, abstract writing, and appendix preparation as part of our doctoral dissertation service.' ],
                        [ 'question' => 'What is your revision policy for dissertation writing service?', 'answer' => 'Our dissertation writing service includes unlimited free revisions within 60 days of delivery to ensure complete satisfaction. If your PhD dissertation help doesn\'t meet your requirements, needs adjustments, or requires modifications based on committee feedback, our professional dissertation writers will revise it at no additional cost. We\'re committed to your doctoral success and will work with you until your dissertation meets all your university requirements, committee standards, and personal expectations.' ],
                        [ 'question' => 'How do you handle confidentiality in dissertation writing?', 'answer' => 'Our professional dissertation writing service maintains strict confidentiality and privacy protection through encrypted communication systems, secure payment processing, anonymous writer assignments, and comprehensive non-disclosure agreements. We never share client information, research topics, or completed dissertations with third parties. All PhD dissertation help orders are processed with complete anonymity, and we delete all client files after project completion. Our dissertation writers sign confidentiality agreements, and our platform uses SSL encryption and secure servers.' ],
                        [ 'question' => 'Do you provide dissertation defense preparation support?', 'answer' => 'Yes, our doctoral dissertation service includes comprehensive dissertation defense preparation including presentation development, question anticipation, defense strategy coaching, and oral examination support. Our PhD dissertation writers help prepare defense presentations, anticipate committee questions, develop response strategies, and provide mock defense sessions. We also offer post-defense revision support if committee recommendations require dissertation modifications.' ],
                        [ 'question' => 'What if I\'m running out of time for my PhD dissertation deadline?', 'answer' => 'Our fast PhD dissertation writing service specializes in tight deadline situations. We offer emergency dissertation help with 7-14 day delivery for urgent PhD dissertation writing assistance. Our professional writers for PhD dissertation can work around the clock to meet your deadline while maintaining quality standards. We also provide rush services for specific chapters or sections when you need immediate help with your doctoral thesis.' ],
                        [ 'question' => 'How much does professional PhD dissertation writing service cost?', 'answer' => 'Our affordable PhD dissertation writing help online starts at $25.99 per page for PhD dissertation writing, with pricing based on academic level, deadline urgency, and complexity. We offer transparent pricing with no hidden fees, and our doctoral dissertation writing service includes unlimited revisions, plagiarism reports, and 24/7 support. We also provide discounts for international students and bulk orders for multiple chapters.' ],
                        [ 'question' => 'Can you help with specific PhD dissertation methodology sections?', 'answer' => 'Yes, our PhD dissertation help professional includes comprehensive methodology assistance. Our expert PhD thesis writers specialize in research design, data collection methods, statistical analysis, qualitative research methods, and mixed-methods approaches. We help justify your methodological choices, develop research instruments, and ensure your methodology meets university standards for doctoral dissertation writing service excellence.' ],
                        [ 'question' => 'Do you provide samples of your PhD dissertation writing work?', 'answer' => 'Yes, we provide sample dissertation chapters and excerpts (with client permission) to demonstrate our professional dissertation writing service quality. Our samples showcase different academic disciplines, writing styles, and research methodologies. You can also request samples specific to your field of study to see how our PhD-qualified dissertation writers handle similar research topics and academic requirements.' ],
                        [ 'question' => 'What makes your PhD writing services different from competitors?', 'answer' => 'Our professional writers for PhD dissertation are all PhD-qualified with verified credentials from top universities. We offer direct communication with your assigned writer, unlimited revisions, 24/7 emergency support, and comprehensive plagiarism reports. Our doctoral dissertation writing service includes methodology consultation, defense preparation, and post-delivery support. We also maintain strict confidentiality and delete all client work after completion.' ],
                        [ 'question' => 'Can you help international PhD students with dissertation writing?', 'answer' => 'Absolutely! Our PhD dissertation writing assistance is designed for international students who may struggle with English academic writing standards. Our professional dissertation writers understand cultural differences in academic writing and help international PhD students adapt their work to meet Western university standards. We provide language editing, cultural context guidance, and support for non-native English speakers throughout their doctoral journey.' ],
                        [ 'question' => 'What if my dissertation committee requests major revisions?', 'answer' => 'Our professional dissertation writing service includes comprehensive revision support for committee feedback. If your PhD dissertation help requires major revisions based on committee recommendations, our expert writers will work with you to address all concerns at no additional cost. We provide detailed explanations of changes, updated methodology if needed, and ensure your revised dissertation meets all committee requirements and university standards.' ],
                    ],
                ],
            ],
            [
                'block_type' => 'service_cta',
                'block_key' => 'dissertation_cta',
                'order' => 7,
                'content' => [
                    'title' => 'Ready to Get Professional Dissertation Help That Guarantees PhD Success?',
                    'description' => 'Join over 1,000 successful doctoral students who\'ve achieved PhD completion with our expert dissertation writing service. Get professional dissertation help from PhD-qualified writers today.',
                    'benefits' => [
                        [ 'text' => 'PhD-qualified dissertation writers with doctoral expertise', 'icon' => '🏆' ],
                        [ 'text' => '100% original research with advanced methodology', 'icon' => '🔍' ],
                        [ 'text' => 'Perfect citations and doctoral formatting guaranteed', 'icon' => '📄' ],
                        [ 'text' => '24/7 expert support with money-back guarantee', 'icon' => '🛡️' ],
                    ],
                    'primaryButtonText' => 'Get Expert PhD Help Before Deadline',
                    'primaryButtonLink' => '/place-order',
                    'secondaryButtonText' => 'Free Emergency Consultation',
                    'secondaryButtonLink' => '/contact',
                    'bgColor' => 'blue',
                    'urgencyText' => '🚨 LIMITED TIME: 20% OFF for PhD students with tight deadlines',
                    'guarantees' => [
                        '100% Original Research',
                        'On-Time Completion',
                        'Money-Back Guarantee',
                        'Unlimited Revisions',
                        '24/7 Expert Support',
                        'Fast Delivery: 7-14 days',
                    ],
                    'trustBadges' => [
                        'SSL Secured',
                        'Privacy Protected',
                        'PhD-Qualified Writers',
                        'Doctoral Excellence',
                        'Emergency Support Available',
                    ],
                ],
            ],
            [
                'block_type' => 'internal_links',
                'block_key' => 'dissertation_related_services',
                'order' => 8,
                'content' => [
                    'title' => 'Related Academic Writing Services',
                    'description' => 'Explore our comprehensive range of academic writing services designed to support your educational journey',
                    'links' => [
                        [ 'title' => 'Research Paper Writing Service', 'url' => '/services/research-papers', 'description' => 'Professional research paper help and academic research writing' ],
                        [ 'title' => 'Essay Writing Service', 'url' => '/services/essay-writing', 'description' => 'Professional essay writing for all academic levels' ],
                        [ 'title' => 'Literature Review Writing Service', 'url' => '/services/literature-reviews', 'description' => 'Professional literature review assistance and synthesis' ],
                        [ 'title' => 'Thesis Writing Help', 'url' => '/services/thesis-writing', 'description' => 'Master\'s thesis writing assistance and support' ],
                    ],
                ],
            ],
            [
                'block_type' => 'internal_links',
                'block_key' => 'dissertation_support_pages',
                'order' => 9,
                'content' => [
                    'title' => 'Learn More About Our Dissertation Writing Service',
                    'description' => 'Get detailed information about our PhD-qualified writers, process, and pricing for professional dissertation help',
                    'links' => [
                        [ 'title' => 'Our Writers', 'url' => '/writers', 'description' => 'Meet our PhD-qualified dissertation specialists' ],
                        [ 'title' => 'Pricing', 'url' => '/pricing-page', 'description' => 'Transparent dissertation writing pricing' ],
                        [ 'title' => 'Guarantees', 'url' => '/guarantees', 'description' => 'Our quality and satisfaction guarantees' ],
                        [ 'title' => 'Reviews', 'url' => '/reviews', 'description' => 'Read customer reviews and testimonials' ],
                    ],
                ],
            ],
            [
                'block_type' => 'internal_links',
                'block_key' => 'dissertation_resource_links',
                'order' => 10,
                'content' => [
                    'title' => 'Professional Dissertation Writing Service Resources',
                    'links' => [
                        [ 'title' => 'Dissertation Writing Guide', 'url' => '/blog/dissertation-writing-guide', 'description' => 'Complete guide to writing doctoral dissertations' ],
                        [ 'title' => 'Research Methodology Guide', 'url' => '/blog/research-methodology-guide', 'description' => 'Master advanced research methods and design' ],
                        [ 'title' => 'Dissertation Defense Tips', 'url' => '/blog/dissertation-defense-tips', 'description' => 'Prepare for successful dissertation defense' ],
                        [ 'title' => 'PhD Writing Tips', 'url' => '/blog/phd-writing-tips', 'description' => 'Essential tips for doctoral-level writing' ],
                    ],
                ],
            ],
        ];

        // Upsert content blocks
        foreach ($blocks as $i => $block) {
            PageContent::updateOrCreate(
                [
                    'page_id' => $page->id,
                    'block_key' => $block['block_key'],
                ],
                [
                    'block_type' => $block['block_type'],
                    'content' => $block['content'],
                    'order' => $block['order'] ?? ($i + 1),
                    'is_active' => true,
                ]
            );
        }
    }
}
