# Messaging System Setup Guide

## Overview
This document provides instructions for setting up the real-time messaging system in your AcademicScribe application.

## Features Implemented

### ✅ Core Features
- **Floating Chat Widget** - Always accessible from any page
- **Conversation Management** - Create and manage conversations
- **Real-time Messaging** - Instant message delivery with Laravel Echo
- **File Sharing** - Upload and share files in conversations
- **Read Receipts** - Track message read status
- **Unread Count** - Badge showing unread messages
- **User Permissions** - Role-based messaging restrictions

### 🔒 Security Features
- **Client ↔ Admin**: Direct messaging allowed
- **Writer ↔ Admin**: Direct messaging allowed  
- **Client ↔ Writer**: **BLOCKED** (prevents client poaching)
- **Admin ↔ Admin**: Internal admin communication
- **Private Channels** - Secure WebSocket connections
- **File Upload Restrictions** - Size and type limits

## Installation Steps

### 1. Database Setup
The required migrations have been created and should be run:
```bash
php artisan migrate
```

### 2. Dependencies Installation
```bash
# Backend dependencies
composer require pusher/pusher-php-server

# Frontend dependencies
npm install laravel-echo pusher-js
```

### 3. Environment Configuration
Add the following to your `.env` file:

```env
# Broadcasting Configuration
BROADCAST_DRIVER=pusher
PUSHER_APP_ID=your_pusher_app_id
PUSHER_APP_KEY=your_pusher_app_key
PUSHER_APP_SECRET=your_pusher_app_secret
PUSHER_APP_CLUSTER=mt1
PUSHER_HOST=
PUSHER_PORT=443
PUSHER_SCHEME=https
PUSHER_APP_ENCRYPTED=true

# Frontend Pusher Configuration
VITE_PUSHER_APP_KEY="${PUSHER_APP_KEY}"
VITE_PUSHER_APP_CLUSTER="${PUSHER_APP_CLUSTER}"
```

### 4. Pusher Setup
1. Create a free account at [Pusher](https://pusher.com/)
2. Create a new app
3. Copy the app credentials to your `.env` file
4. Enable client events in your Pusher app settings

### 5. Queue Configuration
For real-time messaging, ensure your queue is configured:
```bash
# Start the queue worker
php artisan queue:work

# Or use supervisor for production
```

## Usage Instructions

### For Users

#### Starting a Conversation
1. **Click the floating chat button** in the bottom-right corner
2. **Click the "+" button** to start a new conversation
3. **Search for a user** (admins for clients/writers, anyone for admins)
4. **Select the user** to start chatting

#### Sending Messages
1. **Type your message** in the input field
2. **Press Enter** or click the send button
3. **Attach files** using the paperclip icon (max 10MB)
4. **View real-time updates** as messages arrive

#### Managing Conversations
- **View all conversations** in the conversation list
- **See unread counts** with red badges
- **Minimize/expand** the chat widget
- **Close conversations** to return to the list

### For Admins

#### Managing Multiple Conversations
- **View all active conversations** in the conversation list
- **Respond to multiple users** simultaneously
- **See user types** (Client, Writer, Admin) in the interface
- **Access conversation history** for all users

#### Message Monitoring
- **Real-time notifications** for new messages
- **Read receipts** to track message delivery
- **File sharing capabilities** for documents
- **Conversation management** tools

## Technical Architecture

### Frontend Components
```
resources/js/Components/Messaging/
├── MessagingWidget.jsx      # Main floating chat widget
├── ConversationList.jsx     # Conversation list display
├── MessageList.jsx          # Message display with real-time updates
├── MessageInput.jsx         # Message input with file upload
└── NewConversationModal.jsx # New conversation creation modal
```

### Backend Structure
```
app/
├── Models/
│   ├── Conversation.php     # Conversation model
│   ├── Message.php          # Message model
│   └── MessageRead.php      # Message read tracking
├── Events/
│   ├── MessageSent.php      # Broadcast event for new messages
│   └── MessageRead.php      # Broadcast event for read receipts
├── Http/Controllers/Messaging/
│   └── MessagingController.php # Main messaging controller
└── routes/
    └── web.php              # Messaging routes
```

### Database Tables
- `conversations` - Stores conversation metadata
- `conversation_participants` - Links users to conversations
- `messages` - Stores all messages
- `message_reads` - Tracks message read status

## Real-time Features

### WebSocket Integration
- **Laravel Echo** for real-time communication
- **Pusher** as the WebSocket service
- **Private channels** for secure messaging
- **Event broadcasting** for instant updates

### Real-time Events
- **Message Sent** - Instant message delivery
- **Message Read** - Real-time read receipts
- **Typing Indicators** - Future enhancement
- **Online Status** - Future enhancement

## Security Considerations

### Access Control
- **Role-based permissions** enforced at both frontend and backend
- **Conversation validation** - users can only access their conversations
- **File upload restrictions** - size and type limits enforced
- **CSRF protection** - all requests protected

### Data Protection
- **Private channels** - secure WebSocket connections
- **Message encryption** - ready for future enhancement
- **Audit logging** - all messaging activities tracked
- **Secure file storage** - files stored in protected directory

## Troubleshooting

### Common Issues

#### Messages Not Sending
1. Check if Pusher credentials are correct
2. Ensure queue worker is running
3. Verify WebSocket connection in browser console
4. Check for JavaScript errors

#### Real-time Not Working
1. Verify Pusher app configuration
2. Check if Laravel Echo is properly initialized
3. Ensure private channels are authorized
4. Check browser console for connection errors

#### File Upload Issues
1. Verify file size limits (10MB max)
2. Check allowed file types
3. Ensure storage directory is writable
4. Check file permissions

### Debug Mode
Enable debug mode for detailed error messages:
```env
APP_DEBUG=true
BROADCAST_DRIVER=log
```

## Future Enhancements

### Planned Features
- **Typing indicators** - Show when someone is typing
- **Message reactions** - Like, thumbs up, etc.
- **Voice messages** - Audio recording support
- **Video calls** - Integrated video chat
- **Message search** - Search through conversation history
- **Message encryption** - End-to-end encryption
- **Admin dashboard** - Messaging analytics and management

### Performance Optimizations
- **Message pagination** - Load messages in chunks
- **Image optimization** - Compress uploaded images
- **Caching** - Cache frequently accessed data
- **Database indexing** - Optimize query performance

## Support

For technical support or questions about the messaging system:
1. Check the troubleshooting section above
2. Review the Laravel Echo documentation
3. Check Pusher documentation for WebSocket issues
4. Contact the development team for assistance 