# Order File Management System

## Overview

The Order File Management System provides a streamlined approach to handling files throughout the order lifecycle with proper access control and categorization. This system ensures that clients, writers, and admins can only access files appropriate to their role and the current order status.

## Key Features

### 1. File Categorization
Files are automatically categorized based on their purpose and uploader:

- **Sample Work** (`sample`): Client-uploaded examples of previous work
- **Order Instructions** (`instructions`): Client-uploaded requirements and specifications
- **Draft** (`draft`): Writer-uploaded work-in-progress files
- **Complete Work** (`complete`): Writer-uploaded final deliverables
- **Admin Files** (`admin`): Administrative documents and communications
- **General** (`general`): Miscellaneous files

### 2. Access Control
Files have different access levels based on user roles:

- **Admin Only** (`admin_only`): Only administrators can access
- **Client-Writer** (`client_writer`): Both clients and writers can access
- **Client-Admin** (`client_admin`): Clients and administrators can access
- **Writer-Admin** (`writer_admin`): Writers and administrators can access

### 3. User Role Separation
- **Clients**: Can only upload sample work and order instructions
- **Writers**: Can only upload drafts and complete work
- **Admins**: Can access all files and upload administrative documents

## Implementation Details

### Database Schema

The `order_files` table includes these new fields:

```sql
ALTER TABLE order_files ADD COLUMN file_category VARCHAR(255) DEFAULT 'general';
ALTER TABLE order_files ADD COLUMN description TEXT NULL;
ALTER TABLE order_files ADD COLUMN access_control VARCHAR(255) DEFAULT 'admin_only';
```

### API Endpoints

#### Client File Upload
```
POST /order-files/client-upload
```

**Parameters:**
- `file`: The file to upload (max 40MB)
- `order_id`: ID of the order
- `file_category`: Either 'sample' or 'instructions'
- `description`: Optional description of the file
- `uploader_user_type`: Must be 'client'

**Response:**
```json
{
    "success": true,
    "message": "File uploaded successfully",
    "file": {
        "id": 1,
        "file_name": "example.pdf",
        "file_category": "instructions",
        "description": "Detailed requirements",
        "access_control": "client_admin"
    }
}
```

#### Get Order Files
```
GET /order-files/{orderId}
```

**Response:**
```json
{
    "success": true,
    "files": [
        {
            "id": 1,
            "file_name": "example.pdf",
            "file_category": "instructions",
            "uploader_user_type": "client",
            "canBeAccessedBy": ["client", "admin"]
        }
    ]
}
```

### Frontend Components

#### ClientFileUpload
A React component that allows clients to:
- Select file type (sample or instructions)
- Provide file description
- Upload files with drag-and-drop support
- See upload progress
- Handle multiple file uploads

#### OrderFileDisplay
A React component that displays files with:
- Category-based filtering
- Role-based access control
- File download functionality
- File deletion (for own files or admin)
- Visual indicators for file types and uploaders

## File Upload Flow

### Client Upload Process
1. Client navigates to order page
2. Selects file type (sample or instructions)
3. Provides description
4. Uploads file(s)
5. System stores file with `client_admin` access
6. Admin receives notification
7. File appears in order file list

### Writer Upload Process (Future Implementation)
1. Writer navigates to assigned order
2. Selects file type (draft or complete)
3. Provides description
4. Uploads file(s)
5. System stores file with `writer_admin` access
6. Admin receives notification
7. If complete file, order status updates

## Security Features

### Access Control
- Files are automatically assigned appropriate access levels
- Users can only see files they have permission to access
- File downloads are logged and validated

### File Validation
- File type restrictions (DOC, DOCX, XLS, XLSX, PPT, PPTX, PDF, ZIP)
- File size limits (40MB maximum)
- Malicious file detection and prevention

### User Authentication
- All file operations require authentication
- Users can only upload to their own orders
- Admin access is validated for administrative operations

## Notifications

### Admin Notifications
When clients upload files, administrators receive:
- Email notifications
- Database notifications
- Real-time updates in admin dashboard

### Notification Content
- Order number and client name
- File name and category
- Upload timestamp
- Direct link to order

## Future Enhancements

### Writer File Management
- Draft upload functionality
- Complete work submission
- Automatic order status updates
- Revision request handling

### Advanced Features
- File versioning
- File approval workflows
- Bulk file operations
- File preview capabilities
- Integration with cloud storage

## Usage Examples

### Client Uploading Sample Work
```javascript
// Client selects "Sample Work" category
// Provides description: "Previous essay on similar topic"
// Uploads PDF file
// System categorizes as 'sample' with 'client_admin' access
```

### Admin Reviewing Files
```javascript
// Admin sees notification of new client upload
// Navigates to order page
// Views all files with proper categorization
// Can download and review client instructions
```

## Troubleshooting

### Common Issues
1. **File Upload Fails**: Check file size and type restrictions
2. **Access Denied**: Verify user role and file permissions
3. **Missing Files**: Check file categorization and access control

### Debug Information
- Check Laravel logs for upload errors
- Verify database permissions
- Confirm file storage configuration
- Check notification system status

## Configuration

### Environment Variables
```env
# File upload settings
UPLOAD_MAX_FILESIZE=40M
POST_MAX_SIZE=40M

# Storage configuration
FILESYSTEM_DISK=public
```

### File Storage
Files are stored in `storage/app/public/orders/` with public access for downloads.

## Support

For technical support or questions about the file management system, please contact the development team or refer to the Laravel documentation for file handling best practices. 