# Writer Order File Management System

## Overview

The Writer Order File Management System allows writers to upload files to orders they are assigned to, with two distinct file categories that affect the order workflow differently.

## Features

### File Categories

1. **Draft Files** (`draft`)
   - Work-in-progress files
   - No impact on order timeline or status
   - Writers can upload multiple drafts as they work

2. **Complete Files** (`complete`)
   - Final deliverables
   - Automatically updates order status to "Under Review"
   - Updates order timeline fields
   - Triggers client and admin notifications

### Access Control

- **Writer files**: `writer_admin` access (writer + admin only)
- **Client files**: `client_admin` access (client + admin only)
- **Admin files**: `admin_only` access

## API Endpoints

### Writer File Upload
```
POST /order-files/writer-upload
```

**Request Body:**
```json
{
    "file": "file_upload",
    "order_id": "integer",
    "file_category": "draft|complete",
    "description": "string (optional, max 1000 chars)",
    "uploader_user_type": "writer"
}
```

**Response:**
```json
{
    "success": true,
    "message": "File uploaded successfully",
    "file": "OrderFile object",
    "order_status_updated": "boolean"
}
```

### Get Order Files
```
GET /order-files/{orderId}
```

Returns files accessible to the authenticated user based on their role.

## Business Logic

### Draft Uploads
- File is stored with `draft` category
- Order status remains unchanged
- Admin is notified of file upload
- No timeline impact

### Complete Uploads
- File is stored with `complete` category
- Order status changes from `in_progress` to `under_review`
- Timeline fields updated:
  - `work_submitted_at` = current timestamp
  - `status_changed_at` = current timestamp
  - `status_changed_by` = writer ID
  - `status_change_reason` = "Writer submitted complete work"
- Admin is notified of file upload
- Client is notified of work completion

## File Validation

- **File types**: DOC, DOCX, XLS, XLSX, PPT, PPTX, PDF, ZIP
- **File size**: Maximum 40MB
- **Required fields**: file, order_id, file_category, uploader_user_type
- **Optional fields**: description

## Security

- Writers can only upload to orders they are assigned to
- File access is controlled by role-based permissions
- All uploads are logged with user and order information

## Notifications

### Admin Notifications
- Sent for all file uploads (draft and complete)
- Uses `AdminOrderFileUploadedNotification`

### Client Notifications
- Sent only for complete file uploads
- Uses `ClientWorkCompletedNotification`
- Informs client that work is complete and under admin review

## React Components

### WriterFileUpload.jsx
- File selection with drag & drop
- Category selection (draft/complete)
- Optional description field
- Real-time validation
- Success/error feedback
- Automatic form reset after successful upload

### Integration with Writer Show Page
- Replaces generic FileUpload component
- Provides real-time order status updates
- Maintains local state for immediate UI feedback

## Database Changes

The system uses the existing `order_files` table with the following key fields:

- `file_category`: `draft` or `complete`
- `access_control`: `writer_admin` for writer files
- `uploader_user_type`: `writer` for writer uploads
- `description`: Optional file description

## Testing

A comprehensive test suite (`WriterFileUploadTest.php`) covers:

- Draft file uploads (no status change)
- Complete file uploads (status update)
- Permission validation
- Timeline field updates
- Error handling

## Usage Examples

### Uploading a Draft
```javascript
// Writer uploads work-in-progress
const formData = new FormData();
formData.append('file', draftFile);
formData.append('order_id', orderId);
formData.append('file_category', 'draft');
formData.append('description', 'First draft - introduction complete');
formData.append('uploader_user_type', 'writer');

await axios.post('/order-files/writer-upload', formData);
```

### Uploading Complete Work
```javascript
// Writer submits final deliverable
const formData = new FormData();
formData.append('file', finalFile);
formData.append('order_id', orderId);
formData.append('file_category', 'complete');
formData.append('description', 'Final submission - all requirements met');
formData.append('uploader_user_type', 'writer');

const response = await axios.post('/order-files/writer-upload', formData);
if (response.data.order_status_updated) {
    // Order status changed to "Under Review"
    console.log('Order moved to admin review');
}
```

## Future Enhancements

- File versioning for drafts
- Bulk file uploads
- File preview capabilities
- Integration with plagiarism detection
- Advanced file categorization
- File approval workflows 