import React, { useState } from 'react';
import { Card, CardContent } from '@/Components/ui/Card';
import { Button } from '@/Components/ui/Button';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import RichTextEditor from '@/Components/Admin/RichTextEditor';
import { Trash2, Plus, GripVertical } from 'lucide-react';

export default function GuaranteesGridBlockEditor({ block, onUpdate }) {
    const [data, setData] = useState({
        title: block.content?.title || 'Our Core Guarantees',
        description: block.content?.description || '',
        guarantees: block.content?.guarantees || []
    });

    const updateData = (field, value) => {
        const newData = { ...data, [field]: value };
        setData(newData);
    };

    const updateGuarantee = (index, field, value) => {
        const newGuarantees = [...data.guarantees];
        newGuarantees[index] = { ...newGuarantees[index], [field]: value };
        setData({ ...data, guarantees: newGuarantees });
    };

    const updateGuaranteeFeature = (guaranteeIndex, featureIndex, value) => {
        const newGuarantees = [...data.guarantees];
        const newFeatures = [...(newGuarantees[guaranteeIndex].features || [])];
        newFeatures[featureIndex] = value;
        newGuarantees[guaranteeIndex] = { 
            ...newGuarantees[guaranteeIndex], 
            features: newFeatures 
        };
        setData({ ...data, guarantees: newGuarantees });
    };

    const addGuarantee = () => {
        setData({
            ...data,
            guarantees: [
                ...data.guarantees,
                {
                    icon: '🔒',
                    title: 'New Guarantee',
                    description: 'Description here...',
                    features: [],
                    link: '',
                    linkText: ''
                }
            ]
        });
    };

    const removeGuarantee = (index) => {
        const newGuarantees = data.guarantees.filter((_, i) => i !== index);
        setData({ ...data, guarantees: newGuarantees });
    };

    const addFeature = (guaranteeIndex) => {
        const newGuarantees = [...data.guarantees];
        newGuarantees[guaranteeIndex].features = [
            ...(newGuarantees[guaranteeIndex].features || []),
            'New feature'
        ];
        setData({ ...data, guarantees: newGuarantees });
    };

    const removeFeature = (guaranteeIndex, featureIndex) => {
        const newGuarantees = [...data.guarantees];
        newGuarantees[guaranteeIndex].features = newGuarantees[guaranteeIndex].features.filter(
            (_, i) => i !== featureIndex
        );
        setData({ ...data, guarantees: newGuarantees });
    };

    const handleSave = () => {
        onUpdate(block.id, data);
    };

    return (
        <div className="space-y-6">
            {/* Section Title and Description */}
            <Card>
                <CardContent className="pt-6">
                    <div className="space-y-4">
                        <div>
                            <Label htmlFor="section-title">Section Title</Label>
                            <Input
                                id="section-title"
                                value={data.title}
                                onChange={(e) => updateData('title', e.target.value)}
                                placeholder="Our Core Guarantees"
                            />
                        </div>
                        <div>
                            <Label htmlFor="section-description">Section Description (Optional)</Label>
                            <RichTextEditor
                                value={data.description}
                                onChange={(value) => updateData('description', value)}
                                height={200}
                                placeholder="Optional description for the guarantees section..."
                            />
                        </div>
                    </div>
                </CardContent>
            </Card>

            {/* Guarantees */}
            <div className="space-y-4">
                <div className="flex items-center justify-between">
                    <h3 className="text-lg font-semibold">Guarantees ({data.guarantees.length})</h3>
                    <Button onClick={addGuarantee} size="sm">
                        <Plus className="h-4 w-4 mr-2" />
                        Add Guarantee
                    </Button>
                </div>

                {data.guarantees.map((guarantee, gIndex) => (
                    <Card key={gIndex} className="border-l-4 border-l-blue-500">
                        <CardContent className="pt-6">
                            <div className="space-y-4">
                                {/* Header */}
                                <div className="flex items-center justify-between">
                                    <div className="flex items-center space-x-2">
                                        <GripVertical className="h-5 w-5 text-gray-400" />
                                        <span className="font-semibold">Guarantee #{gIndex + 1}</span>
                                    </div>
                                    <Button
                                        variant="destructive"
                                        size="sm"
                                        onClick={() => removeGuarantee(gIndex)}
                                    >
                                        <Trash2 className="h-4 w-4" />
                                    </Button>
                                </div>

                                {/* Icon */}
                                <div>
                                    <Label>Icon (emoji)</Label>
                                    <Input
                                        value={guarantee.icon}
                                        onChange={(e) => updateGuarantee(gIndex, 'icon', e.target.value)}
                                        placeholder="🔒"
                                        className="text-2xl"
                                    />
                                </div>

                                {/* Title */}
                                <div>
                                    <Label>Guarantee Title</Label>
                                    <Input
                                        value={guarantee.title}
                                        onChange={(e) => updateGuarantee(gIndex, 'title', e.target.value)}
                                        placeholder="100% Plagiarism-Free Guarantee"
                                    />
                                </div>

                                {/* Description with rich text editor */}
                                <div>
                                    <Label>Description</Label>
                                    <RichTextEditor
                                        value={guarantee.description}
                                        onChange={(value) => updateGuarantee(gIndex, 'description', value)}
                                        height={250}
                                        placeholder="Describe this guarantee. You can add links to related services..."
                                    />
                                    <p className="text-sm text-gray-500 mt-1">
                                        Tip: Add internal links to related services for better SEO
                                    </p>
                                </div>

                                {/* Features */}
                                <div>
                                    <div className="flex items-center justify-between mb-2">
                                        <Label>Features (bullet points)</Label>
                                        <Button
                                            variant="outline"
                                            size="sm"
                                            onClick={() => addFeature(gIndex)}
                                        >
                                            <Plus className="h-3 w-3 mr-1" />
                                            Add Feature
                                        </Button>
                                    </div>
                                    <div className="space-y-2">
                                        {(guarantee.features || []).map((feature, fIndex) => (
                                            <div key={fIndex} className="flex items-center space-x-2">
                                                <span className="text-green-500">✓</span>
                                                <Input
                                                    value={feature}
                                                    onChange={(e) => updateGuaranteeFeature(gIndex, fIndex, e.target.value)}
                                                    placeholder="Feature description"
                                                    className="flex-1"
                                                />
                                                <Button
                                                    variant="ghost"
                                                    size="sm"
                                                    onClick={() => removeFeature(gIndex, fIndex)}
                                                >
                                                    <Trash2 className="h-4 w-4 text-red-500" />
                                                </Button>
                                            </div>
                                        ))}
                                    </div>
                                </div>

                                {/* Link */}
                                <div className="grid grid-cols-2 gap-4">
                                    <div>
                                        <Label>Link URL (optional)</Label>
                                        <Input
                                            value={guarantee.link}
                                            onChange={(e) => updateGuarantee(gIndex, 'link', e.target.value)}
                                            placeholder="/services/essay-writing"
                                        />
                                    </div>
                                    <div>
                                        <Label>Link Text</Label>
                                        <Input
                                            value={guarantee.linkText}
                                            onChange={(e) => updateGuarantee(gIndex, 'linkText', e.target.value)}
                                            placeholder="Learn More →"
                                        />
                                    </div>
                                </div>
                            </div>
                        </CardContent>
                    </Card>
                ))}

                {data.guarantees.length === 0 && (
                    <div className="text-center py-8 border-2 border-dashed border-gray-300 rounded-lg">
                        <p className="text-gray-500 mb-4">No guarantees added yet</p>
                        <Button onClick={addGuarantee}>
                            <Plus className="h-4 w-4 mr-2" />
                            Add Your First Guarantee
                        </Button>
                    </div>
                )}
            </div>

            {/* Save Button */}
            <div className="flex justify-end pt-4 border-t">
                <Button onClick={handleSave} size="lg">
                    Save Changes
                </Button>
            </div>
        </div>
    );
}





