import React, { useState } from 'react';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import { Button } from '@/Components/ui/Button';
import RichTextEditor from '@/Components/Admin/RichTextEditor';
import { Plus, X, ChevronUp, ChevronDown, Save } from 'lucide-react';

export default function PricingResourcesBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [resourceCategories, setResourceCategories] = useState(content.resourceCategories || []);

    const handleSave = () => {
        onUpdate(block.id, { ...content, resourceCategories });
    };

    const addCategory = () => {
        setResourceCategories([...resourceCategories, { 
            title: '', 
            links: []
        }]);
    };

    const removeCategory = (index) => {
        setResourceCategories(resourceCategories.filter((_, i) => i !== index));
    };

    const updateCategory = (index, field, value) => {
        const updated = [...resourceCategories];
        updated[index] = { ...updated[index], [field]: value };
        setResourceCategories(updated);
    };

    const addLink = (categoryIndex) => {
        const updated = [...resourceCategories];
        updated[categoryIndex].links = [...(updated[categoryIndex].links || []), { 
            text: '', 
            url: '', 
            external: false 
        }];
        setResourceCategories(updated);
    };

    const removeLink = (categoryIndex, linkIndex) => {
        const updated = [...resourceCategories];
        updated[categoryIndex].links = updated[categoryIndex].links.filter((_, i) => i !== linkIndex);
        setResourceCategories(updated);
    };

    const updateLink = (categoryIndex, linkIndex, field, value) => {
        const updated = [...resourceCategories];
        updated[categoryIndex].links[linkIndex] = { 
            ...updated[categoryIndex].links[linkIndex], 
            [field]: value 
        };
        setResourceCategories(updated);
    };

    const moveCategoryUp = (index) => {
        if (index === 0) return;
        const updated = [...resourceCategories];
        [updated[index - 1], updated[index]] = [updated[index], updated[index - 1]];
        setResourceCategories(updated);
    };

    const moveCategoryDown = (index) => {
        if (index === resourceCategories.length - 1) return;
        const updated = [...resourceCategories];
        [updated[index], updated[index + 1]] = [updated[index + 1], updated[index]];
        setResourceCategories(updated);
    };

    return (
        <div className="space-y-6">
            <div>
                <Label htmlFor="title">Section Title</Label>
                <Input
                    id="title"
                    value={content.title || ''}
                    onChange={(e) => setContent({...content, title: e.target.value})}
                />
            </div>

            <div>
                <Label htmlFor="description">Section Description</Label>
                <RichTextEditor
                    value={content.description || ''}
                    onChange={(value) => setContent({...content, description: value})}
                    height={150}
                />
            </div>

            <div className="border-t pt-4">
                <div className="flex justify-between items-center mb-4">
                    <Label>Resource Categories ({resourceCategories.length})</Label>
                    <Button
                        type="button"
                        onClick={addCategory}
                        className="bg-green-600 hover:bg-green-700 text-white"
                    >
                        <Plus className="h-4 w-4 mr-1" />
                        Add Category
                    </Button>
                </div>

                <div className="space-y-6">
                    {resourceCategories.map((category, categoryIndex) => (
                        <div key={categoryIndex} className="border border-gray-200 rounded-lg p-4 bg-gray-50">
                            <div className="flex justify-between items-center mb-3">
                                <span className="text-sm font-semibold text-gray-700">Category #{categoryIndex + 1}</span>
                                <div className="flex gap-2">
                                    <button
                                        type="button"
                                        onClick={() => moveCategoryUp(categoryIndex)}
                                        disabled={categoryIndex === 0}
                                        className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                        title="Move up"
                                    >
                                        <ChevronUp className="h-4 w-4" />
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => moveCategoryDown(categoryIndex)}
                                        disabled={categoryIndex === resourceCategories.length - 1}
                                        className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                        title="Move down"
                                    >
                                        <ChevronDown className="h-4 w-4" />
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => removeCategory(categoryIndex)}
                                        className="p-1 text-red-500 hover:text-red-700"
                                        title="Remove category"
                                    >
                                        <X className="h-4 w-4" />
                                    </button>
                                </div>
                            </div>

                            <div className="space-y-3">
                                <div>
                                    <Label htmlFor={`categoryTitle-${categoryIndex}`}>Category Title</Label>
                                    <Input
                                        id={`categoryTitle-${categoryIndex}`}
                                        value={category.title || ''}
                                        onChange={(e) => updateCategory(categoryIndex, 'title', e.target.value)}
                                        placeholder="e.g., 📚 Essay Writing Guides"
                                    />
                                </div>

                                <div>
                                    <div className="flex justify-between items-center mb-2">
                                        <Label>Links ({category.links?.length || 0})</Label>
                                        <Button
                                            type="button"
                                            onClick={() => addLink(categoryIndex)}
                                            className="bg-blue-600 hover:bg-blue-700 text-white text-sm"
                                        >
                                            <Plus className="h-3 w-3 mr-1" />
                                            Add Link
                                        </Button>
                                    </div>

                                    <div className="space-y-3">
                                        {category.links?.map((link, linkIndex) => (
                                            <div key={linkIndex} className="border border-gray-300 rounded p-3 bg-white">
                                                <div className="flex justify-between items-center mb-2">
                                                    <span className="text-xs font-medium text-gray-600">Link #{linkIndex + 1}</span>
                                                    <button
                                                        type="button"
                                                        onClick={() => removeLink(categoryIndex, linkIndex)}
                                                        className="p-1 text-red-500 hover:text-red-700"
                                                        title="Remove link"
                                                    >
                                                        <X className="h-3 w-3" />
                                                    </button>
                                                </div>
                                                <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                                                    <div>
                                                        <Label htmlFor={`linkText-${categoryIndex}-${linkIndex}`}>Link Text</Label>
                                                        <Input
                                                            id={`linkText-${categoryIndex}-${linkIndex}`}
                                                            value={link.text || ''}
                                                            onChange={(e) => updateLink(categoryIndex, linkIndex, 'text', e.target.value)}
                                                            placeholder="e.g., How to Choose Essay Writing Service"
                                                        />
                                                    </div>
                                                    <div>
                                                        <Label htmlFor={`linkUrl-${categoryIndex}-${linkIndex}`}>Link URL</Label>
                                                        <Input
                                                            id={`linkUrl-${categoryIndex}-${linkIndex}`}
                                                            value={link.url || ''}
                                                            onChange={(e) => updateLink(categoryIndex, linkIndex, 'url', e.target.value)}
                                                            placeholder="/blog/essay-writing-guide"
                                                        />
                                                    </div>
                                                </div>
                                                <div className="mt-2">
                                                    <label className="flex items-center">
                                                        <input
                                                            type="checkbox"
                                                            checked={link.external || false}
                                                            onChange={(e) => updateLink(categoryIndex, linkIndex, 'external', e.target.checked)}
                                                            className="mr-2"
                                                        />
                                                        <span className="text-sm text-gray-600">External link (opens in new tab)</span>
                                                    </label>
                                                </div>
                                            </div>
                                        ))}

                                        {(!category.links || category.links.length === 0) && (
                                            <div className="text-center py-4 text-gray-500 border-2 border-dashed border-gray-300 rounded-lg">
                                                No links yet. Click "Add Link" to get started.
                                            </div>
                                        )}
                                    </div>
                                </div>
                            </div>
                        </div>
                    ))}

                    {resourceCategories.length === 0 && (
                        <div className="text-center py-8 text-gray-500 border-2 border-dashed border-gray-300 rounded-lg">
                            No categories yet. Click "Add Category" to get started.
                        </div>
                    )}
                </div>
            </div>
            
            <Button
                type="button"
                onClick={handleSave}
                className="bg-blue-600 hover:bg-blue-700 text-white"
            >
                <Save className="h-4 w-4 mr-2" />
                Save Changes
            </Button>
        </div>
    );
}



