import React, { useState } from 'react';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import { Button } from '@/Components/ui/Button';
import RichTextEditor from '@/Components/Admin/RichTextEditor';
import { Plus, X, ChevronUp, ChevronDown, Save } from 'lucide-react';

export default function PricingSecurityBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [securityFeatures, setSecurityFeatures] = useState(content.securityFeatures || []);

    const handleSave = () => {
        onUpdate(block.id, { ...content, securityFeatures });
    };

    const addFeature = () => {
        setSecurityFeatures([...securityFeatures, { 
            title: '', 
            description: '',
            iconBgColor: 'bg-green-100',
            iconColor: 'text-green-600',
            iconPath: 'M2.166 4.999A11.954 11.954 0 0010 1.944 11.954 11.954 0 0017.834 5c.11.65.166 1.32.166 2.001 0 5.225-3.34 9.67-8 11.317C5.34 16.67 2 12.225 2 7c0-.682.057-1.35.166-2.001zm11.541 3.708a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z'
        }]);
    };

    const removeFeature = (index) => {
        setSecurityFeatures(securityFeatures.filter((_, i) => i !== index));
    };

    const updateFeature = (index, field, value) => {
        const updated = [...securityFeatures];
        updated[index] = { ...updated[index], [field]: value };
        setSecurityFeatures(updated);
    };

    const moveFeatureUp = (index) => {
        if (index === 0) return;
        const updated = [...securityFeatures];
        [updated[index - 1], updated[index]] = [updated[index], updated[index - 1]];
        setSecurityFeatures(updated);
    };

    const moveFeatureDown = (index) => {
        if (index === securityFeatures.length - 1) return;
        const updated = [...securityFeatures];
        [updated[index], updated[index + 1]] = [updated[index + 1], updated[index]];
        setSecurityFeatures(updated);
    };

    const iconPresets = [
        {
            name: 'SSL Encryption',
            bgColor: 'bg-green-100',
            color: 'text-green-600',
            path: 'M2.166 4.999A11.954 11.954 0 0010 1.944 11.954 11.954 0 0017.834 5c.11.65.166 1.32.166 2.001 0 5.225-3.34 9.67-8 11.317C5.34 16.67 2 12.225 2 7c0-.682.057-1.35.166-2.001zm11.541 3.708a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z'
        },
        {
            name: 'Privacy Protected',
            bgColor: 'bg-blue-100',
            color: 'text-blue-600',
            path: 'M5 9V7a5 5 0 0110 0v2a2 2 0 012 2v5a2 2 0 01-2 2H5a2 2 0 01-2-2v-5a2 2 0 012-2zm8-2v2H7V7a3 3 0 016 0z'
        },
        {
            name: 'Secure Payments',
            bgColor: 'bg-purple-100',
            color: 'text-purple-600',
            path: 'M4 4a2 2 0 00-2 2v4a2 2 0 002 2V6h10a2 2 0 00-2-2H4zm2 6a2 2 0 012-2h8a2 2 0 012 2v4a2 2 0 01-2 2H8a2 2 0 01-2-2v-4zm6 4a2 2 0 100-4 2 2 0 000 4z'
        },
        {
            name: 'Verified Service',
            bgColor: 'bg-yellow-100',
            color: 'text-yellow-600',
            path: 'M6.267 3.455a3.066 3.066 0 001.745-.723 3.066 3.066 0 013.976 0 3.066 3.066 0 001.745.723 3.066 3.066 0 012.812 2.812c.051.643.304 1.254.723 1.745a3.066 3.066 0 010 3.976 3.066 3.066 0 00-.723 1.745 3.066 3.066 0 01-2.812 2.812 3.066 3.066 0 00-1.745.723 3.066 3.066 0 01-3.976 0 3.066 3.066 0 00-1.745-.723 3.066 3.066 0 01-2.812-2.812 3.066 3.066 0 00-.723-1.745 3.066 3.066 0 010-3.976 3.066 3.066 0 00.723-1.745 3.066 3.066 0 012.812-2.812zm7.44 5.252a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z'
        }
    ];

    return (
        <div className="space-y-6">
            <div>
                <Label htmlFor="title">Section Title</Label>
                <Input
                    id="title"
                    value={content.title || ''}
                    onChange={(e) => setContent({...content, title: e.target.value})}
                />
            </div>

            <div>
                <Label htmlFor="description">Section Description</Label>
                <RichTextEditor
                    value={content.description || ''}
                    onChange={(value) => setContent({...content, description: value})}
                    height={150}
                />
            </div>

            <div className="border-t pt-4">
                <div className="flex justify-between items-center mb-4">
                    <Label>Security Features ({securityFeatures.length})</Label>
                    <Button
                        type="button"
                        onClick={addFeature}
                        className="bg-green-600 hover:bg-green-700 text-white"
                    >
                        <Plus className="h-4 w-4 mr-1" />
                        Add Feature
                    </Button>
                </div>

                <div className="space-y-6">
                    {securityFeatures.map((feature, index) => (
                        <div key={index} className="border border-gray-200 rounded-lg p-4 bg-gray-50">
                            <div className="flex justify-between items-center mb-3">
                                <span className="text-sm font-semibold text-gray-700">Feature #{index + 1}</span>
                                <div className="flex gap-2">
                                    <button
                                        type="button"
                                        onClick={() => moveFeatureUp(index)}
                                        disabled={index === 0}
                                        className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                        title="Move up"
                                    >
                                        <ChevronUp className="h-4 w-4" />
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => moveFeatureDown(index)}
                                        disabled={index === securityFeatures.length - 1}
                                        className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                        title="Move down"
                                    >
                                        <ChevronDown className="h-4 w-4" />
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => removeFeature(index)}
                                        className="p-1 text-red-500 hover:text-red-700"
                                        title="Remove feature"
                                    >
                                        <X className="h-4 w-4" />
                                    </button>
                                </div>
                            </div>

                            <div className="space-y-3">
                                <div>
                                    <Label htmlFor={`title-${index}`}>Feature Title</Label>
                                    <Input
                                        id={`title-${index}`}
                                        value={feature.title || ''}
                                        onChange={(e) => updateFeature(index, 'title', e.target.value)}
                                        placeholder="e.g., SSL Encryption"
                                    />
                                </div>

                                <div>
                                    <Label htmlFor={`description-${index}`}>Description</Label>
                                    <RichTextEditor
                                        value={feature.description || ''}
                                        onChange={(value) => updateFeature(index, 'description', value)}
                                        height={100}
                                    />
                                </div>

                                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                                    <div>
                                        <Label htmlFor={`iconBgColor-${index}`}>Icon Background Color</Label>
                                        <select
                                            id={`iconBgColor-${index}`}
                                            value={feature.iconBgColor || ''}
                                            onChange={(e) => updateFeature(index, 'iconBgColor', e.target.value)}
                                            className="w-full px-3 py-2 border border-gray-300 rounded-md"
                                        >
                                            <option value="bg-green-100">Green</option>
                                            <option value="bg-blue-100">Blue</option>
                                            <option value="bg-purple-100">Purple</option>
                                            <option value="bg-yellow-100">Yellow</option>
                                            <option value="bg-red-100">Red</option>
                                        </select>
                                    </div>
                                    <div>
                                        <Label htmlFor={`iconColor-${index}`}>Icon Color</Label>
                                        <select
                                            id={`iconColor-${index}`}
                                            value={feature.iconColor || ''}
                                            onChange={(e) => updateFeature(index, 'iconColor', e.target.value)}
                                            className="w-full px-3 py-2 border border-gray-300 rounded-md"
                                        >
                                            <option value="text-green-600">Green</option>
                                            <option value="text-blue-600">Blue</option>
                                            <option value="text-purple-600">Purple</option>
                                            <option value="text-yellow-600">Yellow</option>
                                            <option value="text-red-600">Red</option>
                                        </select>
                                    </div>
                                    <div>
                                        <Label htmlFor={`iconPreset-${index}`}>Icon Preset</Label>
                                        <select
                                            id={`iconPreset-${index}`}
                                            onChange={(e) => {
                                                const preset = iconPresets[parseInt(e.target.value)];
                                                if (preset) {
                                                    updateFeature(index, 'iconBgColor', preset.bgColor);
                                                    updateFeature(index, 'iconColor', preset.color);
                                                    updateFeature(index, 'iconPath', preset.path);
                                                }
                                            }}
                                            className="w-full px-3 py-2 border border-gray-300 rounded-md"
                                        >
                                            <option value="">Select preset...</option>
                                            {iconPresets.map((preset, presetIndex) => (
                                                <option key={presetIndex} value={presetIndex}>{preset.name}</option>
                                            ))}
                                        </select>
                                    </div>
                                </div>

                                <div>
                                    <Label htmlFor={`iconPath-${index}`}>Icon SVG Path</Label>
                                    <Input
                                        id={`iconPath-${index}`}
                                        value={feature.iconPath || ''}
                                        onChange={(e) => updateFeature(index, 'iconPath', e.target.value)}
                                        placeholder="SVG path data"
                                    />
                                </div>
                            </div>
                        </div>
                    ))}

                    {securityFeatures.length === 0 && (
                        <div className="text-center py-8 text-gray-500 border-2 border-dashed border-gray-300 rounded-lg">
                            No security features yet. Click "Add Feature" to get started.
                        </div>
                    )}
                </div>
            </div>

            {/* Payment Methods Section */}
            <div className="border-t pt-4">
                <h3 className="text-lg font-semibold mb-4">Payment Methods Section</h3>
                <div>
                    <Label htmlFor="paymentMethodsTitle">Payment Methods Title</Label>
                    <Input
                        id="paymentMethodsTitle"
                        value={content.paymentMethods?.title || ''}
                        onChange={(e) => setContent({
                            ...content, 
                            paymentMethods: {...content.paymentMethods, title: e.target.value}
                        })}
                        placeholder="Accepted Payment Methods"
                    />
                </div>
                <div className="mt-4">
                    <Label>Payment Methods</Label>
                    <div className="space-y-2">
                        {content.paymentMethods?.methods?.map((method, index) => (
                            <div key={index} className="flex gap-2">
                                <Input
                                    value={method.name || ''}
                                    onChange={(e) => {
                                        const updated = [...(content.paymentMethods?.methods || [])];
                                        updated[index] = { ...updated[index], name: e.target.value };
                                        setContent({
                                            ...content, 
                                            paymentMethods: {...content.paymentMethods, methods: updated}
                                        });
                                    }}
                                    placeholder="Method name (e.g., VISA)"
                                />
                                <select
                                    value={method.color || ''}
                                    onChange={(e) => {
                                        const updated = [...(content.paymentMethods?.methods || [])];
                                        updated[index] = { ...updated[index], color: e.target.value };
                                        setContent({
                                            ...content, 
                                            paymentMethods: {...content.paymentMethods, methods: updated}
                                        });
                                    }}
                                    className="px-3 py-2 border border-gray-300 rounded-md"
                                >
                                    <option value="text-blue-600">Blue</option>
                                    <option value="text-red-600">Red</option>
                                    <option value="text-blue-800">Dark Blue</option>
                                </select>
                            </div>
                        )) || []}
                    </div>
                </div>
            </div>
            
            <Button
                type="button"
                onClick={handleSave}
                className="bg-blue-600 hover:bg-blue-700 text-white"
            >
                <Save className="h-4 w-4 mr-2" />
                Save Changes
            </Button>
        </div>
    );
}



