import React, { useState } from 'react';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import { Button } from '@/Components/ui/Button';
import RichTextEditor from '@/Components/Admin/RichTextEditor';
import { Plus, X, ChevronUp, ChevronDown, Save } from 'lucide-react';

export default function PricingTestimonialsBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [testimonials, setTestimonials] = useState(content.testimonials || []);

    const handleSave = () => {
        onUpdate(block.id, { ...content, testimonials });
    };

    const addTestimonial = () => {
        setTestimonials([...testimonials, { 
            quote: '', 
            author: '',
            position: '',
            service: 'Essay Writing',
            rating: 5,
            link: '/services/essay-writing',
            linkText: 'Get Cheapest Essay Help'
        }]);
    };

    const removeTestimonial = (index) => {
        setTestimonials(testimonials.filter((_, i) => i !== index));
    };

    const updateTestimonial = (index, field, value) => {
        const updated = [...testimonials];
        updated[index] = { ...updated[index], [field]: value };
        setTestimonials(updated);
    };

    const moveTestimonialUp = (index) => {
        if (index === 0) return;
        const updated = [...testimonials];
        [updated[index - 1], updated[index]] = [updated[index], updated[index - 1]];
        setTestimonials(updated);
    };

    const moveTestimonialDown = (index) => {
        if (index === testimonials.length - 1) return;
        const updated = [...testimonials];
        [updated[index], updated[index + 1]] = [updated[index + 1], updated[index]];
        setTestimonials(updated);
    };

    return (
        <div className="space-y-6">
            <div>
                <Label htmlFor="title">Section Title</Label>
                <Input
                    id="title"
                    value={content.title || ''}
                    onChange={(e) => setContent({...content, title: e.target.value})}
                />
            </div>

            <div>
                <Label htmlFor="description">Section Description</Label>
                <RichTextEditor
                    value={content.description || ''}
                    onChange={(value) => setContent({...content, description: value})}
                    height={150}
                />
            </div>

            <div className="border-t pt-4">
                <div className="flex justify-between items-center mb-4">
                    <Label>Testimonials ({testimonials.length})</Label>
                    <Button
                        type="button"
                        onClick={addTestimonial}
                        className="bg-green-600 hover:bg-green-700 text-white"
                    >
                        <Plus className="h-4 w-4 mr-1" />
                        Add Testimonial
                    </Button>
                </div>

                <div className="space-y-6">
                    {testimonials.map((testimonial, index) => (
                        <div key={index} className="border border-gray-200 rounded-lg p-4 bg-gray-50">
                            <div className="flex justify-between items-center mb-3">
                                <span className="text-sm font-semibold text-gray-700">Testimonial #{index + 1}</span>
                                <div className="flex gap-2">
                                    <button
                                        type="button"
                                        onClick={() => moveTestimonialUp(index)}
                                        disabled={index === 0}
                                        className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                        title="Move up"
                                    >
                                        <ChevronUp className="h-4 w-4" />
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => moveTestimonialDown(index)}
                                        disabled={index === testimonials.length - 1}
                                        className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                        title="Move down"
                                    >
                                        <ChevronDown className="h-4 w-4" />
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => removeTestimonial(index)}
                                        className="p-1 text-red-500 hover:text-red-700"
                                        title="Remove testimonial"
                                    >
                                        <X className="h-4 w-4" />
                                    </button>
                                </div>
                            </div>

                            <div className="space-y-3">
                                <div>
                                    <Label htmlFor={`quote-${index}`}>Customer Quote</Label>
                                    <RichTextEditor
                                        value={testimonial.quote || ''}
                                        onChange={(value) => updateTestimonial(index, 'quote', value)}
                                        height={100}
                                    />
                                </div>

                                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                    <div>
                                        <Label htmlFor={`author-${index}`}>Author Name</Label>
                                        <Input
                                            id={`author-${index}`}
                                            value={testimonial.author || ''}
                                            onChange={(e) => updateTestimonial(index, 'author', e.target.value)}
                                            placeholder="e.g., Sarah M."
                                        />
                                    </div>
                                    <div>
                                        <Label htmlFor={`position-${index}`}>Position</Label>
                                        <Input
                                            id={`position-${index}`}
                                            value={testimonial.position || ''}
                                            onChange={(e) => updateTestimonial(index, 'position', e.target.value)}
                                            placeholder="e.g., University Student"
                                        />
                                    </div>
                                </div>

                                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                    <div>
                                        <Label htmlFor={`service-${index}`}>Service Type</Label>
                                        <Input
                                            id={`service-${index}`}
                                            value={testimonial.service || ''}
                                            onChange={(e) => updateTestimonial(index, 'service', e.target.value)}
                                            placeholder="e.g., Essay Writing"
                                        />
                                    </div>
                                    <div>
                                        <Label htmlFor={`rating-${index}`}>Rating (1-5)</Label>
                                        <Input
                                            id={`rating-${index}`}
                                            type="number"
                                            min="1"
                                            max="5"
                                            value={testimonial.rating || 5}
                                            onChange={(e) => updateTestimonial(index, 'rating', parseInt(e.target.value))}
                                        />
                                    </div>
                                </div>

                                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                    <div>
                                        <Label htmlFor={`link-${index}`}>Link URL</Label>
                                        <Input
                                            id={`link-${index}`}
                                            value={testimonial.link || ''}
                                            onChange={(e) => updateTestimonial(index, 'link', e.target.value)}
                                            placeholder="/services/essay-writing"
                                        />
                                    </div>
                                    <div>
                                        <Label htmlFor={`linkText-${index}`}>Link Text</Label>
                                        <Input
                                            id={`linkText-${index}`}
                                            value={testimonial.linkText || ''}
                                            onChange={(e) => updateTestimonial(index, 'linkText', e.target.value)}
                                            placeholder="Get Cheapest Essay Help"
                                        />
                                    </div>
                                </div>
                            </div>
                        </div>
                    ))}

                    {testimonials.length === 0 && (
                        <div className="text-center py-8 text-gray-500 border-2 border-dashed border-gray-300 rounded-lg">
                            No testimonials yet. Click "Add Testimonial" to get started.
                        </div>
                    )}
                </div>
            </div>

            {/* Bottom CTA Section */}
            <div className="border-t pt-4">
                <h3 className="text-lg font-semibold mb-4">Bottom CTA Section</h3>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <Label htmlFor="viewMoreReviewsText">CTA Button Text</Label>
                        <Input
                            id="viewMoreReviewsText"
                            value={content.viewMoreReviewsText || ''}
                            onChange={(e) => setContent({...content, viewMoreReviewsText: e.target.value})}
                            placeholder="Read More Reviews"
                        />
                    </div>
                    <div>
                        <Label htmlFor="viewMoreReviewsLink">CTA Button Link</Label>
                        <Input
                            id="viewMoreReviewsLink"
                            value={content.viewMoreReviewsLink || ''}
                            onChange={(e) => setContent({...content, viewMoreReviewsLink: e.target.value})}
                            placeholder="/reviews"
                        />
                    </div>
                </div>
            </div>
            
            <Button
                type="button"
                onClick={handleSave}
                className="bg-blue-600 hover:bg-blue-700 text-white"
            >
                <Save className="h-4 w-4 mr-2" />
                Save Changes
            </Button>
        </div>
    );
}



