import React, { useState } from 'react';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import { Button } from '@/Components/ui/Button';
import RichTextEditor from '@/Components/Admin/RichTextEditor';
import { Plus, X, ChevronUp, ChevronDown, Save } from 'lucide-react';

export default function PricingTiersBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [tiers, setTiers] = useState(content.tiers || []);

    const handleSave = () => {
        onUpdate(block.id, { ...content, tiers });
    };

    const addTier = () => {
        setTiers([...tiers, { 
            level: '', 
            startingPrice: '', 
            description: '',
            orderLink: '/place-order',
            serviceLink: '/services/essay-writing'
        }]);
    };

    const removeTier = (index) => {
        setTiers(tiers.filter((_, i) => i !== index));
    };

    const updateTier = (index, field, value) => {
        const updated = [...tiers];
        updated[index] = { ...updated[index], [field]: value };
        setTiers(updated);
    };

    const moveTierUp = (index) => {
        if (index === 0) return;
        const updated = [...tiers];
        [updated[index - 1], updated[index]] = [updated[index], updated[index - 1]];
        setTiers(updated);
    };

    const moveTierDown = (index) => {
        if (index === tiers.length - 1) return;
        const updated = [...tiers];
        [updated[index], updated[index + 1]] = [updated[index + 1], updated[index]];
        setTiers(updated);
    };

    return (
        <div className="space-y-6">
            <div>
                <Label htmlFor="title">Section Title</Label>
                <Input
                    id="title"
                    value={content.title || ''}
                    onChange={(e) => setContent({...content, title: e.target.value})}
                />
            </div>

            <div>
                <Label htmlFor="description">Section Description</Label>
                <RichTextEditor
                    value={content.description || ''}
                    onChange={(value) => setContent({...content, description: value})}
                    height={150}
                />
            </div>

            <div className="border-t pt-4">
                <div className="flex justify-between items-center mb-4">
                    <Label>Pricing Tiers ({tiers.length})</Label>
                    <Button
                        type="button"
                        onClick={addTier}
                        className="bg-green-600 hover:bg-green-700 text-white"
                    >
                        <Plus className="h-4 w-4 mr-1" />
                        Add Tier
                    </Button>
                </div>

                <div className="space-y-6">
                    {tiers.map((tier, index) => (
                        <div key={index} className="border border-gray-200 rounded-lg p-4 bg-gray-50">
                            <div className="flex justify-between items-center mb-3">
                                <span className="text-sm font-semibold text-gray-700">Tier #{index + 1}</span>
                                <div className="flex gap-2">
                                    <button
                                        type="button"
                                        onClick={() => moveTierUp(index)}
                                        disabled={index === 0}
                                        className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                        title="Move up"
                                    >
                                        <ChevronUp className="h-4 w-4" />
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => moveTierDown(index)}
                                        disabled={index === tiers.length - 1}
                                        className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                        title="Move down"
                                    >
                                        <ChevronDown className="h-4 w-4" />
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => removeTier(index)}
                                        className="p-1 text-red-500 hover:text-red-700"
                                        title="Remove tier"
                                    >
                                        <X className="h-4 w-4" />
                                    </button>
                                </div>
                            </div>

                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div>
                                    <Label htmlFor={`level-${index}`}>Level Name</Label>
                                    <Input
                                        id={`level-${index}`}
                                        value={tier.level || ''}
                                        onChange={(e) => updateTier(index, 'level', e.target.value)}
                                        placeholder="e.g., High School, University"
                                    />
                                </div>
                                <div>
                                    <Label htmlFor={`price-${index}`}>Starting Price</Label>
                                    <Input
                                        id={`price-${index}`}
                                        value={tier.startingPrice || ''}
                                        onChange={(e) => updateTier(index, 'startingPrice', e.target.value)}
                                        placeholder="e.g., 8.5"
                                    />
                                </div>
                            </div>

                            <div className="mt-4">
                                <Label htmlFor={`description-${index}`}>Description</Label>
                                <RichTextEditor
                                    value={tier.description || ''}
                                    onChange={(value) => updateTier(index, 'description', value)}
                                    height={100}
                                />
                            </div>

                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
                                <div>
                                    <Label htmlFor={`orderLink-${index}`}>Order Link</Label>
                                    <Input
                                        id={`orderLink-${index}`}
                                        value={tier.orderLink || ''}
                                        onChange={(e) => updateTier(index, 'orderLink', e.target.value)}
                                        placeholder="/place-order"
                                    />
                                </div>
                                <div>
                                    <Label htmlFor={`serviceLink-${index}`}>Service Link</Label>
                                    <Input
                                        id={`serviceLink-${index}`}
                                        value={tier.serviceLink || ''}
                                        onChange={(e) => updateTier(index, 'serviceLink', e.target.value)}
                                        placeholder="/services/essay-writing"
                                    />
                                </div>
                            </div>
                        </div>
                    ))}

                    {tiers.length === 0 && (
                        <div className="text-center py-8 text-gray-500 border-2 border-dashed border-gray-300 rounded-lg">
                            No pricing tiers yet. Click "Add Tier" to get started.
                        </div>
                    )}
                </div>
            </div>

            {/* Price Comparison Section */}
            <div className="border-t pt-4">
                <h3 className="text-lg font-semibold mb-4">Price Comparison Section</h3>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <Label htmlFor="comparisonTitle">Comparison Title</Label>
                        <Input
                            id="comparisonTitle"
                            value={content.priceComparison?.title || ''}
                            onChange={(e) => setContent({
                                ...content, 
                                priceComparison: {...content.priceComparison, title: e.target.value}
                            })}
                        />
                    </div>
                    <div>
                        <Label htmlFor="competitorPrice">Competitor Price</Label>
                        <Input
                            id="competitorPrice"
                            value={content.priceComparison?.competitorPrice || ''}
                            onChange={(e) => setContent({
                                ...content, 
                                priceComparison: {...content.priceComparison, competitorPrice: e.target.value}
                            })}
                            placeholder="$15-20"
                        />
                    </div>
                </div>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
                    <div>
                        <Label htmlFor="ourPrice">Our Price</Label>
                        <Input
                            id="ourPrice"
                            value={content.priceComparison?.ourPrice || ''}
                            onChange={(e) => setContent({
                                ...content, 
                                priceComparison: {...content.priceComparison, ourPrice: e.target.value}
                            })}
                            placeholder="$8.50-18"
                        />
                    </div>
                    <div>
                        <Label htmlFor="savings">Savings</Label>
                        <Input
                            id="savings"
                            value={content.priceComparison?.savings || ''}
                            onChange={(e) => setContent({
                                ...content, 
                                priceComparison: {...content.priceComparison, savings: e.target.value}
                            })}
                            placeholder="Save 30-50%"
                        />
                    </div>
                </div>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
                    <div>
                        <Label htmlFor="ctaText">CTA Text</Label>
                        <Input
                            id="ctaText"
                            value={content.priceComparison?.ctaText || ''}
                            onChange={(e) => setContent({
                                ...content, 
                                priceComparison: {...content.priceComparison, ctaText: e.target.value}
                            })}
                            placeholder="Order Cheapest Essays Now - Save 30%+"
                        />
                    </div>
                    <div>
                        <Label htmlFor="ctaLink">CTA Link</Label>
                        <Input
                            id="ctaLink"
                            value={content.priceComparison?.ctaLink || ''}
                            onChange={(e) => setContent({
                                ...content, 
                                priceComparison: {...content.priceComparison, ctaLink: e.target.value}
                            })}
                            placeholder="/place-order"
                        />
                    </div>
                </div>
            </div>
            
            <Button
                type="button"
                onClick={handleSave}
                className="bg-blue-600 hover:bg-blue-700 text-white"
            >
                <Save className="h-4 w-4 mr-2" />
                Save Changes
            </Button>
        </div>
    );
}



