import React, { useState } from 'react';
import { Card, CardContent } from '@/Components/ui/Card';
import { Button } from '@/Components/ui/Button';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import RichTextEditor from '@/Components/Admin/RichTextEditor';
import { Trash2, Plus, GripVertical } from 'lucide-react';

export default function QualityStandardsBlockEditor({ block, onUpdate }) {
    const [data, setData] = useState({
        title: block.content?.title || 'Our Quality Standards',
        description: block.content?.description || '',
        standards: block.content?.standards || []
    });

    const updateData = (field, value) => {
        const newData = { ...data, [field]: value };
        setData(newData);
    };

    const updateStandard = (index, field, value) => {
        const newStandards = [...data.standards];
        newStandards[index] = { ...newStandards[index], [field]: value };
        setData({ ...data, standards: newStandards });
    };

    const updateStandardFeature = (standardIndex, featureIndex, value) => {
        const newStandards = [...data.standards];
        const newFeatures = [...(newStandards[standardIndex].features || [])];
        newFeatures[featureIndex] = value;
        newStandards[standardIndex] = { 
            ...newStandards[standardIndex], 
            features: newFeatures 
        };
        setData({ ...data, standards: newStandards });
    };

    const addStandard = () => {
        setData({
            ...data,
            standards: [
                ...data.standards,
                {
                    icon: '⭐',
                    title: 'New Quality Standard',
                    description: 'Description here...',
                    features: [],
                    link: '',
                    linkText: ''
                }
            ]
        });
    };

    const removeStandard = (index) => {
        const newStandards = data.standards.filter((_, i) => i !== index);
        setData({ ...data, standards: newStandards });
    };

    const addFeature = (standardIndex) => {
        const newStandards = [...data.standards];
        newStandards[standardIndex].features = [
            ...(newStandards[standardIndex].features || []),
            'New feature'
        ];
        setData({ ...data, standards: newStandards });
    };

    const removeFeature = (standardIndex, featureIndex) => {
        const newStandards = [...data.standards];
        newStandards[standardIndex].features = newStandards[standardIndex].features.filter(
            (_, i) => i !== featureIndex
        );
        setData({ ...data, standards: newStandards });
    };

    const handleSave = () => {
        onUpdate(block.id, data);
    };

    return (
        <div className="space-y-6">
            {/* Section Title and Description */}
            <Card>
                <CardContent className="pt-6">
                    <div className="space-y-4">
                        <div>
                            <Label htmlFor="section-title">Section Title</Label>
                            <Input
                                id="section-title"
                                value={data.title}
                                onChange={(e) => updateData('title', e.target.value)}
                                placeholder="Our Quality Standards"
                            />
                        </div>
                        <div>
                            <Label htmlFor="section-description">Section Description (Optional)</Label>
                            <RichTextEditor
                                value={data.description}
                                onChange={(value) => updateData('description', value)}
                                height={200}
                                placeholder="Optional description for the quality standards section..."
                            />
                        </div>
                    </div>
                </CardContent>
            </Card>

            {/* Quality Standards */}
            <div className="space-y-4">
                <div className="flex items-center justify-between">
                    <h3 className="text-lg font-semibold">Quality Standards ({data.standards.length})</h3>
                    <Button onClick={addStandard} size="sm">
                        <Plus className="h-4 w-4 mr-2" />
                        Add Standard
                    </Button>
                </div>

                {data.standards.map((standard, sIndex) => (
                    <Card key={sIndex} className="border-l-4 border-l-green-500">
                        <CardContent className="pt-6">
                            <div className="space-y-4">
                                {/* Header */}
                                <div className="flex items-center justify-between">
                                    <div className="flex items-center space-x-2">
                                        <GripVertical className="h-5 w-5 text-gray-400" />
                                        <span className="font-semibold">Standard #{sIndex + 1}</span>
                                    </div>
                                    <Button
                                        variant="destructive"
                                        size="sm"
                                        onClick={() => removeStandard(sIndex)}
                                    >
                                        <Trash2 className="h-4 w-4" />
                                    </Button>
                                </div>

                                {/* Icon */}
                                <div>
                                    <Label>Icon (emoji, optional)</Label>
                                    <Input
                                        value={standard.icon || ''}
                                        onChange={(e) => updateStandard(sIndex, 'icon', e.target.value)}
                                        placeholder="⭐"
                                        className="text-2xl"
                                    />
                                </div>

                                {/* Title */}
                                <div>
                                    <Label>Standard Title</Label>
                                    <Input
                                        value={standard.title}
                                        onChange={(e) => updateStandard(sIndex, 'title', e.target.value)}
                                        placeholder="Academic Writing Standards"
                                    />
                                </div>

                                {/* Description with rich text editor */}
                                <div>
                                    <Label>Description</Label>
                                    <RichTextEditor
                                        value={standard.description}
                                        onChange={(value) => updateStandard(sIndex, 'description', value)}
                                        height={250}
                                        placeholder="Describe this quality standard. You can add links to related pages..."
                                    />
                                    <p className="text-sm text-gray-500 mt-1">
                                        Tip: Add internal links to relevant service pages or resources
                                    </p>
                                </div>

                                {/* Features (Optional) */}
                                <div>
                                    <div className="flex items-center justify-between mb-2">
                                        <Label>Features (optional bullet points)</Label>
                                        <Button
                                            variant="outline"
                                            size="sm"
                                            onClick={() => addFeature(sIndex)}
                                        >
                                            <Plus className="h-3 w-3 mr-1" />
                                            Add Feature
                                        </Button>
                                    </div>
                                    {standard.features && standard.features.length > 0 && (
                                        <div className="space-y-2">
                                            {standard.features.map((feature, fIndex) => (
                                                <div key={fIndex} className="flex items-center space-x-2">
                                                    <span className="text-green-500">✓</span>
                                                    <Input
                                                        value={feature}
                                                        onChange={(e) => updateStandardFeature(sIndex, fIndex, e.target.value)}
                                                        placeholder="Feature description"
                                                        className="flex-1"
                                                    />
                                                    <Button
                                                        variant="ghost"
                                                        size="sm"
                                                        onClick={() => removeFeature(sIndex, fIndex)}
                                                    >
                                                        <Trash2 className="h-4 w-4 text-red-500" />
                                                    </Button>
                                                </div>
                                            ))}
                                        </div>
                                    )}
                                </div>

                                {/* Link */}
                                <div className="grid grid-cols-2 gap-4">
                                    <div>
                                        <Label>Link URL (optional)</Label>
                                        <Input
                                            value={standard.link}
                                            onChange={(e) => updateStandard(sIndex, 'link', e.target.value)}
                                            placeholder="/services/literature-reviews"
                                        />
                                    </div>
                                    <div>
                                        <Label>Link Text</Label>
                                        <Input
                                            value={standard.linkText}
                                            onChange={(e) => updateStandard(sIndex, 'linkText', e.target.value)}
                                            placeholder="Learn More →"
                                        />
                                    </div>
                                </div>
                            </div>
                        </CardContent>
                    </Card>
                ))}

                {data.standards.length === 0 && (
                    <div className="text-center py-8 border-2 border-dashed border-gray-300 rounded-lg">
                        <p className="text-gray-500 mb-4">No quality standards added yet</p>
                        <Button onClick={addStandard}>
                            <Plus className="h-4 w-4 mr-2" />
                            Add Your First Quality Standard
                        </Button>
                    </div>
                )}
            </div>

            {/* Save Button */}
            <div className="flex justify-end pt-4 border-t">
                <Button onClick={handleSave} size="lg">
                    Save Changes
                </Button>
            </div>
        </div>
    );
}





