import React, { useState } from 'react';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import { Button } from '@/Components/ui/Button';
import RichTextEditor from '@/Components/Admin/RichTextEditor';
import { Plus, X, ChevronUp, ChevronDown, Save } from 'lucide-react';

export default function ServicePricingGridBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [services, setServices] = useState(content.services || []);

    const handleSave = () => {
        onUpdate(block.id, { ...content, services });
    };

    const addService = () => {
        setServices([...services, { 
            name: '', 
            price: '', 
            description: '',
            learnMoreLink: '/services/essay-writing',
            orderLink: '/place-order'
        }]);
    };

    const removeService = (index) => {
        setServices(services.filter((_, i) => i !== index));
    };

    const updateService = (index, field, value) => {
        const updated = [...services];
        updated[index] = { ...updated[index], [field]: value };
        setServices(updated);
    };

    const moveServiceUp = (index) => {
        if (index === 0) return;
        const updated = [...services];
        [updated[index - 1], updated[index]] = [updated[index], updated[index - 1]];
        setServices(updated);
    };

    const moveServiceDown = (index) => {
        if (index === services.length - 1) return;
        const updated = [...services];
        [updated[index], updated[index + 1]] = [updated[index + 1], updated[index]];
        setServices(updated);
    };

    return (
        <div className="space-y-6">
            <div>
                <Label htmlFor="title">Section Title</Label>
                <Input
                    id="title"
                    value={content.title || ''}
                    onChange={(e) => setContent({...content, title: e.target.value})}
                />
            </div>

            <div>
                <Label htmlFor="description">Section Description</Label>
                <RichTextEditor
                    value={content.description || ''}
                    onChange={(value) => setContent({...content, description: value})}
                    height={150}
                />
            </div>

            <div className="border-t pt-4">
                <div className="flex justify-between items-center mb-4">
                    <Label>Services ({services.length})</Label>
                    <Button
                        type="button"
                        onClick={addService}
                        className="bg-green-600 hover:bg-green-700 text-white"
                    >
                        <Plus className="h-4 w-4 mr-1" />
                        Add Service
                    </Button>
                </div>

                <div className="space-y-6">
                    {services.map((service, index) => (
                        <div key={index} className="border border-gray-200 rounded-lg p-4 bg-gray-50">
                            <div className="flex justify-between items-center mb-3">
                                <span className="text-sm font-semibold text-gray-700">Service #{index + 1}</span>
                                <div className="flex gap-2">
                                    <button
                                        type="button"
                                        onClick={() => moveServiceUp(index)}
                                        disabled={index === 0}
                                        className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                        title="Move up"
                                    >
                                        <ChevronUp className="h-4 w-4" />
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => moveServiceDown(index)}
                                        disabled={index === services.length - 1}
                                        className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                        title="Move down"
                                    >
                                        <ChevronDown className="h-4 w-4" />
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => removeService(index)}
                                        className="p-1 text-red-500 hover:text-red-700"
                                        title="Remove service"
                                    >
                                        <X className="h-4 w-4" />
                                    </button>
                                </div>
                            </div>

                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div>
                                    <Label htmlFor={`name-${index}`}>Service Name</Label>
                                    <Input
                                        id={`name-${index}`}
                                        value={service.name || ''}
                                        onChange={(e) => updateService(index, 'name', e.target.value)}
                                        placeholder="e.g., Essay Writing"
                                    />
                                </div>
                                <div>
                                    <Label htmlFor={`price-${index}`}>Price Range</Label>
                                    <Input
                                        id={`price-${index}`}
                                        value={service.price || ''}
                                        onChange={(e) => updateService(index, 'price', e.target.value)}
                                        placeholder="e.g., $10-18"
                                    />
                                </div>
                            </div>

                            <div className="mt-4">
                                <Label htmlFor={`description-${index}`}>Description</Label>
                                <RichTextEditor
                                    value={service.description || ''}
                                    onChange={(value) => updateService(index, 'description', value)}
                                    height={100}
                                />
                            </div>

                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
                                <div>
                                    <Label htmlFor={`learnMoreLink-${index}`}>Learn More Link</Label>
                                    <Input
                                        id={`learnMoreLink-${index}`}
                                        value={service.learnMoreLink || ''}
                                        onChange={(e) => updateService(index, 'learnMoreLink', e.target.value)}
                                        placeholder="/services/essay-writing"
                                    />
                                </div>
                                <div>
                                    <Label htmlFor={`orderLink-${index}`}>Order Link</Label>
                                    <Input
                                        id={`orderLink-${index}`}
                                        value={service.orderLink || ''}
                                        onChange={(e) => updateService(index, 'orderLink', e.target.value)}
                                        placeholder="/place-order"
                                    />
                                </div>
                            </div>
                        </div>
                    ))}

                    {services.length === 0 && (
                        <div className="text-center py-8 text-gray-500 border-2 border-dashed border-gray-300 rounded-lg">
                            No services yet. Click "Add Service" to get started.
                        </div>
                    )}
                </div>
            </div>

            {/* Bottom CTA Section */}
            <div className="border-t pt-4">
                <h3 className="text-lg font-semibold mb-4">Bottom CTA Section</h3>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <Label htmlFor="viewAllServicesText">CTA Button Text</Label>
                        <Input
                            id="viewAllServicesText"
                            value={content.viewAllServicesText || ''}
                            onChange={(e) => setContent({...content, viewAllServicesText: e.target.value})}
                            placeholder="View All Essay Writing Services"
                        />
                    </div>
                    <div>
                        <Label htmlFor="viewAllServicesLink">CTA Button Link</Label>
                        <Input
                            id="viewAllServicesLink"
                            value={content.viewAllServicesLink || ''}
                            onChange={(e) => setContent({...content, viewAllServicesLink: e.target.value})}
                            placeholder="/services"
                        />
                    </div>
                </div>
            </div>
            
            <Button
                type="button"
                onClick={handleSave}
                className="bg-blue-600 hover:bg-blue-700 text-white"
            >
                <Save className="h-4 w-4 mr-2" />
                Save Changes
            </Button>
        </div>
    );
}



