import React, { useState, useCallback } from 'react';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import {
    faFileWord,
    faFileExcel,
    faFilePowerpoint,
    faFilePdf,
    faFileArchive,
    faUpload,
    faXmark,
    faCircleExclamation,
    faUser,
    faPenNib,
    faDownload,
    faTrash
} from '@fortawesome/free-solid-svg-icons';
import { useDropzone } from 'react-dropzone';
import axios from 'axios';
import clsx from 'clsx';

// File type configurations with icons
const ALLOWED_TYPES = {
    // Microsoft Word
    "application/msword": { ext: ".doc", icon: faFileWord },
    "application/vnd.openxmlformats-officedocument.wordprocessingml.document": {
        ext: ".docx",
        icon: faFileWord,
    },

    // Microsoft Excel
    "application/vnd.ms-excel": { ext: ".xls", icon: faFileExcel },
    "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet": {
        ext: ".xlsx",
        icon: faFileExcel,
    },

    // Microsoft PowerPoint
    "application/vnd.ms-powerpoint": { ext: ".ppt", icon: faFilePowerpoint },
    "application/vnd.openxmlformats-officedocument.presentationml.presentation": {
        ext: ".pptx",
        icon: faFilePowerpoint,
    },

    // PDF
    "application/pdf": { ext: ".pdf", icon: faFilePdf },

    // ZIP
    "application/zip": { ext: ".zip", icon: faFileArchive },
    "application/x-zip-compressed": { ext: ".zip", icon: faFileArchive },
};

// Maximum total file size allowed (40MB)
const MAX_FILE_SIZE = 40 * 1024 * 1024;

export default function AdminFileUpload({ orderId, onFileUploaded }) {
    const [selectedFiles, setSelectedFiles] = useState([]);
    const [uploadProgress, setUploadProgress] = useState({});
    const [error, setError] = useState("");
    const [success, setSuccess] = useState("");
    const [isUploading, setIsUploading] = useState(false);
    const [uploadForm, setUploadForm] = useState({
        target_type: 'client',
        file_category: 'draft',
        description: ''
    });

    const formatFileSize = (bytes) => {
        if (bytes === 0) return "0 Bytes";
        const k = 1024;
        const sizes = ["Bytes", "KB", "MB", "GB"];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return `${parseFloat((bytes / Math.pow(k, i)).toFixed(2))} ${sizes[i]}`;
    };

    const onDrop = useCallback((acceptedFiles) => {
        setError("");
        setSuccess("");
        const validFiles = acceptedFiles.filter((file) => {
            // Validate file type
            if (!Object.keys(ALLOWED_TYPES).includes(file.type)) {
                setError(`${file.name} is not an allowed file type`);
                return false;
            }
            // Validate total file size against MAX_FILE_SIZE
            if (file.size > MAX_FILE_SIZE) {
                setError(`${file.name} exceeds 40MB limit`);
                return false;
            }
            return true;
        });
        setSelectedFiles((prev) => [...prev, ...validFiles]);
    }, []);

    const { getRootProps, getInputProps, isDragActive } = useDropzone({
        onDrop,
        accept: Object.keys(ALLOWED_TYPES).reduce((acc, curr) => {
            acc[curr] = [ALLOWED_TYPES[curr].ext];
            return acc;
        }, {}),
        maxSize: MAX_FILE_SIZE,
    });

    const removeFile = (index) => {
        setSelectedFiles(selectedFiles.filter((_, i) => i !== index));
    };

    const handleUpload = async () => {
        if (selectedFiles.length === 0) {
            setError("Please select at least one file to upload");
            return;
        }

        if (!uploadForm.target_type || !uploadForm.file_category) {
            setError("Please select target type and file category");
            return;
        }

        setIsUploading(true);
        setError("");
        setSuccess("");

        try {
            const uploadPromises = selectedFiles.map(async (file, index) => {
                const formData = new FormData();
                formData.append('file', file);
                formData.append('order_id', orderId);
                formData.append('target_type', uploadForm.target_type);
                formData.append('file_category', uploadForm.file_category);
                formData.append('description', uploadForm.description);
                formData.append('uploader_user_type', 'admin');

                // Update progress
                setUploadProgress(prev => ({ ...prev, [index]: 0 }));

                const response = await axios.post('/order-files/admin-upload', formData, {
                    headers: {
                        'Content-Type': 'multipart/form-data',
                    },
                    onUploadProgress: (progressEvent) => {
                        const progress = Math.round((progressEvent.loaded * 100) / progressEvent.total);
                        setUploadProgress(prev => ({ ...prev, [index]: progress }));
                    },
                });

                if (response.data.success) {
                    setUploadProgress(prev => ({ ...prev, [index]: 100 }));
                    return response.data.file;
                } else {
                    throw new Error(response.data.message || 'Upload failed');
                }
            });

            const uploadedFiles = await Promise.all(uploadPromises);
            
            setSuccess(`${uploadedFiles.length} file(s) uploaded successfully!`);
            setSelectedFiles([]);
            setUploadForm({
                target_type: 'client',
                file_category: 'draft',
                description: ''
            });
            
            // Notify parent component
            if (onFileUploaded) {
                onFileUploaded(uploadedFiles);
            }

            // Clear success message after 5 seconds
            setTimeout(() => setSuccess(""), 5000);

        } catch (error) {
            console.error('Upload error:', error);
            setError(error.response?.data?.message || error.message || 'Upload failed');
        } finally {
            setIsUploading(false);
            setUploadProgress({});
        }
    };

    const getFileIcon = (fileType) => {
        const fileConfig = ALLOWED_TYPES[fileType];
        return fileConfig ? fileConfig.icon : faFilePdf;
    };

    const getTargetTypeLabel = (type) => {
        return type === 'client' ? 'Client' : 'Writer';
    };

    const getCategoryLabel = (category) => {
        const labels = {
            'draft': 'Draft',
            'complete': 'Complete',
            'admin': 'Admin'
        };
        return labels[category] || category;
    };

    return (
        <div className="space-y-6">
            {/* Upload Configuration */}
            <div className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-6">
                <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">
                    Admin File Upload Configuration
                </h3>
                
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                    {/* Target Type Selection */}
                    <div>
                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Target Type
                        </label>
                        <select
                            value={uploadForm.target_type}
                            onChange={(e) => setUploadForm({...uploadForm, target_type: e.target.value})}
                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                        >
                            <option value="client">Client</option>
                            <option value="writer">Writer</option>
                        </select>
                        <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                            Choose who this file is intended for
                        </p>
                    </div>

                    {/* File Category Selection */}
                    <div>
                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            File Category
                        </label>
                        <select
                            value={uploadForm.file_category}
                            onChange={(e) => setUploadForm({...uploadForm, file_category: e.target.value})}
                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                        >
                            <option value="draft">Draft</option>
                            <option value="complete">Complete</option>
                            <option value="admin">Admin</option>
                        </select>
                        <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                            {uploadForm.file_category === 'complete' && uploadForm.target_type === 'client' && 
                                'Complete files for clients will update order status and notify the client'
                            }
                        </p>
                    </div>

                    {/* Description */}
                    <div>
                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Description (Optional)
                        </label>
                        <input
                            type="text"
                            value={uploadForm.description}
                            onChange={(e) => setUploadForm({...uploadForm, description: e.target.value})}
                            placeholder="Brief description of the file"
                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                        />
                    </div>
                </div>

                {/* Current Configuration Display */}
                <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
                    <div className="flex items-center space-x-2 text-sm text-blue-800 dark:text-blue-200">
                        <FontAwesomeIcon icon={uploadForm.target_type === 'client' ? faUser : faPenNib} className="w-4 h-4" />
                        <span>
                            <strong>Target:</strong> {getTargetTypeLabel(uploadForm.target_type)} | 
                            <strong>Category:</strong> {getCategoryLabel(uploadForm.file_category)}
                        </span>
                    </div>
                    {uploadForm.description && (
                        <div className="mt-2 text-sm text-blue-700 dark:text-blue-300">
                            <strong>Description:</strong> {uploadForm.description}
                        </div>
                    )}
                </div>
            </div>

            {/* File Upload Area */}
            <div className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-6">
                <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">
                    Select Files to Upload
                </h3>

                {/* Dropzone */}
                <div
                    {...getRootProps()}
                    className={clsx(
                        "border-2 border-dashed rounded-lg p-8 text-center cursor-pointer transition-colors",
                        isDragActive
                            ? "border-blue-500 bg-blue-50 dark:bg-blue-900/20"
                            : "border-gray-300 dark:border-gray-600 hover:border-gray-400 dark:hover:border-gray-500"
                    )}
                >
                    <input {...getInputProps()} />
                    <FontAwesomeIcon
                        icon={faUpload}
                        className="w-8 h-8 text-gray-400 dark:text-gray-500 mx-auto mb-4"
                    />
                    <p className="text-gray-600 dark:text-gray-400 mb-2">
                        {isDragActive
                            ? "Drop the files here..."
                            : "Drag & drop files here, or click to select files"
                        }
                    </p>
                    <p className="text-sm text-gray-500 dark:text-gray-400">
                        Supported formats: DOC, DOCX, XLS, XLSX, PPT, PPTX, PDF, ZIP (Max: 40MB)
                    </p>
                </div>

                {/* Error and Success Messages */}
                {error && (
                    <div className="mt-4 p-4 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg">
                        <div className="flex items-center space-x-2 text-red-800 dark:text-red-200">
                            <FontAwesomeIcon icon={faCircleExclamation} className="w-4 h-4" />
                            <span>{error}</span>
                        </div>
                    </div>
                )}

                {success && (
                    <div className="mt-4 p-4 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg">
                        <div className="flex items-center space-x-2 text-green-800 dark:text-green-200">
                            <FontAwesomeIcon icon={faDownload} className="w-4 h-4" />
                            <span>{success}</span>
                        </div>
                    </div>
                )}

                {/* Selected Files List */}
                {selectedFiles.length > 0 && (
                    <div className="mt-6">
                        <h4 className="text-sm font-medium text-gray-900 dark:text-white mb-3">
                            Selected Files ({selectedFiles.length})
                        </h4>
                        <div className="space-y-2">
                            {selectedFiles.map((file, index) => (
                                <div
                                    key={index}
                                    className="flex items-center justify-between p-3 bg-gray-50 dark:bg-gray-700 rounded-lg"
                                >
                                    <div className="flex items-center space-x-3">
                                        <FontAwesomeIcon
                                            icon={getFileIcon(file.type)}
                                            className="w-5 h-5 text-gray-500 dark:text-gray-400"
                                        />
                                        <div>
                                            <p className="text-sm font-medium text-gray-900 dark:text-white">
                                                {file.name}
                                            </p>
                                            <p className="text-xs text-gray-500 dark:text-gray-400">
                                                {formatFileSize(file.size)}
                                            </p>
                                        </div>
                                    </div>

                                    <div className="flex items-center space-x-2">
                                        {/* Upload Progress */}
                                        {uploadProgress[index] !== undefined && (
                                            <div className="w-16 bg-gray-200 dark:bg-gray-600 rounded-full h-2">
                                                <div
                                                    className="bg-blue-600 h-2 rounded-full transition-all duration-300"
                                                    style={{ width: `${uploadProgress[index]}%` }}
                                                ></div>
                                            </div>
                                        )}
                                        
                                        {/* Remove Button */}
                                        <button
                                            onClick={() => removeFile(index)}
                                            className="p-1 text-red-600 hover:text-red-800 dark:text-red-400 dark:hover:text-red-300 transition-colors"
                                            disabled={isUploading}
                                        >
                                            <FontAwesomeIcon icon={faXmark} className="w-4 h-4" />
                                        </button>
                                    </div>
                                </div>
                            ))}
                        </div>
                    </div>
                )}

                {/* Upload Button */}
                {selectedFiles.length > 0 && (
                    <div className="mt-6">
                        <button
                            onClick={handleUpload}
                            disabled={isUploading}
                            className="w-full px-4 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                        >
                            {isUploading ? (
                                <div className="flex items-center justify-center space-x-2">
                                    <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white"></div>
                                    <span>Uploading...</span>
                                </div>
                            ) : (
                                <div className="flex items-center justify-center space-x-2">
                                    <FontAwesomeIcon icon={faUpload} className="w-4 h-4" />
                                    <span>
                                        Upload {selectedFiles.length} File{selectedFiles.length !== 1 ? 's' : ''} 
                                        to {getTargetTypeLabel(uploadForm.target_type)}
                                    </span>
                                </div>
                            )}
                        </button>
                    </div>
                )}
            </div>
        </div>
    );
} 