import React, { useState, useCallback, useRef } from "react";
import { router } from "@inertiajs/react";
import { motion, AnimatePresence } from "framer-motion";
import {
    Upload,
    FileText,
    AlertCircle,
    CheckCircle,
    X,
    Loader2,
    RefreshCw,
    Eye,
    Trash2,
    Plus,
} from "lucide-react";

const BlogFileUpload = ({
    onUploadComplete,
    onUploadError,
    maxFiles = 10,
    maxFileSize = 2 * 1024 * 1024, // 2MB
    className = "",
}) => {
    // State management
    const [files, setFiles] = useState([]);
    const [uploading, setUploading] = useState(false);
    const [uploadProgress, setUploadProgress] = useState({});
    const [uploadId, setUploadId] = useState(null);
    const [errors, setErrors] = useState({});
    const [dragActive, setDragActive] = useState(false);

    // Refs
    const fileInputRef = useRef(null);
    const uploadRef = useRef(null);

    // Allowed file types
    const allowedTypes = ["application/javascript", "text/javascript"];
    const allowedExtensions = [".js"];

    // Handle file selection
    const handleFileSelect = useCallback(
        (selectedFiles) => {
            const fileArray = Array.from(selectedFiles);
            const validFiles = [];
            const newErrors = {};

            fileArray.forEach((file, index) => {
                // Validate file type
                if (
                    !allowedTypes.includes(file.type) &&
                    !allowedExtensions.some((ext) =>
                        file.name.toLowerCase().endsWith(ext)
                    )
                ) {
                    newErrors[`file-${index}`] =
                        "Only JavaScript (.js) files are allowed";
                    return;
                }

                // Validate file size
                if (file.size > maxFileSize) {
                    newErrors[
                        `file-${index}`
                    ] = `File size must be less than ${formatFileSize(
                        maxFileSize
                    )}`;
                    return;
                }

                // Validate file name
                if (!/^[a-zA-Z0-9._-]+\.js$/.test(file.name)) {
                    newErrors[`file-${index}`] =
                        "Invalid file name. Only letters, numbers, dots, underscores, and hyphens are allowed";
                    return;
                }

                validFiles.push({
                    id: `file-${Date.now()}-${index}`,
                    file,
                    name: file.name,
                    size: file.size,
                    status: "pending",
                    progress: 0,
                });
            });

            // Check total file count
            if (files.length + validFiles.length > maxFiles) {
                newErrors.general = `Maximum ${maxFiles} files allowed`;
                setErrors(newErrors);
                return;
            }

            setFiles((prev) => [...prev, ...validFiles]);
            setErrors(newErrors);
        },
        [files, maxFiles, maxFileSize]
    );

    // Handle drag events
    const handleDrag = useCallback((e) => {
        e.preventDefault();
        e.stopPropagation();
        if (e.type === "dragenter" || e.type === "dragover") {
            setDragActive(true);
        } else if (e.type === "dragleave") {
            setDragActive(false);
        }
    }, []);

    // Handle drop
    const handleDrop = useCallback(
        (e) => {
            e.preventDefault();
            e.stopPropagation();
            setDragActive(false);

            if (e.dataTransfer.files && e.dataTransfer.files.length > 0) {
                handleFileSelect(e.dataTransfer.files);
            }
        },
        [handleFileSelect]
    );

    // Handle file input change
    const handleFileInputChange = useCallback(
        (e) => {
            if (e.target.files && e.target.files.length > 0) {
                handleFileSelect(e.target.files);
            }
        },
        [handleFileSelect]
    );

    // Remove file
    const removeFile = useCallback((fileId) => {
        setFiles((prev) => prev.filter((f) => f.id !== fileId));
        setErrors((prev) => {
            const newErrors = { ...prev };
            delete newErrors[fileId];
            return newErrors;
        });
    }, []);

    // Upload files
    const uploadFiles = useCallback(async () => {
        if (files.length === 0) return;

        setUploading(true);
        setErrors({});

        const formData = new FormData();
        files.forEach((fileObj) => {
            formData.append("blog_files[]", fileObj.file);
        });

        try {
            const response = await fetch("/blog/upload", {
                method: "POST",
                body: formData,
                headers: {
                    "X-CSRF-TOKEN": document
                        .querySelector('meta[name="csrf-token"]')
                        .getAttribute("content"),
                },
            });

            const result = await response.json();

            if (result.success) {
                setUploadId(result.upload_id);

                // Update file statuses
                setFiles((prev) =>
                    prev.map((f) => ({
                        ...f,
                        status: "uploaded",
                        progress: 100,
                    }))
                );

                if (onUploadComplete) {
                    onUploadComplete(result);
                }
            } else {
                setErrors({ general: result.error || "Upload failed" });
                if (onUploadError) {
                    onUploadError(result);
                }
            }
        } catch (error) {
            console.error("Upload error:", error);
            setErrors({ general: "Upload failed: " + error.message });
            if (onUploadError) {
                onUploadError(error);
            }
        } finally {
            setUploading(false);
        }
    }, [files, onUploadComplete, onUploadError]);

    // Clear all files
    const clearAllFiles = useCallback(() => {
        setFiles([]);
        setErrors({});
        setUploadProgress({});
        setUploadId(null);
        if (fileInputRef.current) {
            fileInputRef.current.value = "";
        }
    }, []);

    // Preview file content
    const previewFile = useCallback((fileObj) => {
        const reader = new FileReader();
        reader.onload = (e) => {
            const content = e.target.result;
            // You can implement a modal or preview component here
            console.log("File content:", content);
        };
        reader.readAsText(fileObj.file);
    }, []);

    // Format file size
    const formatFileSize = (bytes) => {
        if (bytes === 0) return "0 Bytes";
        const k = 1024;
        const sizes = ["Bytes", "KB", "MB", "GB"];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + " " + sizes[i];
    };

    // Animation variants
    const fadeInUp = {
        initial: { opacity: 0, y: 20 },
        animate: { opacity: 1, y: 0 },
        exit: { opacity: 0, y: -20 },
    };

    return (
        <div className={`blog-file-upload ${className}`}>
            {/* Upload Area */}
            <div
                className={`relative border-2 border-dashed rounded-lg p-8 text-center transition-all duration-300 ${
                    dragActive
                        ? "border-blue-500 bg-blue-50 dark:bg-blue-900/20"
                        : "border-gray-300 dark:border-gray-600 hover:border-gray-400 dark:hover:border-gray-500"
                }`}
                onDragEnter={handleDrag}
                onDragLeave={handleDrag}
                onDragOver={handleDrag}
                onDrop={handleDrop}
            >
                <input
                    ref={fileInputRef}
                    type="file"
                    multiple
                    accept=".js,application/javascript,text/javascript"
                    onChange={handleFileInputChange}
                    className="absolute inset-0 w-full h-full opacity-0 cursor-pointer"
                    disabled={uploading}
                />

                <div className="flex flex-col items-center justify-center space-y-4">
                    <div className="w-16 h-16 bg-gray-100 dark:bg-gray-700 rounded-full flex items-center justify-center">
                        <Upload className="w-8 h-8 text-gray-600 dark:text-gray-400" />
                    </div>

                    <div>
                        <p className="text-lg font-medium text-gray-900 dark:text-white">
                            Drop your JavaScript files here
                        </p>
                        <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">
                            or click to browse files
                        </p>
                    </div>

                    <div className="flex items-center space-x-4 text-xs text-gray-500 dark:text-gray-400">
                        <span>Max {maxFiles} files</span>
                        <span>•</span>
                        <span>Max {formatFileSize(maxFileSize)} per file</span>
                        <span>•</span>
                        <span>JS files only</span>
                    </div>

                    {!uploading && (
                        <button
                            type="button"
                            onClick={() => fileInputRef.current?.click()}
                            className="mt-4 px-6 py-2 bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white rounded-lg font-medium transition-colors"
                        >
                            <Plus className="w-4 h-4 mr-2 inline" />
                            Select Files
                        </button>
                    )}
                </div>
            </div>

            {/* Error Display */}
            {errors.general && (
                <motion.div
                    initial={{ opacity: 0, y: -10 }}
                    animate={{ opacity: 1, y: 0 }}
                    className="mt-4 p-4 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg"
                >
                    <div className="flex items-center">
                        <AlertCircle className="w-5 h-5 text-red-500 dark:text-red-400 mr-3" />
                        <p className="text-sm text-red-700 dark:text-red-300">
                            {errors.general}
                        </p>
                    </div>
                </motion.div>
            )}

            {/* File List */}
            <AnimatePresence>
                {files.length > 0 && (
                    <motion.div
                        initial={{ opacity: 0, height: 0 }}
                        animate={{ opacity: 1, height: "auto" }}
                        exit={{ opacity: 0, height: 0 }}
                        className="mt-6 space-y-3"
                    >
                        <div className="flex items-center justify-between">
                            <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                Selected Files ({files.length})
                            </h3>
                            <button
                                onClick={clearAllFiles}
                                className="text-sm text-red-600 hover:text-red-800 dark:text-red-400 dark:hover:text-red-300"
                                disabled={uploading}
                            >
                                Clear All
                            </button>
                        </div>

                        <div className="space-y-2">
                            {files.map((fileObj) => (
                                <motion.div
                                    key={fileObj.id}
                                    variants={fadeInUp}
                                    initial="initial"
                                    animate="animate"
                                    exit="exit"
                                    className="flex items-center justify-between p-3 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg"
                                >
                                    <div className="flex items-center space-x-3">
                                        <div className="w-10 h-10 bg-blue-100 dark:bg-blue-900/30 rounded-lg flex items-center justify-center">
                                            <FileText className="w-5 h-5 text-blue-600 dark:text-blue-400" />
                                        </div>

                                        <div>
                                            <p className="text-sm font-medium text-gray-900 dark:text-white">
                                                {fileObj.name}
                                            </p>
                                            <p className="text-xs text-gray-500 dark:text-gray-400">
                                                {formatFileSize(fileObj.size)}
                                            </p>
                                        </div>
                                    </div>

                                    <div className="flex items-center space-x-2">
                                        {/* Status Icon */}
                                        {fileObj.status === "pending" && (
                                            <div className="w-5 h-5 bg-gray-200 dark:bg-gray-600 rounded-full" />
                                        )}
                                        {fileObj.status === "uploading" && (
                                            <Loader2 className="w-5 h-5 text-blue-600 dark:text-blue-400 animate-spin" />
                                        )}
                                        {fileObj.status === "uploaded" && (
                                            <CheckCircle className="w-5 h-5 text-green-600 dark:text-green-400" />
                                        )}
                                        {fileObj.status === "error" && (
                                            <AlertCircle className="w-5 h-5 text-red-600 dark:text-red-400" />
                                        )}

                                        {/* Action Buttons */}
                                        <button
                                            onClick={() => previewFile(fileObj)}
                                            className="p-1 text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 transition-colors"
                                            title="Preview file"
                                        >
                                            <Eye className="w-4 h-4" />
                                        </button>

                                        <button
                                            onClick={() =>
                                                removeFile(fileObj.id)
                                            }
                                            className="p-1 text-gray-400 hover:text-red-600 dark:hover:text-red-400 transition-colors"
                                            title="Remove file"
                                            disabled={uploading}
                                        >
                                            <Trash2 className="w-4 h-4" />
                                        </button>
                                    </div>
                                </motion.div>
                            ))}
                        </div>

                        {/* Upload Progress */}
                        {uploading && (
                            <motion.div
                                initial={{ opacity: 0 }}
                                animate={{ opacity: 1 }}
                                className="mt-4 p-4 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg"
                            >
                                <div className="flex items-center justify-between mb-2">
                                    <span className="text-sm font-medium text-blue-900 dark:text-blue-300">
                                        Uploading files...
                                    </span>
                                    <div className="flex items-center space-x-2">
                                        <Loader2 className="w-4 h-4 text-blue-600 dark:text-blue-400 animate-spin" />
                                        <span className="text-xs text-blue-700 dark:text-blue-300">
                                            Please wait
                                        </span>
                                    </div>
                                </div>
                                <div className="w-full bg-blue-200 dark:bg-blue-800 rounded-full h-2">
                                    <div
                                        className="bg-blue-600 dark:bg-blue-400 h-2 rounded-full transition-all duration-300"
                                        style={{ width: "100%" }}
                                    />
                                </div>
                            </motion.div>
                        )}

                        {/* Upload Success */}
                        {uploadId && !uploading && (
                            <motion.div
                                initial={{ opacity: 0, y: 10 }}
                                animate={{ opacity: 1, y: 0 }}
                                className="mt-4 p-4 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg"
                            >
                                <div className="flex items-center">
                                    <CheckCircle className="w-5 h-5 text-green-500 dark:text-green-400 mr-3" />
                                    <div>
                                        <p className="text-sm font-medium text-green-900 dark:text-green-300">
                                            Files uploaded successfully!
                                        </p>
                                        <p className="text-xs text-green-700 dark:text-green-400 mt-1">
                                            Upload ID: {uploadId}
                                        </p>
                                    </div>
                                </div>
                            </motion.div>
                        )}

                        {/* Upload Actions */}
                        <div className="flex justify-between items-center pt-4">
                            <div className="text-sm text-gray-500 dark:text-gray-400">
                                {files.length} file
                                {files.length !== 1 ? "s" : ""} selected
                            </div>

                            <div className="flex space-x-3">
                                <button
                                    onClick={clearAllFiles}
                                    className="px-4 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
                                    disabled={uploading}
                                >
                                    Clear All
                                </button>

                                <button
                                    onClick={uploadFiles}
                                    disabled={uploading || files.length === 0}
                                    className={`px-6 py-2 text-sm font-medium rounded-lg transition-colors ${
                                        uploading || files.length === 0
                                            ? "bg-gray-300 dark:bg-gray-600 text-gray-500 dark:text-gray-400 cursor-not-allowed"
                                            : "bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white"
                                    }`}
                                >
                                    {uploading ? (
                                        <>
                                            <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                                            Uploading...
                                        </>
                                    ) : (
                                        <>
                                            <Upload className="w-4 h-4 mr-2" />
                                            Upload Files
                                        </>
                                    )}
                                </button>
                            </div>
                        </div>
                    </motion.div>
                )}
            </AnimatePresence>

            {/* File Input Errors */}
            {Object.keys(errors).length > 0 && (
                <div className="mt-4 space-y-2">
                    {Object.entries(errors).map(
                        ([key, error]) =>
                            key !== "general" && (
                                <motion.div
                                    key={key}
                                    initial={{ opacity: 0, x: -10 }}
                                    animate={{ opacity: 1, x: 0 }}
                                    className="text-sm text-red-600 dark:text-red-400 flex items-center"
                                >
                                    <AlertCircle className="w-4 h-4 mr-2" />
                                    {error}
                                </motion.div>
                            )
                    )}
                </div>
            )}

            {/* Usage Instructions */}
            <div className="mt-6 p-4 bg-gray-50 dark:bg-gray-800 rounded-lg">
                <h4 className="text-sm font-medium text-gray-900 dark:text-white mb-2">
                    File Requirements:
                </h4>
                <ul className="text-sm text-gray-600 dark:text-gray-400 space-y-1">
                    <li>• JavaScript files (.js) only</li>
                    <li>• Maximum {maxFiles} files per upload</li>
                    <li>• Maximum {formatFileSize(maxFileSize)} per file</li>
                    <li>• Files must export blog post data structure</li>
                    <li>
                        • Valid file names: letters, numbers, dots, underscores,
                        hyphens
                    </li>
                </ul>
            </div>
        </div>
    );
};

export default BlogFileUpload;
