import React, { useState, useEffect, useCallback } from "react";
import { motion, AnimatePresence } from "framer-motion";
import {
    Upload,
    FileText,
    CheckCircle,
    AlertCircle,
    Loader2,
    Database,
    RefreshCw,
    Clock,
    Activity,
    X,
    ChevronDown,
    ChevronUp,
    Eye,
    Download,
} from "lucide-react";

const BlogUploadProgress = ({
    uploadId,
    onComplete,
    onError,
    autoRefresh = true,
    refreshInterval = 2000,
    className = "",
}) => {
    // State management
    const [progress, setProgress] = useState({
        stage: "idle", // idle, uploading, processing, importing, complete, error
        uploadProgress: 0,
        processingProgress: 0,
        importProgress: 0,
        totalFiles: 0,
        processedFiles: 0,
        importedFiles: 0,
        currentFile: null,
        errors: [],
        startTime: null,
        endTime: null,
    });

    const [expanded, setExpanded] = useState(true);
    const [details, setDetails] = useState({});
    const [logs, setLogs] = useState([]);

    // Auto-refresh interval
    useEffect(() => {
        if (!uploadId || !autoRefresh) return;

        const interval = setInterval(() => {
            fetchProgress();
        }, refreshInterval);

        return () => clearInterval(interval);
    }, [uploadId, autoRefresh, refreshInterval]);

    // Initial fetch
    useEffect(() => {
        if (uploadId) {
            fetchProgress();
        }
    }, [uploadId]);

    // Fetch progress from API
    const fetchProgress = useCallback(async () => {
        if (!uploadId) return;

        try {
            const response = await fetch(`/blog/process/${uploadId}/status`, {
                headers: {
                    "X-CSRF-TOKEN": document
                        .querySelector('meta[name="csrf-token"]')
                        .getAttribute("content"),
                },
            });

            const result = await response.json();

            if (result.success) {
                updateProgress(result.status);
            } else {
                setProgress((prev) => ({
                    ...prev,
                    stage: "error",
                    errors: [...prev.errors, result.error],
                }));
            }
        } catch (error) {
            console.error("Progress fetch error:", error);
            setProgress((prev) => ({
                ...prev,
                stage: "error",
                errors: [...prev.errors, error.message],
            }));
        }
    }, [uploadId]);

    // Update progress state
    const updateProgress = useCallback(
        (status) => {
            setProgress((prev) => {
                const newProgress = { ...prev };

                // Update basic stats
                newProgress.totalFiles = status.total_files || 0;
                newProgress.processedFiles = status.processed_files || 0;
                newProgress.importedFiles = status.imported_files || 0;

                // Calculate percentages
                if (newProgress.totalFiles > 0) {
                    newProgress.processingProgress = Math.round(
                        (newProgress.processedFiles / newProgress.totalFiles) *
                            100
                    );
                }

                // Determine stage
                if (status.processing_complete && status.import_complete) {
                    newProgress.stage = "complete";
                    newProgress.endTime = new Date();
                    if (onComplete) onComplete(status);
                } else if (status.importing) {
                    newProgress.stage = "importing";
                } else if (status.processing) {
                    newProgress.stage = "processing";
                } else if (status.uploading) {
                    newProgress.stage = "uploading";
                }

                return newProgress;
            });
        },
        [onComplete]
    );

    // Process files
    const processFiles = useCallback(async () => {
        if (!uploadId) return;

        setProgress((prev) => ({ ...prev, stage: "processing" }));

        try {
            const response = await fetch("/blog/process/files", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                    "X-CSRF-TOKEN": document
                        .querySelector('meta[name="csrf-token"]')
                        .getAttribute("content"),
                },
                body: JSON.stringify({ upload_id: uploadId }),
            });

            const result = await response.json();

            if (result.success) {
                addLog("Files processed successfully", "success");
                fetchProgress();
            } else {
                setProgress((prev) => ({
                    ...prev,
                    stage: "error",
                    errors: [...prev.errors, result.error],
                }));
                addLog(`Processing failed: ${result.error}`, "error");
            }
        } catch (error) {
            console.error("Processing error:", error);
            setProgress((prev) => ({
                ...prev,
                stage: "error",
                errors: [...prev.errors, error.message],
            }));
            addLog(`Processing error: ${error.message}`, "error");
        }
    }, [uploadId]);

    // Import files
    const importFiles = useCallback(async () => {
        if (!uploadId) return;

        setProgress((prev) => ({ ...prev, stage: "importing" }));

        try {
            const response = await fetch("/blog/process/import", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                    "X-CSRF-TOKEN": document
                        .querySelector('meta[name="csrf-token"]')
                        .getAttribute("content"),
                },
                body: JSON.stringify({ upload_id: uploadId }),
            });

            const result = await response.json();

            if (result.success) {
                addLog("Files imported successfully", "success");
                setProgress((prev) => ({ ...prev, stage: "complete" }));
                if (onComplete) onComplete(result);
            } else {
                setProgress((prev) => ({
                    ...prev,
                    stage: "error",
                    errors: [...prev.errors, result.error],
                }));
                addLog(`Import failed: ${result.error}`, "error");
            }
        } catch (error) {
            console.error("Import error:", error);
            setProgress((prev) => ({
                ...prev,
                stage: "error",
                errors: [...prev.errors, error.message],
            }));
            addLog(`Import error: ${error.message}`, "error");
        }
    }, [uploadId, onComplete]);

    // Add log entry
    const addLog = useCallback((message, type = "info") => {
        setLogs((prev) => [
            ...prev,
            {
                id: Date.now(),
                message,
                type,
                timestamp: new Date(),
            },
        ]);
    }, []);

    // Calculate elapsed time
    const getElapsedTime = useCallback(() => {
        if (!progress.startTime) return null;

        const endTime = progress.endTime || new Date();
        const elapsed = Math.floor((endTime - progress.startTime) / 1000);

        const minutes = Math.floor(elapsed / 60);
        const seconds = elapsed % 60;

        return `${minutes}:${seconds.toString().padStart(2, "0")}`;
    }, [progress.startTime, progress.endTime]);

    // Get stage icon
    const getStageIcon = (stage) => {
        switch (stage) {
            case "uploading":
                return (
                    <Upload className="w-5 h-5 text-blue-600 dark:text-blue-400" />
                );
            case "processing":
                return (
                    <Loader2 className="w-5 h-5 text-yellow-600 dark:text-yellow-400 animate-spin" />
                );
            case "importing":
                return (
                    <Database className="w-5 h-5 text-purple-600 dark:text-purple-400" />
                );
            case "complete":
                return (
                    <CheckCircle className="w-5 h-5 text-green-600 dark:text-green-400" />
                );
            case "error":
                return (
                    <AlertCircle className="w-5 h-5 text-red-600 dark:text-red-400" />
                );
            default:
                return (
                    <Clock className="w-5 h-5 text-gray-600 dark:text-gray-400" />
                );
        }
    };

    // Get stage color
    const getStageColor = (stage) => {
        switch (stage) {
            case "uploading":
                return "blue";
            case "processing":
                return "yellow";
            case "importing":
                return "purple";
            case "complete":
                return "green";
            case "error":
                return "red";
            default:
                return "gray";
        }
    };

    // Get progress percentage
    const getOverallProgress = () => {
        const weights = { upload: 0.3, processing: 0.4, importing: 0.3 };

        let total = 0;
        if (progress.stage === "uploading")
            total = progress.uploadProgress * weights.upload;
        else if (progress.stage === "processing")
            total = 30 + progress.processingProgress * weights.processing;
        else if (progress.stage === "importing")
            total = 70 + progress.importProgress * weights.importing;
        else if (progress.stage === "complete") total = 100;

        return Math.min(100, Math.max(0, total));
    };

    if (!uploadId) {
        return null;
    }

    return (
        <div className={`blog-upload-progress ${className}`}>
            <div className="bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg shadow-sm">
                {/* Header */}
                <div className="p-4 border-b border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-between">
                        <div className="flex items-center space-x-3">
                            {getStageIcon(progress.stage)}
                            <div>
                                <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                    Upload Progress
                                </h3>
                                <p className="text-sm text-gray-500 dark:text-gray-400">
                                    {uploadId} •{" "}
                                    {progress.stage.charAt(0).toUpperCase() +
                                        progress.stage.slice(1)}
                                </p>
                            </div>
                        </div>

                        <div className="flex items-center space-x-2">
                            {getElapsedTime() && (
                                <span className="text-sm text-gray-500 dark:text-gray-400">
                                    {getElapsedTime()}
                                </span>
                            )}
                            <button
                                onClick={() => setExpanded(!expanded)}
                                className="p-1 text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                            >
                                {expanded ? (
                                    <ChevronUp className="w-5 h-5" />
                                ) : (
                                    <ChevronDown className="w-5 h-5" />
                                )}
                            </button>
                        </div>
                    </div>
                </div>

                {/* Progress Bar */}
                <div className="p-4">
                    <div className="flex items-center justify-between mb-2">
                        <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
                            Overall Progress
                        </span>
                        <span className="text-sm text-gray-500 dark:text-gray-400">
                            {Math.round(getOverallProgress())}%
                        </span>
                    </div>
                    <div className="w-full bg-gray-200 dark:bg-gray-700 rounded-full h-2">
                        <motion.div
                            className={`h-2 rounded-full bg-${getStageColor(
                                progress.stage
                            )}-600 dark:bg-${getStageColor(
                                progress.stage
                            )}-400`}
                            initial={{ width: 0 }}
                            animate={{ width: `${getOverallProgress()}%` }}
                            transition={{ duration: 0.5, ease: "easeOut" }}
                        />
                    </div>
                </div>

                {/* Expanded Details */}
                <AnimatePresence>
                    {expanded && (
                        <motion.div
                            initial={{ opacity: 0, height: 0 }}
                            animate={{ opacity: 1, height: "auto" }}
                            exit={{ opacity: 0, height: 0 }}
                            className="border-t border-gray-200 dark:border-gray-700"
                        >
                            {/* Statistics */}
                            <div className="p-4 grid grid-cols-3 gap-4">
                                <div className="text-center">
                                    <div className="text-2xl font-bold text-gray-900 dark:text-white">
                                        {progress.totalFiles}
                                    </div>
                                    <div className="text-sm text-gray-500 dark:text-gray-400">
                                        Total Files
                                    </div>
                                </div>
                                <div className="text-center">
                                    <div className="text-2xl font-bold text-yellow-600 dark:text-yellow-400">
                                        {progress.processedFiles}
                                    </div>
                                    <div className="text-sm text-gray-500 dark:text-gray-400">
                                        Processed
                                    </div>
                                </div>
                                <div className="text-center">
                                    <div className="text-2xl font-bold text-green-600 dark:text-green-400">
                                        {progress.importedFiles}
                                    </div>
                                    <div className="text-sm text-gray-500 dark:text-gray-400">
                                        Imported
                                    </div>
                                </div>
                            </div>

                            {/* Action Buttons */}
                            {progress.stage === "processing" && (
                                <div className="p-4 border-t border-gray-200 dark:border-gray-700">
                                    <div className="flex space-x-3">
                                        <button
                                            onClick={processFiles}
                                            className="flex-1 bg-yellow-600 hover:bg-yellow-700 dark:bg-yellow-500 dark:hover:bg-yellow-600 text-white px-4 py-2 rounded-lg font-medium transition-colors"
                                        >
                                            <Activity className="w-4 h-4 mr-2 inline" />
                                            Process Files
                                        </button>
                                    </div>
                                </div>
                            )}

                            {progress.stage === "importing" && (
                                <div className="p-4 border-t border-gray-200 dark:border-gray-700">
                                    <div className="flex space-x-3">
                                        <button
                                            onClick={importFiles}
                                            className="flex-1 bg-purple-600 hover:bg-purple-700 dark:bg-purple-500 dark:hover:bg-purple-600 text-white px-4 py-2 rounded-lg font-medium transition-colors"
                                        >
                                            <Database className="w-4 h-4 mr-2 inline" />
                                            Import to Database
                                        </button>
                                    </div>
                                </div>
                            )}

                            {/* Error Display */}
                            {progress.errors.length > 0 && (
                                <div className="p-4 border-t border-gray-200 dark:border-gray-700">
                                    <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg p-3">
                                        <div className="flex items-center mb-2">
                                            <AlertCircle className="w-5 h-5 text-red-500 dark:text-red-400 mr-2" />
                                            <span className="text-sm font-medium text-red-800 dark:text-red-200">
                                                Errors ({progress.errors.length}
                                                )
                                            </span>
                                        </div>
                                        <div className="space-y-1">
                                            {progress.errors.map(
                                                (error, index) => (
                                                    <div
                                                        key={index}
                                                        className="text-sm text-red-700 dark:text-red-300"
                                                    >
                                                        {error}
                                                    </div>
                                                )
                                            )}
                                        </div>
                                    </div>
                                </div>
                            )}

                            {/* Activity Logs */}
                            {logs.length > 0 && (
                                <div className="p-4 border-t border-gray-200 dark:border-gray-700">
                                    <div className="mb-3">
                                        <h4 className="text-sm font-medium text-gray-900 dark:text-white">
                                            Activity Log
                                        </h4>
                                    </div>
                                    <div className="max-h-32 overflow-y-auto space-y-2">
                                        {logs.map((log) => (
                                            <div
                                                key={log.id}
                                                className="flex items-start space-x-2"
                                            >
                                                <div
                                                    className={`w-2 h-2 rounded-full mt-2 ${
                                                        log.type === "success"
                                                            ? "bg-green-500"
                                                            : log.type ===
                                                              "error"
                                                            ? "bg-red-500"
                                                            : log.type ===
                                                              "warning"
                                                            ? "bg-yellow-500"
                                                            : "bg-blue-500"
                                                    }`}
                                                />
                                                <div className="flex-1">
                                                    <div className="text-sm text-gray-900 dark:text-white">
                                                        {log.message}
                                                    </div>
                                                    <div className="text-xs text-gray-500 dark:text-gray-400">
                                                        {log.timestamp.toLocaleTimeString()}
                                                    </div>
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                </div>
                            )}

                            {/* Stage Details */}
                            <div className="p-4 border-t border-gray-200 dark:border-gray-700">
                                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                                    {/* Upload Stage */}
                                    <div
                                        className={`p-3 rounded-lg border ${
                                            progress.stage === "uploading"
                                                ? "border-blue-200 bg-blue-50 dark:border-blue-800 dark:bg-blue-900/20"
                                                : progress.uploadProgress ===
                                                  100
                                                ? "border-green-200 bg-green-50 dark:border-green-800 dark:bg-green-900/20"
                                                : "border-gray-200 bg-gray-50 dark:border-gray-700 dark:bg-gray-800"
                                        }`}
                                    >
                                        <div className="flex items-center justify-between mb-2">
                                            <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                                Upload
                                            </span>
                                            <Upload
                                                className={`w-4 h-4 ${
                                                    progress.stage ===
                                                    "uploading"
                                                        ? "text-blue-600 dark:text-blue-400"
                                                        : progress.uploadProgress ===
                                                          100
                                                        ? "text-green-600 dark:text-green-400"
                                                        : "text-gray-400"
                                                }`}
                                            />
                                        </div>
                                        <div className="text-xs text-gray-500 dark:text-gray-400">
                                            {progress.uploadProgress}% complete
                                        </div>
                                    </div>

                                    {/* Processing Stage */}
                                    <div
                                        className={`p-3 rounded-lg border ${
                                            progress.stage === "processing"
                                                ? "border-yellow-200 bg-yellow-50 dark:border-yellow-800 dark:bg-yellow-900/20"
                                                : progress.processingProgress ===
                                                  100
                                                ? "border-green-200 bg-green-50 dark:border-green-800 dark:bg-green-900/20"
                                                : "border-gray-200 bg-gray-50 dark:border-gray-700 dark:bg-gray-800"
                                        }`}
                                    >
                                        <div className="flex items-center justify-between mb-2">
                                            <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                                Processing
                                            </span>
                                            <Activity
                                                className={`w-4 h-4 ${
                                                    progress.stage ===
                                                    "processing"
                                                        ? "text-yellow-600 dark:text-yellow-400"
                                                        : progress.processingProgress ===
                                                          100
                                                        ? "text-green-600 dark:text-green-400"
                                                        : "text-gray-400"
                                                }`}
                                            />
                                        </div>
                                        <div className="text-xs text-gray-500 dark:text-gray-400">
                                            {progress.processedFiles}/
                                            {progress.totalFiles} files
                                        </div>
                                    </div>

                                    {/* Import Stage */}
                                    <div
                                        className={`p-3 rounded-lg border ${
                                            progress.stage === "importing"
                                                ? "border-purple-200 bg-purple-50 dark:border-purple-800 dark:bg-purple-900/20"
                                                : progress.stage === "complete"
                                                ? "border-green-200 bg-green-50 dark:border-green-800 dark:bg-green-900/20"
                                                : "border-gray-200 bg-gray-50 dark:border-gray-700 dark:bg-gray-800"
                                        }`}
                                    >
                                        <div className="flex items-center justify-between mb-2">
                                            <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                                Import
                                            </span>
                                            <Database
                                                className={`w-4 h-4 ${
                                                    progress.stage ===
                                                    "importing"
                                                        ? "text-purple-600 dark:text-purple-400"
                                                        : progress.stage ===
                                                          "complete"
                                                        ? "text-green-600 dark:text-green-400"
                                                        : "text-gray-400"
                                                }`}
                                            />
                                        </div>
                                        <div className="text-xs text-gray-500 dark:text-gray-400">
                                            {progress.importedFiles}/
                                            {progress.totalFiles} imported
                                        </div>
                                    </div>
                                </div>
                            </div>

                            {/* Actions */}
                            <div className="p-4 border-t border-gray-200 dark:border-gray-700">
                                <div className="flex justify-between items-center">
                                    <div className="flex space-x-2">
                                        <button
                                            onClick={fetchProgress}
                                            className="px-3 py-1 text-sm bg-gray-100 hover:bg-gray-200 dark:bg-gray-700 dark:hover:bg-gray-600 text-gray-700 dark:text-gray-300 rounded-md transition-colors"
                                        >
                                            <RefreshCw className="w-4 h-4 mr-1 inline" />
                                            Refresh
                                        </button>

                                        {progress.stage === "complete" && (
                                            <button
                                                onClick={() => {
                                                    // View imported posts
                                                    window.open(
                                                        "/admin/blog/posts",
                                                        "_blank"
                                                    );
                                                }}
                                                className="px-3 py-1 text-sm bg-green-100 hover:bg-green-200 dark:bg-green-900/30 dark:hover:bg-green-900/50 text-green-700 dark:text-green-300 rounded-md transition-colors"
                                            >
                                                <Eye className="w-4 h-4 mr-1 inline" />
                                                View Posts
                                            </button>
                                        )}
                                    </div>

                                    <div className="text-xs text-gray-500 dark:text-gray-400">
                                        Auto-refresh:{" "}
                                        {autoRefresh ? "On" : "Off"}
                                    </div>
                                </div>
                            </div>
                        </motion.div>
                    )}
                </AnimatePresence>
            </div>
        </div>
    );
};

export default BlogUploadProgress;
