import OptimizedImage from "./OptimizedImage";
import {
    getBlogPostImage,
    getBlogCategoryImage,
    getBlogLayoutImage,
    getBlogImageAlt,
    getBlogThumbnailImage,
    getBlogThumbnailAlt,
    getImageSizes,
    getImageSourceSet,
    blogPostHasImages,
} from "@/data/blog/images";

/**
 * BlogImage Component - Centralized blog image management
 * Handles all blog-related images with optimization and fallbacks
 */
const BlogImage = ({
    post = null,
    category = null,
    author = null,
    layout = null,
    type = "featured", // featured, thumbnail, category, author, layout
    size = "default", // default, thumbnail, social
    alt = null,
    className = "w-full h-64 object-cover rounded-lg shadow-lg",
    ...props
}) => {
    // Determine image source and alt text based on props
    let imageSrc, imageAlt, imageData;

    if (post) {
        // Blog post image
        if (type === "thumbnail") {
            imageSrc = getBlogThumbnailImage(post, "webp");
            imageAlt = alt || getBlogThumbnailAlt(post);
        } else {
            imageSrc = getBlogPostImage(post, "featured", "webp");
            imageAlt = alt || getBlogImageAlt(post, "featured");
        }
        imageData = getImageSizes(
            type === "thumbnail" ? "thumbnail" : "featured"
        );
    } else if (category) {
        // Blog category image
        imageSrc = getBlogCategoryImage(category, "hero", "webp");
        imageAlt = alt || getBlogImageAlt(null, "hero", category);
        imageData = getImageSizes("hero");
    } else if (layout) {
        // Blog layout image
        imageSrc = getBlogLayoutImage(layout, "webp");
        imageAlt = alt || `Blog ${layout} image`;
        imageData = getImageSizes("featured");
    } else {
        // Fallback to default blog image
        imageSrc = getBlogLayoutImage("default", "webp");
        imageAlt = alt || "Academic Scribe Blog";
        imageData = getImageSizes("featured");
    }

    // Generate sizes attribute based on type and size
    const getSizesAttribute = () => {
        if (size === "thumbnail" || type === "thumbnail") {
            return "(max-width: 768px) 100vw, (max-width: 1024px) 50vw, 400px";
        } else if (type === "category") {
            return "(max-width: 768px) 100vw, (max-width: 1024px) 80vw, 1200px";
        } else if (type === "featured") {
            return "(max-width: 768px) 100vw, (max-width: 1024px) 80vw, 800px";
        } else if (type === "author") {
            return "(max-width: 768px) 200px, 400px";
        } else {
            return "(max-width: 768px) 100vw, (max-width: 1024px) 50vw, 800px";
        }
    };

    // Get fallback image source (JPG version)
    const getFallbackSrc = () => {
        if (post) {
            if (type === "thumbnail") {
                return getBlogThumbnailImage(post, "jpg");
            } else {
                return getBlogPostImage(post, "featured", "jpg");
            }
        } else if (category) {
            return getBlogCategoryImage(category, "hero", "jpg");
        } else if (layout) {
            return getBlogLayoutImage(layout, "jpg");
        } else {
            return getBlogLayoutImage("default", "jpg");
        }
    };

    // Generate structured data for SEO
    const getStructuredData = () => {
        if (typeof window === "undefined") return null;

        return {
            "@context": "https://schema.org",
            "@type": "ImageObject",
            url: `${window.location.origin}${imageSrc}`,
            width: imageData.width,
            height: imageData.height,
            description: imageAlt,
            name: imageAlt,
        };
    };

    return (
        <>
            {/* Use picture element for WebP support with fallback */}
            <picture className={className}>
                {/* WebP source for modern browsers */}
                <source
                    srcSet={imageSrc}
                    type="image/webp"
                    sizes={getSizesAttribute()}
                />

                {/* Fallback for browsers that don't support WebP */}
                <OptimizedImage
                    src={getFallbackSrc()}
                    alt={imageAlt}
                    className="w-full h-full object-cover"
                    width={imageData.width}
                    height={imageData.height}
                    sizes={getSizesAttribute()}
                    loading={type === "featured" ? "eager" : "lazy"}
                    {...props}
                />
            </picture>

            {/* Add structured data for SEO */}
            {type === "featured" && (
                <script
                    type="application/ld+json"
                    dangerouslySetInnerHTML={{
                        __html: JSON.stringify(getStructuredData()),
                    }}
                />
            )}
        </>
    );
};

export default BlogImage;
