import React, { useState } from 'react';
import { usePage } from '@inertiajs/react';
import { PriceCalculatorPopup, PriceCalculatorTrigger } from '@/Components/PriceCalculator';

/**
 * PriceCalculatorSection - Frontend rendering component for Price Calculator
 * 
 * This component renders the price calculator section on any page
 * based on the content configured in the CMS
 */
export default function PriceCalculatorSection({ content = {} }) {
    const { auth } = usePage().props;
    const [calculatorOpen, setCalculatorOpen] = useState(false);

    // Default values
    const {
        eyebrow = '',
        title = 'Get Instant Pricing',
        subtitle = 'Calculate the cost of your assignment in seconds. No commitment required.',
        description = '',
        buttonText = 'Calculate Your Price',
        buttonVariant = 'primary',
        showEyebrow = true,
        showDescription = false,
        backgroundColor = 'gradient',
        alignment = 'center',
    } = content;

    // Background class mapping
    const backgroundClasses = {
        gradient: 'bg-gradient-to-br from-blue-50 to-indigo-50 dark:from-gray-800 dark:to-gray-900',
        white: 'bg-white dark:bg-gray-900',
        gray: 'bg-gray-50 dark:bg-gray-800',
        blue: 'bg-blue-50 dark:bg-blue-900/20',
    };

    // Button variant class mapping
    const buttonClasses = {
        primary: 'bg-blue-600 hover:bg-blue-700 text-white',
        secondary: 'bg-gray-600 hover:bg-gray-700 text-white',
        outline: 'border-2 border-blue-600 text-blue-600 hover:bg-blue-50 dark:hover:bg-blue-900/20',
    };

    // Alignment class mapping
    const alignmentClass = alignment === 'center' ? 'text-center' : 'text-left';

    return (
        <>
            <section className={`py-12 ${backgroundClasses[backgroundColor] || backgroundClasses.gradient}`}>
                <div className={`max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 ${alignmentClass}`}>
                    {/* Eyebrow */}
                    {showEyebrow && eyebrow && (
                        <div className="text-sm font-semibold text-blue-600 dark:text-blue-400 uppercase tracking-wide mb-2">
                            {eyebrow}
                        </div>
                    )}

                    {/* Title */}
                    <h2 className="text-3xl font-bold text-gray-900 dark:text-white mb-4">
                        {title}
                    </h2>

                    {/* Subtitle */}
                    {subtitle && (
                        <p className="text-lg text-gray-600 dark:text-gray-300 mb-6">
                            {subtitle}
                        </p>
                    )}

                    {/* Extended Description (Rich Text) */}
                    {showDescription && description && (
                        <div 
                            className="prose prose-lg dark:prose-invert mx-auto mb-8"
                            dangerouslySetInnerHTML={{ __html: description }}
                        />
                    )}

                    {/* CTA Button */}
                    <PriceCalculatorTrigger
                        onClick={() => setCalculatorOpen(true)}
                        variant={buttonVariant}
                        source="price-calculator-section"
                        className={`text-lg px-8 py-4 ${buttonClasses[buttonVariant] || buttonClasses.primary}`}
                    >
                        {buttonText}
                    </PriceCalculatorTrigger>
                </div>
            </section>

            {/* Price Calculator Popup */}
            <PriceCalculatorPopup
                isOpen={calculatorOpen}
                onClose={() => setCalculatorOpen(false)}
                triggerSource="price-calculator-section"
                isAuthenticated={!!auth?.user}
                user={auth?.user || null}
            />
        </>
    );
}
