import React from 'react';
import { Link } from '@inertiajs/react';
import { motion } from 'framer-motion';
import {
    FileText,
    BookOpen,
    GraduationCap,
    ClipboardList,
    BarChart3,
    Search,
    Target,
    User,
    Code,
    ArrowRight,
    CheckCircle,
} from 'lucide-react';

const iconMap = {
    'file-text': FileText,
    'search': Search,
    'graduation-cap': GraduationCap,
    'clipboard-list': ClipboardList,
    'bar-chart-3': BarChart3,
    'book-open': BookOpen,
    'target': Target,
    'user': User,
    'code': Code,
};

export default function ServicesBlock({ content }) {
    if (!content) {
        return null;
    }

    // Animation variants
    const fadeIn = {
        hidden: { opacity: 0, y: 20 },
        visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
    };

    const staggerContainer = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: {
                staggerChildren: 0.1,
            },
        },
    };

    const getIconComponent = (iconName) => {
        const IconComponent = iconMap[iconName];
        return IconComponent ? <IconComponent className="h-8 w-8" /> : <FileText className="h-8 w-8" />;
    };

    const getColorClasses = (color) => {
        const colorMap = {
            blue: {
                gradient: 'from-blue-500 to-blue-600',
                bgGradient: 'from-blue-50 to-blue-100',
                darkBgGradient: 'from-blue-900/20 to-blue-800/20',
            },
            green: {
                gradient: 'from-green-500 to-green-600',
                bgGradient: 'from-green-50 to-green-100',
                darkBgGradient: 'from-green-900/20 to-green-800/20',
            },
            purple: {
                gradient: 'from-purple-500 to-purple-600',
                bgGradient: 'from-purple-50 to-purple-100',
                darkBgGradient: 'from-purple-900/20 to-purple-800/20',
            },
            orange: {
                gradient: 'from-orange-500 to-orange-600',
                bgGradient: 'from-orange-50 to-orange-100',
                darkBgGradient: 'from-orange-900/20 to-orange-800/20',
            },
            indigo: {
                gradient: 'from-indigo-500 to-indigo-600',
                bgGradient: 'from-indigo-50 to-indigo-100',
                darkBgGradient: 'from-indigo-900/20 to-indigo-800/20',
            },
            teal: {
                gradient: 'from-teal-500 to-teal-600',
                bgGradient: 'from-teal-50 to-teal-100',
                darkBgGradient: 'from-teal-900/20 to-teal-800/20',
            },
            pink: {
                gradient: 'from-pink-500 to-pink-600',
                bgGradient: 'from-pink-50 to-pink-100',
                darkBgGradient: 'from-pink-900/20 to-pink-800/20',
            },
            red: {
                gradient: 'from-red-500 to-red-600',
                bgGradient: 'from-red-50 to-red-100',
                darkBgGradient: 'from-red-900/20 to-red-800/20',
            },
        };
        return colorMap[color] || colorMap.blue;
    };

    return (
        <section className="py-16 bg-gray-50 dark:bg-gray-800">
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <motion.div
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true }}
                    variants={fadeIn}
                    className="text-center mb-12"
                >
                    <h2 className="text-3xl sm:text-4xl font-bold text-gray-900 dark:text-white mb-4">
                        {content.title}
                    </h2>
                    {content.subtitle && (
                        <p className="text-lg text-gray-600 dark:text-gray-300 max-w-3xl mx-auto">
                            {content.subtitle}
                        </p>
                    )}
                </motion.div>

                <motion.div
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true }}
                    variants={staggerContainer}
                    className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8"
                >
                    {content.services && content.services.map((service, index) => {
                        const colorClasses = getColorClasses(service.color);
                        return (
                            <motion.div
                                key={index}
                                variants={fadeIn}
                                className="group"
                            >
                                <div className={`bg-gradient-to-br ${colorClasses.bgGradient} dark:bg-gradient-to-br dark:${colorClasses.darkBgGradient} rounded-xl p-6 h-full transition-all duration-300 group-hover:shadow-lg group-hover:-translate-y-1`}>
                                    <div className="flex items-center justify-center w-16 h-16 mx-auto mb-4 bg-white dark:bg-gray-800 rounded-lg shadow-sm">
                                        <div className={`text-${service.color}-600 dark:text-${service.color}-400`}>
                                            {getIconComponent(service.icon)}
                                        </div>
                                    </div>

                                    <h3 className="text-xl font-semibold text-gray-900 dark:text-white mb-3 text-center">
                                        {(service.href || service.link) ? (
                                            <Link href={service.href || service.link} className="no-link-style text-inherit">
                                                {service.name || service.title}
                                            </Link>
                                        ) : (
                                            service.name || service.title
                                        )}
                                    </h3>

                                    <div 
                                        className="text-gray-600 dark:text-gray-300 text-center mb-4 text-sm prose prose-sm dark:prose-invert max-w-none"
                                        dangerouslySetInnerHTML={{ __html: service.description }}
                                    />

                                    {service.features && service.features.length > 0 && (
                                        <ul className="space-y-2">
                                            {service.features.map((feature, featureIndex) => (
                                                <li key={featureIndex} className="flex items-center text-sm text-gray-600 dark:text-gray-300">
                                                    <CheckCircle className="h-4 w-4 text-green-500 mr-2 flex-shrink-0" />
                                                    {feature}
                                                </li>
                                            ))}
                                        </ul>
                                    )}
                                </div>
                            </motion.div>
                        );
                    })}
                </motion.div>
            </div>
        </section>
    );
}

