import React, { useState } from 'react';
import { Link } from '@inertiajs/react';
import {
    Star,
    Quote,
    ChevronLeft,
    ChevronRight,
    Award,
    GraduationCap,
    BookOpen,
    Users,
    CheckCircle,
    Heart,
    TrendingUp,
    Globe,
    Calendar,
    User,
} from 'lucide-react';

export default function TestimonialsBlock({ content }) {
    if (!content) return null;

    const title = content.title || '';
    const subtitle = content.subtitle || '';
    const categories = content.categories || [];
    const overallStats = content.overallStats || {};
    const serviceHighlights = content.serviceHighlights || [];
    const trustIndicators = content.trustIndicators || [];

    const [activeTab, setActiveTab] = useState(0);
    const [currentTestimonial, setCurrentTestimonial] = useState(0);

    const currentTestimonials = categories[activeTab]?.testimonials || [];

    const nextTestimonial = () => {
        setCurrentTestimonial((prev) =>
            prev === currentTestimonials.length - 1 ? 0 : prev + 1
        );
    };

    const prevTestimonial = () => {
        setCurrentTestimonial((prev) =>
            prev === 0 ? currentTestimonials.length - 1 : prev - 1
        );
    };

    const renderStars = (rating) => {
        return Array.from({ length: 5 }, (_, i) => (
            <Star
                key={i}
                className={`w-5 h-5 ${
                    i < rating
                        ? "text-yellow-400 fill-current"
                        : "text-gray-300 dark:text-gray-600"
                }`}
            />
        ));
    };

    return (
        <section className="py-20 bg-gradient-to-br from-gray-50 to-blue-50 dark:from-gray-900 dark:to-gray-800 w-full">
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                {/* Section Header */}
                <div className="text-center mb-16">
                    {title && (
                        <h2 className="text-4xl md:text-5xl font-bold text-gray-900 dark:text-white mb-6">
                            {title}
                        </h2>
                    )}
                    <div className="mt-4 h-1 w-24 bg-gradient-to-r from-blue-600 to-purple-600 mx-auto rounded-full"></div>
                    {subtitle && (
                        <div className="content-area mt-6 max-w-4xl mx-auto text-xl text-gray-600 dark:text-gray-300 leading-relaxed"
                           dangerouslySetInnerHTML={{ __html: subtitle }}
                        />
                    )}
                </div>
                
                {/* Overall Statistics */}
                {Object.keys(overallStats).length > 0 && (
                    <div className="mb-16">
                        <div className="bg-white dark:bg-gray-800 rounded-2xl shadow-xl p-8 border border-gray-200 dark:border-gray-700">
                            <h3 className="text-2xl font-bold text-gray-900 dark:text-white text-center mb-8">
                                Trusted Academic Writing Services - Client Satisfaction Overview
                            </h3>
                            <div className="grid grid-cols-2 md:grid-cols-4 gap-8">
                                <div className="text-center">
                                    <div className="flex justify-center mb-3">
                                        <Star className="w-12 h-12 text-yellow-400 fill-current" />
                                    </div>
                                    <div className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                                        {overallStats.averageRating}/5
                                    </div>
                                    <div className="text-sm text-gray-600 dark:text-gray-300">
                                        Average Rating
                                    </div>
                                </div>
                                <div className="text-center">
                                    <div className="flex justify-center mb-3">
                                        <Users className="w-12 h-12 text-blue-500" />
                                    </div>
                                    <div className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                                        {overallStats.totalReviews?.toLocaleString()}+
                                    </div>
                                    <div className="text-sm text-gray-600 dark:text-gray-300">
                                        Verified Reviews
                                    </div>
                                </div>
                                <div className="text-center">
                                    <div className="flex justify-center mb-3">
                                        <Heart className="w-12 h-12 text-red-500" />
                                    </div>
                                    <div className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                                        {overallStats.recommendationRate}%
                                    </div>
                                    <div className="text-sm text-gray-600 dark:text-gray-300">
                                        Recommend Us
                                    </div>
                                </div>
                                <div className="text-center">
                                    <div className="flex justify-center mb-3">
                                        <TrendingUp className="w-12 h-12 text-green-500" />
                                    </div>
                                    <div className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                                        {overallStats.repeatCustomers}%
                                    </div>
                                    <div className="text-sm text-gray-600 dark:text-gray-300">
                                        Repeat Clients
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                )}

                {/* Category Tabs */}
                {categories.length > 0 && (
                    <div className="flex justify-center mb-12">
                        <div className="flex rounded-xl bg-white dark:bg-gray-800 p-2 shadow-lg border border-gray-200 dark:border-gray-700">
                            {categories.map((category, idx) => (
                                <button
                                    key={idx}
                                    className={`flex items-center px-6 py-3 rounded-lg font-medium transition-all duration-300 ${
                                        activeTab === idx
                                            ? "bg-gradient-to-r from-blue-600 to-purple-600 text-white shadow-md transform scale-105"
                                            : "text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white hover:bg-gray-50 dark:hover:bg-gray-700"
                                    }`}
                                    onClick={() => setActiveTab(idx)}
                                >
                                    {category.name}
                                </button>
                            ))}
                        </div>
                    </div>
                )}

                {/* Featured Testimonial Carousel */}
                {currentTestimonials.length > 0 && (
                    <div className="mb-16">
                        <div className="relative bg-white dark:bg-gray-800 rounded-2xl shadow-2xl overflow-hidden border border-gray-200 dark:border-gray-700">
                            <div className="absolute top-0 left-0 w-full h-2 bg-gradient-to-r from-blue-600 to-purple-600"></div>

                            <div className="p-8 md:p-12">
                                <div className="flex items-center justify-between mb-8">
                                    <div className="flex items-center space-x-4">
                                        <div className="flex items-center space-x-1">
                                            {renderStars(currentTestimonials[currentTestimonial]?.rating || 5)}
                                        </div>
                                        {currentTestimonials[currentTestimonial]?.verified && (
                                            <div className="flex items-center text-green-700 dark:text-green-400">
                                                <CheckCircle className="w-5 h-5 mr-1" />
                                                <span className="text-sm font-medium">Verified Review</span>
                                            </div>
                                        )}
                                    </div>
                                    <div className="flex items-center space-x-2">
                                        <button
                                            onClick={prevTestimonial}
                                            aria-label="Previous testimonial"
                                            className="p-2 rounded-full bg-gray-100 dark:bg-gray-700 hover:bg-gray-200 dark:hover:bg-gray-600 transition-colors duration-200"
                                        >
                                            <ChevronLeft className="w-5 h-5 text-gray-600 dark:text-gray-300" />
                                        </button>
                                        <button
                                            onClick={nextTestimonial}
                                            aria-label="Next testimonial"
                                            className="p-2 rounded-full bg-gray-100 dark:bg-gray-700 hover:bg-gray-200 dark:hover:bg-gray-600 transition-colors duration-200"
                                        >
                                            <ChevronRight className="w-5 h-5 text-gray-600 dark:text-gray-300" />
                                        </button>
                                    </div>
                                </div>

                                <div className="grid md:grid-cols-3 gap-8 items-center">
                                    {/* Testimonial Content */}
                                    <div className="md:col-span-2">
                                        <Quote className="w-12 h-12 text-blue-600 dark:text-blue-400 mb-4" />
                                        <blockquote className="text-xl md:text-2xl text-gray-700 dark:text-gray-300 leading-relaxed mb-6 font-medium">
                                            "{currentTestimonials[currentTestimonial]?.text}"
                                        </blockquote>

                                        <div className="space-y-2">
                                            <div className="flex items-center space-x-3">
                                                <User className="w-5 h-5 text-gray-500 dark:text-gray-400" />
                                                <span className="font-bold text-gray-900 dark:text-white text-lg">
                                                    {currentTestimonials[currentTestimonial]?.author}
                                                </span>
                                            </div>
                                            <div className="flex items-center space-x-3">
                                                <GraduationCap className="w-5 h-5 text-gray-500 dark:text-gray-400" />
                                                <span className="text-gray-600 dark:text-gray-300">
                                                    {currentTestimonials[currentTestimonial]?.role}
                                                </span>
                                            </div>
                                            <div className="flex items-center space-x-3">
                                                <Globe className="w-5 h-5 text-gray-500 dark:text-gray-400" />
                                                <span className="text-gray-600 dark:text-gray-300">
                                                    {currentTestimonials[currentTestimonial]?.university}
                                                </span>
                                            </div>
                                            <div className="flex items-center space-x-3">
                                                <Calendar className="w-5 h-5 text-gray-500 dark:text-gray-400" />
                                                <span className="text-gray-600 dark:text-gray-300">
                                                    {currentTestimonials[currentTestimonial]?.date}
                                                </span>
                                            </div>
                                        </div>
                                    </div>

                                    {/* Client Info Card */}
                                    <div className="bg-gradient-to-br from-blue-50 to-purple-50 dark:from-gray-700 dark:to-gray-600 rounded-xl p-6">
                                        <div className="text-center">
                                            <div className="w-20 h-20 bg-gradient-to-r from-blue-600 to-purple-600 rounded-full flex items-center justify-center mx-auto mb-4">
                                                <User className="w-10 h-10 text-white" />
                                            </div>
                                            <h4 className="font-bold text-gray-900 dark:text-white mb-2">Service Used</h4>
                                            <p className="text-blue-600 dark:text-blue-400 font-medium mb-4">
                                                {currentTestimonials[currentTestimonial]?.service}
                                            </p>
                                            <div className="space-y-3">
                                                <div className="flex justify-between items-center">
                                                    <span className="text-gray-600 dark:text-gray-300">Grade Achieved:</span>
                                                    <span className="font-bold text-green-600 dark:text-green-400">
                                                        {currentTestimonials[currentTestimonial]?.grade}
                                                    </span>
                                                </div>
                                                <div className="flex justify-between items-center">
                                                    <span className="text-gray-600 dark:text-gray-300">Rating:</span>
                                                    <div className="flex items-center space-x-1">
                                                        {renderStars(currentTestimonials[currentTestimonial]?.rating || 5)}
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                {/* Testimonial Navigation Dots */}
                                <div className="flex justify-center mt-8 space-x-3" role="tablist" aria-label="Testimonial navigation">
                                    {currentTestimonials.map((_, index) => (
                                        <button
                                            key={index}
                                            onClick={() => setCurrentTestimonial(index)}
                                            role="tab"
                                            aria-label={`View testimonial ${index + 1} of ${currentTestimonials.length}`}
                                            aria-selected={index === currentTestimonial}
                                            className={`w-4 h-4 p-2 rounded-full transition-all duration-300 ${
                                                index === currentTestimonial
                                                    ? "bg-blue-600 dark:bg-blue-400 scale-125"
                                                    : "bg-gray-300 dark:bg-gray-600 hover:bg-gray-400 dark:hover:bg-gray-500"
                                            }`}
                                            style={{ minWidth: '48px', minHeight: '48px' }}
                                        />
                                    ))}
                                </div>
                            </div>
                        </div>
                                    </div>
                                )}
                                
                {/* All Testimonials Grid */}
                {currentTestimonials.length > 0 && (
                    <div className="mb-16">
                        <h3 className="text-3xl font-bold text-gray-900 dark:text-white text-center mb-12">
                            More Success Stories from Our {categories[activeTab]?.name}
                        </h3>
                        <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
                            {currentTestimonials.map((testimonial, index) => (
                                <div
                                    key={index}
                                    className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-lg hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1 border border-gray-200 dark:border-gray-700"
                                >
                                    {/* Header */}
                                    <div className="flex items-center justify-between mb-4">
                                        <div className="flex items-center space-x-1">
                                            {renderStars(testimonial.rating || 5)}
                                        </div>
                                        {testimonial.verified && (
                                            <div className="flex items-center text-green-700 dark:text-green-400">
                                                <CheckCircle className="w-4 h-4 mr-1" />
                                                <span className="text-xs font-medium">Verified</span>
                                            </div>
                                        )}
                                    </div>

                                    {/* Quote */}
                                    <Quote className="w-8 h-8 text-blue-600 dark:text-blue-400 mb-3" />
                                    <blockquote 
                                        className="text-gray-700 dark:text-gray-300 mb-4 leading-relaxed prose prose-sm dark:prose-invert max-w-none"
                                        dangerouslySetInnerHTML={{ __html: testimonial.text?.length > 150 ? testimonial.text.substring(0, 150) + "..." : testimonial.text }}
                                    />
                                
                                    {/* Author Info */}
                                    <div className="border-t border-gray-200 dark:border-gray-700 pt-4">
                                        <div className="flex items-center space-x-3 mb-2">
                                            <div className="w-10 h-10 bg-gradient-to-r from-blue-600 to-purple-600 rounded-full flex items-center justify-center">
                                                <User className="w-5 h-5 text-white" />
                                            </div>
                                            <div>
                                                <h4 className="font-bold text-gray-900 dark:text-white">{testimonial.author}</h4>
                                                <p className="text-sm text-gray-600 dark:text-gray-300">{testimonial.role}</p>
                                            </div>
                                        </div>

                                        <div className="space-y-1 text-sm">
                                            <div className="flex items-center justify-between">
                                                <span className="text-gray-500 dark:text-gray-400">Service:</span>
                                                <span className="text-blue-600 dark:text-blue-400 font-medium">{testimonial.service}</span>
                                            </div>
                                            <div className="flex items-center justify-between">
                                                <span className="text-gray-500 dark:text-gray-400">Grade:</span>
                                                <span className="text-green-700 dark:text-green-400 font-bold">{testimonial.grade}</span>
                                            </div>
                                            <div className="flex items-center justify-between">
                                                <span className="text-gray-500 dark:text-gray-400">Date:</span>
                                                <span className="text-gray-600 dark:text-gray-300">{testimonial.date}</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    </div>
                )}

                {/* Service-Specific Testimonial Highlights */}
                {serviceHighlights.length > 0 && (
                    <div className="mb-16">
                        <h3 className="text-3xl font-bold text-gray-900 dark:text-white text-center mb-12">
                            Academic Writing Services Success by Category
                        </h3>
                        <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
                            {serviceHighlights.map((service, index) => (
                                <div
                                    key={index}
                                    className="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-lg border border-gray-200 dark:border-gray-700"
                                >
                                    <div className={`bg-gradient-to-r ${service.color} p-4 rounded-lg mb-6`}>
                                        <div className="flex items-center justify-between text-white">
                                            <span className="text-2xl font-bold">{service.satisfaction}</span>
                                        </div>
                                    </div>

                                    <h4 className="text-xl font-bold text-gray-900 dark:text-white mb-4">
                                        {service.service}
                                        </h4>

                                    <div className="space-y-3">
                                        <div className="flex justify-between items-center">
                                            <span className="text-gray-600 dark:text-gray-300">Client Satisfaction:</span>
                                            <span className="font-bold text-green-700 dark:text-green-400">{service.satisfaction}</span>
                                        </div>
                                        <div className="flex justify-between items-center">
                                            <span className="text-gray-600 dark:text-gray-300">Average Grade:</span>
                                            <span className="font-bold text-blue-700 dark:text-blue-400">{service.avgGrade}</span>
                                        </div>
                                        <div className="flex justify-between items-center">
                                            <span className="text-gray-600 dark:text-gray-300">Completion Rate:</span>
                                            <span className="font-bold text-purple-700 dark:text-purple-400">{service.completionRate}</span>
                                        </div>
                                        <div className="flex justify-between items-center">
                                            <span className="text-gray-600 dark:text-gray-300">Reviews:</span>
                                            <span className="font-bold text-gray-900 dark:text-white">{service.testimonialCount}+</span>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    </div>
                )}

                {/* Trust Indicators */}
                {trustIndicators.length > 0 && (
                    <div className="mb-16">
                        <div className="bg-gradient-to-r from-blue-600 to-purple-600 rounded-2xl p-8 text-white">
                            <h3 className="text-3xl font-bold text-center mb-8">
                                Why Students Trust Our Academic Writing Services
                            </h3>
                            <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
                                {trustIndicators.map((feature, index) => (
                                    <div key={index} className="text-center">
                                        <div className="flex justify-center mb-4">
                                            {feature.icon}
                                        </div>
                                        <h4 className="font-bold mb-2">{feature.title}</h4>
                                        <p className="text-blue-100 text-sm">{feature.description}</p>
                                    </div>
                                ))}
                            </div>
                        </div>
                    </div>
                )}

                {/* Call to Action */}
                <div className="text-center">
                    <div className="bg-white dark:bg-gray-800 rounded-2xl p-8 md:p-12 shadow-xl border border-gray-200 dark:border-gray-700">
                        <h3 className="text-3xl md:text-4xl font-bold text-gray-900 dark:text-white mb-6">
                            Ready to Join Our Success Stories?
                        </h3>
                        <p className="text-xl text-gray-600 dark:text-gray-300 mb-8 max-w-3xl mx-auto leading-relaxed">
                            Experience the same exceptional results as our satisfied clients. Our professional essay writers and comprehensive academic writing services are ready to help you achieve academic excellence.
                        </p>

                        <div className="flex flex-col sm:flex-row justify-center space-y-4 sm:space-y-0 sm:space-x-6 mb-8">
                            <Link
                                href="/place-order"
                                className="bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white px-8 py-4 rounded-lg font-bold transition-all duration-300 transform hover:scale-105 shadow-lg"
                            >
                                Start Your Success Story
                            </Link>
                            <Link
                                href="/reviews"
                                className="border-2 border-blue-600 text-blue-600 dark:text-blue-400 hover:bg-blue-600 hover:text-white dark:hover:bg-blue-600 dark:hover:text-white px-8 py-4 rounded-lg font-bold transition-all duration-300 transform hover:scale-105"
                            >
                                View More Reviews
                            </Link>
                        </div>

                        {/* Additional Trust Elements */}
                        <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm text-gray-600 dark:text-gray-300">
                            <div className="flex items-center justify-center">
                                <CheckCircle className="w-4 h-4 mr-2 text-green-500" />
                                <span>Money-Back Guarantee</span>
                            </div>
                            <div className="flex items-center justify-center">
                                <CheckCircle className="w-4 h-4 mr-2 text-green-500" />
                                <span>100% Original Work</span>
                            </div>
                            <div className="flex items-center justify-center">
                                <CheckCircle className="w-4 h-4 mr-2 text-green-500" />
                                <span>24/7 Support</span>
                            </div>
                            <div className="flex items-center justify-center">
                                <CheckCircle className="w-4 h-4 mr-2 text-green-500" />
                                <span>Unlimited Revisions</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </section>
    );
}
