import React, { useState } from 'react';

export default function VideoBlock({ content }) {
    if (!content) return null;

    const {
        title = '',
        description = '',
        video_url = '',
        thumbnail = ''
    } = content;

    const [isPlaying, setIsPlaying] = useState(false);

    const getVideoEmbedUrl = (url) => {
        if (!url) return '';
        
        // YouTube
        if (url.includes('youtube.com') || url.includes('youtu.be')) {
            let videoId = '';
            if (url.includes('youtu.be/')) {
                videoId = url.split('youtu.be/')[1].split('?')[0];
            } else if (url.includes('youtube.com/watch')) {
                videoId = url.split('v=')[1].split('&')[0];
            } else if (url.includes('youtube.com/embed/')) {
                videoId = url.split('embed/')[1].split('?')[0];
            }
            return `https://www.youtube.com/embed/${videoId}`;
        }
        
        // Vimeo
        if (url.includes('vimeo.com')) {
            const videoId = url.split('vimeo.com/')[1].split('?')[0];
            return `https://player.vimeo.com/video/${videoId}`;
        }
        
        // Direct video file
        return url;
    };

    const embedUrl = getVideoEmbedUrl(video_url);

    return (
        <section className="video-section py-16 bg-gray-50">
            <div className="container mx-auto px-4">
                <div className="max-w-4xl mx-auto">
                    {title && (
                        <div className="text-center mb-8">
                            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
                                {title}
                            </h2>
                            
                            {description && (
                                <p className="text-lg text-gray-600">
                                    {description}
                                </p>
                            )}
                        </div>
                    )}
                    
                    <div className="relative aspect-video bg-black rounded-lg overflow-hidden shadow-lg">
                        {isPlaying && embedUrl ? (
                            <iframe
                                src={embedUrl}
                                title={title || 'Video'}
                                className="w-full h-full"
                                frameBorder="0"
                                allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
                                allowFullScreen
                            />
                        ) : (
                            <div 
                                className="w-full h-full flex items-center justify-center cursor-pointer group"
                                onClick={() => setIsPlaying(true)}
                            >
                                {thumbnail ? (
                                    <div className="relative w-full h-full">
                                        <img
                                            src={thumbnail}
                                            alt={title || 'Video thumbnail'}
                                            className="w-full h-full object-cover"
                                        />
                                        <div className="absolute inset-0 bg-black bg-opacity-30 flex items-center justify-center">
                                            <div className="w-20 h-20 bg-white bg-opacity-90 rounded-full flex items-center justify-center group-hover:bg-opacity-100 transition-all">
                                                <svg className="w-8 h-8 text-gray-900 ml-1" fill="currentColor" viewBox="0 0 20 20">
                                                    <path d="M8 5v10l8-5-8-5z" />
                                                </svg>
                                            </div>
                                        </div>
                                    </div>
                                ) : (
                                    <div className="text-center">
                                        <div className="w-20 h-20 bg-white bg-opacity-90 rounded-full flex items-center justify-center mx-auto mb-4 group-hover:bg-opacity-100 transition-all">
                                            <svg className="w-8 h-8 text-gray-900 ml-1" fill="currentColor" viewBox="0 0 20 20">
                                                <path d="M8 5v10l8-5-8-5z" />
                                            </svg>
                                        </div>
                                        <p className="text-white text-lg font-semibold">Click to play video</p>
                                    </div>
                                )}
                            </div>
                        )}
                    </div>
                </div>
            </div>
        </section>
    );
}
