import { useState, useEffect } from "react";

export default function ClearDeadlineCountdown({ deadline }) {
    const [timeDisplay, setTimeDisplay] = useState("");
    const [isExpired, setIsExpired] = useState(false);

    useEffect(() => {
        const calculateTime = () => {
            const now = new Date();
            const deadlineDate = new Date(deadline);
            const diff = deadlineDate - now;

            if (diff > 0) {
                // Future deadline - show countdown
                const totalDays = Math.floor(diff / (1000 * 60 * 60 * 24));
                const years = Math.floor(totalDays / 365);
                const remainingDaysAfterYears = totalDays % 365;
                const months = Math.floor(remainingDaysAfterYears / 30);
                const days = remainingDaysAfterYears % 30;

                const hours = Math.floor(
                    (diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60)
                );
                const minutes = Math.floor(
                    (diff % (1000 * 60 * 60)) / (1000 * 60)
                );
                const seconds = Math.floor((diff % (1000 * 60)) / 1000);

                setIsExpired(false);

                // Build the display string with years, months, days
                let displayParts = [];

                if (years > 0) {
                    displayParts.push(`${years}y`);
                }

                if (months > 0 || years > 0) {
                    displayParts.push(`${months}m`);
                }

                displayParts.push(`${days}d`);
                displayParts.push(`${hours}h`);
                displayParts.push(`${minutes}m`);
                displayParts.push(`${seconds}s`);

                setTimeDisplay(displayParts.join(" "));
            } else {
                // Past deadline - show time elapsed
                const elapsed = Math.abs(diff);
                const minutes = Math.floor(elapsed / (1000 * 60));
                const hours = Math.floor(elapsed / (1000 * 60 * 60));
                const days = Math.floor(elapsed / (1000 * 60 * 60 * 24));
                const months = Math.floor(days / 30);
                const years = Math.floor(days / 365);

                setIsExpired(true);

                if (years > 0) {
                    setTimeDisplay(`${years} year${years > 1 ? "s" : ""} ago`);
                } else if (months > 0) {
                    setTimeDisplay(
                        `${months} month${months > 1 ? "s" : ""} ago`
                    );
                } else if (days > 0) {
                    setTimeDisplay(`${days} day${days > 1 ? "s" : ""} ago`);
                } else if (hours > 0) {
                    setTimeDisplay(`${hours} hour${hours > 1 ? "s" : ""} ago`);
                } else {
                    setTimeDisplay(
                        `${minutes} minute${minutes > 1 ? "s" : ""} ago`
                    );
                }
            }
        };

        calculateTime();
        const timer = setInterval(calculateTime, 1000);
        return () => clearInterval(timer);
    }, [deadline]);

    return (
        <div className="text-center ">
            <h3
                className={`text-md font-bold mb-2 ${
                    isExpired
                        ? "text-red-600 dark:text-red-400"
                        : "text-orange-700 dark:text-orange-700"
                }`}
            >
                {isExpired ? "Expired" : ""}
            </h3>
            <span
                className={`font-mono text-md ${
                    isExpired
                        ? "text-red-500 dark:text-red-300"
                        : "text-orange-600 font-semibold dark:text-orange-500"
                }`}
            >
                {timeDisplay}
            </span>
        </div>
    );
}
