import { useState, useEffect } from 'react';
import { Link } from '@inertiajs/react';
import { X, Sparkles } from 'lucide-react';

export default function ExitIntentPopup() {
    const [isVisible, setIsVisible] = useState(false);
    const [shouldShow, setShouldShow] = useState(true);

    useEffect(() => {
        // Check if user has seen the popup recently
        const lastShown = localStorage.getItem('exitPopupLastShown');
        const now = Date.now();
        const oneDay = 24 * 60 * 60 * 1000; // 24 hours

        if (lastShown && (now - parseInt(lastShown)) < oneDay) {
            setShouldShow(false);
            return;
        }

        // Exit intent detection
        const handleMouseLeave = (e) => {
            // Detect when mouse leaves from the top of the viewport
            if (e.clientY <= 0 && shouldShow) {
                setIsVisible(true);
                localStorage.setItem('exitPopupLastShown', now.toString());
            }
        };

        // Add small delay before activating to avoid false triggers
        const timer = setTimeout(() => {
            document.addEventListener('mouseleave', handleMouseLeave);
        }, 3000); // Wait 3 seconds after page load

        return () => {
            clearTimeout(timer);
            document.removeEventListener('mouseleave', handleMouseLeave);
        };
    }, [shouldShow]);

    const handleClose = () => {
        setIsVisible(false);
    };

    const handleCtaClick = () => {
        setIsVisible(false);
        // Don't show again for this session
        setShouldShow(false);
    };

    if (!isVisible) return null;

    return (
        <>
            {/* Backdrop */}
            <div
                className="fixed inset-0 bg-black/60 backdrop-blur-sm z-[9998] animate-in fade-in duration-300"
                onClick={handleClose}
            />

            {/* Popup */}
            <div className="fixed inset-0 z-[9999] flex items-center justify-center p-4 pointer-events-none">
                <div
                    className="relative bg-white dark:bg-gray-800 rounded-2xl shadow-2xl max-w-2xl w-full pointer-events-auto animate-in zoom-in-95 slide-in-from-bottom-5 duration-500 flex flex-col overflow-hidden"
                    onClick={(e) => e.stopPropagation()}
                >
                    {/* Close Button */}
                    <button
                        onClick={handleClose}
                        className="absolute top-4 right-4 text-gray-400 hover:text-gray-600 dark:hover:text-gray-300 transition-colors z-10"
                        aria-label="Close popup"
                    >
                        <X className="w-6 h-6" />
                    </button>

                    {/* Discount Banner */}
                    <div className="bg-gradient-to-r from-amber-500 via-orange-500 to-red-500 text-white px-6 py-3 rounded-t-2xl flex-shrink-0">
                        <div className="flex items-center justify-center gap-2 text-center">
                            <Sparkles className="w-5 h-5 animate-pulse" />
                            <p className="font-bold text-lg">
                                🎉 FIRST-TIME CLIENT SPECIAL: <span className="text-2xl">25% OFF</span> Your First Order!
                            </p>
                            <Sparkles className="w-5 h-5 animate-pulse" />
                        </div>
                    </div>

                    {/* Main Content */}
                    <div className="flex-1 overflow-hidden p-6 md:p-8 space-y-6 flex flex-col">
                        {/* Headline */}
                        <div className="text-center flex-shrink-0">
                            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 dark:text-white mb-3">
                                Wait! Before You Go...
                            </h2>
                        </div>

                        {/* Service Description */}
                        <div className="p-5 bg-gradient-to-br from-indigo-50 to-purple-50 dark:from-gray-700 dark:to-gray-700 rounded-xl border border-indigo-100 dark:border-indigo-500 flex-shrink-0">
                            <p className="text-gray-800 dark:text-gray-100 text-center leading-relaxed text-lg font-semibold mb-2">
                                 Let our writers complete the assignment for you.
                            </p>
                            <p className="text-gray-700 dark:text-gray-200 text-center leading-relaxed">
                                <span className="font-bold text-indigo-600 dark:text-indigo-300">100% Original Work Guaranteed!</span> 
                                <br className="hidden sm:block" />
                                <span className="text-sm"> Deadlines stressing you out? That 10-page paper due tomorrow making you question your life choices? 😅 We've got your back! Our expert writers handle the hard stuff so you can actually sleep. No more all-nighters, no more caffeine overdoses, just quality work delivered on time. Because let's be real—some assignments are just too brutal to tackle alone!</span>
                            </p>
                        </div>

                        {/* CTAs */}
                        <div className="flex flex-col sm:flex-row gap-3 flex-shrink-0">
                            <Link
                                href="/place-order"
                                onClick={handleCtaClick}
                                className="flex-1 group relative overflow-hidden bg-gradient-to-r from-indigo-600 to-purple-600 hover:from-indigo-700 hover:to-purple-700 text-white font-bold py-4 px-6 rounded-xl shadow-lg hover:shadow-xl transition-all duration-300 text-center"
                            >
                                <span className="relative z-10 flex items-center justify-center gap-2">
                                    <Sparkles className="w-5 h-5" />
                                    Order Now - Get 25% Off
                                    <Sparkles className="w-5 h-5" />
                                </span>
                                <div className="absolute inset-0 bg-white/20 transform -skew-x-12 -translate-x-full group-hover:translate-x-full transition-transform duration-700" />
                            </Link>

                            <Link
                                href="/contact"
                                onClick={handleCtaClick}
                                className="flex-1 bg-white dark:bg-gray-700 border-2 border-indigo-600 dark:border-indigo-400 text-indigo-600 dark:text-indigo-400 font-semibold py-4 px-6 rounded-xl hover:bg-indigo-50 dark:hover:bg-gray-600 transition-all duration-300 text-center shadow-md hover:shadow-lg"
                            >
                                Contact Us
                            </Link>
                        </div>

                        {/* Trust Badge */}
                        <p className="text-center text-xs text-gray-500 dark:text-gray-300 flex-shrink-0">
                            🔒 Secure & Confidential • Money-Back Guarantee • No Hidden Fees
                        </p>
                    </div>
                </div>
            </div>
        </>
    );
}

