import { useState, useCallback } from "react";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import {
    faFileWord,
    faFileExcel,
    faFilePowerpoint,
    faFilePdf,
    faFileArchive,
    faUpload,
    faXmark,
    faCircleExclamation,
} from "@fortawesome/free-solid-svg-icons";
import { useDropzone } from "react-dropzone";
import axios from "axios";
import clsx from "clsx";

// File type configurations with icons
const ALLOWED_TYPES = {
    // Microsoft Word
    "application/msword": { ext: ".doc", icon: faFileWord },
    "application/vnd.openxmlformats-officedocument.wordprocessingml.document": {
        ext: ".docx",
        icon: faFileWord,
    },

    // Microsoft Excel
    "application/vnd.ms-excel": { ext: ".xls", icon: faFileExcel },
    "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet": {
        ext: ".xlsx",
        icon: faFileExcel,
    },

    // Microsoft PowerPoint
    "application/vnd.ms-powerpoint": { ext: ".ppt", icon: faFilePowerpoint },
    "application/vnd.openxmlformats-officedocument.presentationml.presentation":
        {
            ext: ".pptx",
            icon: faFilePowerpoint,
        },

    // PDF
    "application/pdf": { ext: ".pdf", icon: faFilePdf },

    // ZIP
    "application/zip": { ext: ".zip", icon: faFileArchive },
    "application/x-zip-compressed": { ext: ".zip", icon: faFileArchive },
};

// Maximum total file size allowed (40MB)
const MAX_FILE_SIZE = 40 * 1024 * 1024;

// Size of each chunk for upload (2MB chunks for efficient upload)
const CHUNK_SIZE = 2 * 1024 * 1024;

// Enable debug mode for detailed logging
const DEBUG = true;

export default function FileUpload({ orderId }) {
    const [selectedFiles, setSelectedFiles] = useState([]);
    const [uploadProgress, setUploadProgress] = useState({});
    const [error, setError] = useState("");
    const [isUploading, setIsUploading] = useState(false);

    const formatFileSize = (bytes) => {
        if (bytes === 0) return "0 Bytes";
        const k = 1024;
        const sizes = ["Bytes", "KB", "MB", "GB"];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return `${parseFloat((bytes / Math.pow(k, i)).toFixed(2))} ${sizes[i]}`;
    };

    const onDrop = useCallback((acceptedFiles) => {
        setError("");
        const validFiles = acceptedFiles.filter((file) => {
            // Validate file type
            if (!Object.keys(ALLOWED_TYPES).includes(file.type)) {
                setError(`${file.name} is not an allowed file type`);
                return false;
            }
            // Validate total file size against MAX_FILE_SIZE
            if (file.size > MAX_FILE_SIZE) {
                setError(`${file.name} exceeds 40MB limit`);
                return false;
            }
            return true;
        });
        setSelectedFiles((prev) => [...prev, ...validFiles]);
    }, []);

    const { getRootProps, getInputProps, isDragActive } = useDropzone({
        onDrop,
        accept: Object.keys(ALLOWED_TYPES).reduce((acc, curr) => {
            acc[curr] = [ALLOWED_TYPES[curr].ext];
            return acc;
        }, {}),
        maxSize: MAX_FILE_SIZE, // Use MAX_FILE_SIZE for dropzone limit
    });

    const uploadChunk = async (file, chunk, chunkIndex, totalChunks) => {
        const formData = new FormData();

        // Create a blob with the chunk data but without specifying a type
        // This prevents the browser from trying to determine the MIME type of the chunk
        const blob = new Blob([chunk]);

        formData.append("file", blob);
        formData.append("chunk", chunkIndex);
        formData.append("totalChunks", totalChunks);
        formData.append("filename", file.name);
        formData.append("fileType", file.type); // Pass the original file type
        formData.append("order_id", orderId);
        formData.append("totalFileSize", file.size);

        if (DEBUG) {
            console.log('Form data being sent:', {
                chunk: chunkIndex,
                totalChunks: totalChunks,
                filename: file.name,
                fileType: file.type,
                order_id: orderId,
                totalFileSize: file.size,
                chunkSize: chunk.size
            });
        }

        try {
            if (DEBUG) {
                console.log(
                    `Uploading chunk ${chunkIndex + 1}/${totalChunks} for ${
                        file.name
                    }: size ${chunk.size} bytes`
                );
            }

            const response = await axios.post(
                route("files.store-chunk"),
                formData,
                {
                    headers: {
                        "Content-Type": "multipart/form-data",
                        Accept: "application/json",
                    },
                    timeout: 120000, // 2 minutes timeout for larger chunks
                    onUploadProgress: (progressEvent) => {
                        const percentComplete =
                            (progressEvent.loaded / progressEvent.total) * 100;
                        setUploadProgress((prev) => ({
                            ...prev,
                            [file.name]: {
                                ...prev[file.name],
                                current:
                                    (chunkIndex / totalChunks) * 100 +
                                    percentComplete / totalChunks,
                            },
                        }));
                    },
                }
            );

            if (DEBUG) {
                console.log(
                    `Chunk ${chunkIndex + 1}/${totalChunks} upload complete:`,
                    response.data
                );
            }

            return response.data;
        } catch (error) {
            console.error(
                `Error uploading chunk ${chunkIndex + 1}/${totalChunks}:`,
                error
            );

            // Get detailed error information
            let errorMessage = "Upload failed";
            
            if (error.response?.data?.errors) {
                // Handle validation errors
                const errors = error.response.data.errors;
                if (errors.order_id) {
                    errorMessage = errors.order_id[0];
                } else if (errors.file) {
                    errorMessage = errors.file[0];
                } else if (errors.filename) {
                    errorMessage = errors.filename[0];
                } else if (errors.fileType) {
                    errorMessage = errors.fileType[0];
                } else {
                    errorMessage = Object.values(errors)[0][0];
                }
            } else if (error.response?.data?.message) {
                errorMessage = error.response.data.message;
            } else if (error.response?.data?.error) {
                errorMessage = error.response.data.error;
            }
            
            throw new Error(errorMessage);
        }
    };

    const handleUpload = async (e) => {
        e.preventDefault();
        setError("");

        if (!selectedFiles.length) {
            setError("Please select files to upload");
            return;
        }

        setIsUploading(true);
        try {
            for (const file of selectedFiles) {
                // Validate total file size again before upload
                if (file.size > MAX_FILE_SIZE) {
                    throw new Error(`${file.name} exceeds 40MB limit`);
                }

                // Calculate number of chunks needed
                const chunks = Math.ceil(file.size / CHUNK_SIZE);

                if (DEBUG) {
                    console.log(
                        `Processing file: ${file.name}, size: ${formatFileSize(
                            file.size
                        )}, chunks: ${chunks}`
                    );
                }

                // Initialize progress tracking
                setUploadProgress((prev) => ({
                    ...prev,
                    [file.name]: { current: 0 },
                }));

                // Process each chunk
                for (let i = 0; i < chunks; i++) {
                    const start = i * CHUNK_SIZE;
                    const end = Math.min((i + 1) * CHUNK_SIZE, file.size);
                    const chunk = file.slice(start, end);

                    if (DEBUG) {
                        console.log(
                            `Uploading chunk ${i + 1}/${chunks} for ${
                                file.name
                            }: ${start}-${end} (${chunk.size} bytes)`
                        );
                    }

                    await uploadChunk(file, chunk, i, chunks);
                }
            }

            // Reset state after successful upload
            setSelectedFiles([]);
            setUploadProgress({});
            window.location.reload();
        } catch (error) {
            console.error("Upload error:", error);
            setError(error.message || "Failed to upload file");
        } finally {
            setIsUploading(false);
        }
    };

    const removeFile = (index) => {
        setSelectedFiles((prev) => prev.filter((_, i) => i !== index));
        setError("");
    };

    return (
        <div className="w-full space-y-4">
            {error && (
                <div className="p-4 bg-red-100 dark:bg-red-900/50 border border-red-400 dark:border-red-800 text-red-700 dark:text-red-300 rounded-lg flex items-start gap-2">
                    <FontAwesomeIcon
                        icon={faCircleExclamation}
                        className="w-5 h-5 mt-0.5 flex-shrink-0"
                    />
                    <p className="text-sm">{error}</p>
                </div>
            )}

            <div
                {...getRootProps()}
                className={clsx(
                    "border-2 border-dashed rounded-lg p-6 cursor-pointer transition-colors",
                    "bg-gray-50 dark:bg-gray-800 border-gray-300 dark:border-gray-700",
                    "hover:bg-gray-100 dark:hover:bg-gray-700",
                    isDragActive && "border-blue-500 dark:border-blue-400"
                )}
            >
                <input {...getInputProps()} />
                <div className="flex flex-col items-center justify-center text-gray-500 dark:text-gray-400">
                    <FontAwesomeIcon
                        icon={faUpload}
                        className="w-12 h-12 mb-3"
                    />
                    <p className="mb-2 text-sm font-semibold">
                        {isDragActive
                            ? "Drop files here"
                            : "Drag & drop files or click to browse"}
                    </p>
                    <p className="text-xs text-center">
                        Allowed files: DOC, DOCX, XLS, XLSX, PPT, PPTX, PDF, ZIP
                        <br />
                        Maximum file size: 40MB
                    </p>
                </div>
            </div>

            {selectedFiles.length > 0 && (
                <div className="space-y-4">
                    {selectedFiles.map((file, index) => {
                        const fileIcon =
                            ALLOWED_TYPES[file.type]?.icon || faFileWord;
                        const progress =
                            uploadProgress[file.name]?.current || 0;

                        return (
                            <div
                                key={index}
                                className="flex items-center gap-4 p-4 bg-white dark:bg-gray-800 rounded-lg border dark:border-gray-700"
                            >
                                <FontAwesomeIcon
                                    icon={fileIcon}
                                    className="w-8 h-8 text-gray-500 dark:text-gray-400"
                                />
                                <div className="flex-1 min-w-0">
                                    <p className="text-sm font-medium text-gray-900 dark:text-gray-100 truncate">
                                        {file.name}
                                    </p>
                                    <p className="text-xs text-gray-500 dark:text-gray-400">
                                        {formatFileSize(file.size)}
                                    </p>
                                    {progress > 0 && (
                                        <div className="w-full h-1 bg-gray-200 dark:bg-gray-700 rounded-full mt-2">
                                            <div
                                                className="h-full bg-blue-500 rounded-full transition-all duration-300"
                                                style={{
                                                    width: `${progress}%`,
                                                }}
                                            />
                                        </div>
                                    )}
                                </div>
                                <button
                                    type="button"
                                    onClick={() => removeFile(index)}
                                    className="p-1 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-full"
                                    disabled={isUploading}
                                >
                                    <FontAwesomeIcon
                                        icon={faXmark}
                                        className="w-5 h-5 text-gray-500 dark:text-gray-400"
                                    />
                                </button>
                            </div>
                        );
                    })}

                    <button
                        type="button"
                        onClick={handleUpload}
                        disabled={isUploading}
                        className={clsx(
                            "w-full px-4 py-2 text-sm font-medium rounded-lg transition-colors",
                            "text-white bg-blue-500 hover:bg-blue-600",
                            "dark:bg-blue-600 dark:hover:bg-blue-700",
                            "disabled:opacity-50 disabled:cursor-not-allowed"
                        )}
                    >
                        {isUploading ? "Uploading..." : "Upload Files"}
                    </button>
                </div>
            )}
        </div>
    );
}
