import React from "react";
import { MessageSquare, Clock, User, Plus, CheckCircle, Archive, AlertTriangle } from "lucide-react";

export default function ConversationList({ conversations, onSelectConversation, isLoading, error, currentUser, onNewConversation }) {
    if (isLoading) {
        return (
            <div className="flex-1 flex items-center justify-center">
                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-cyan-600"></div>
            </div>
        );
    }

    if (error) {
        return (
            <div className="flex-1 flex items-center justify-center text-red-500">
                <p>{error}</p>
            </div>
        );
    }

    const formatTime = (timestamp) => {
        const date = new Date(timestamp);
        const now = new Date();
        const diffInHours = (now - date) / (1000 * 60 * 60);

        if (diffInHours < 24) {
            return date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
        } else if (diffInHours < 48) {
            return 'Yesterday';
        } else {
            return date.toLocaleDateString();
        }
    };

    const getOtherParticipant = (conversation) => {
        return conversation.participants.find(p => p.id !== currentUser.id);
    };

    const getDisplayName = (participant) => {
        if (!participant) return 'Support';
        
        // If current user is client or writer, show admin/super_admin as "Support"
        if ((currentUser.user_type === 'client' || currentUser.user_type === 'writer') && 
            (participant.user_type === 'admin' || participant.user_type === 'super_admin')) {
            return 'Support';
        }
        
        // Otherwise show the actual name
        return participant.name || 'Support';
    };

    const getStatusIcon = (status) => {
        switch (status) {
            case 'resolved':
                return <CheckCircle className="w-4 h-4 text-gray-500" />;
            case 'archived':
                return <Archive className="w-4 h-4 text-gray-400" />;
            case 'escalated':
                return <AlertTriangle className="w-4 h-4 text-red-500" />;
            default:
                return null;
        }
    };

    const getStatusColor = (status) => {
        switch (status) {
            case 'active':
                return 'text-green-600';
            case 'resolved':
                return 'text-gray-500';
            case 'archived':
                return 'text-gray-400';
            case 'escalated':
                return 'text-red-500';
            default:
                return 'text-gray-500';
        }
    };

    return (
        <div className="flex-1 overflow-y-auto">
            <div className="p-4 border-b border-gray-200 dark:border-gray-700">
                <div className="flex items-center justify-between">
                    <h4 className="font-semibold text-gray-900 dark:text-white">Conversations</h4>
                    <button
                        onClick={onNewConversation}
                        className="p-2 bg-gradient-to-r from-cyan-600 to-indigo-800 hover:from-cyan-700 hover:to-indigo-900 text-white rounded-lg transition-colors"
                        title="New Conversation"
                    >
                        <Plus className="w-4 h-4" />
                    </button>
                </div>
            </div>
            
            {conversations.length === 0 ? (
                <div className="flex-1 flex flex-col items-center justify-center text-gray-500 p-4">
                    <MessageSquare className="w-12 h-12 mb-4 opacity-50" />
                    <p className="text-center">No conversations yet</p>
                    <p className="text-sm text-center mt-2">
                        Start a conversation with an admin to get help
                    </p>
                    <button
                        onClick={onNewConversation}
                        className="mt-4 px-4 py-2 bg-gradient-to-r from-cyan-600 to-indigo-800 hover:from-cyan-700 hover:to-indigo-900 text-white rounded-lg transition-colors flex items-center space-x-2"
                    >
                        <Plus className="w-4 h-4" />
                        <span>New Conversation</span>
                    </button>
                </div>
            ) : (
                <div className="divide-y divide-gray-200 dark:divide-gray-700">
                    {conversations.map((conversation) => {
                        const otherParticipant = getOtherParticipant(conversation);
                        const lastMessage = conversation.last_message;
                        
                        return (
                            <div
                                key={conversation.id}
                                onClick={() => onSelectConversation(conversation)}
                                className="p-4 hover:bg-gray-50 dark:hover:bg-gray-700 cursor-pointer transition-colors"
                            >
                                <div className="flex items-center space-x-3">
                                    <div className="flex-shrink-0">
                                        <div className="w-10 h-10 bg-gradient-to-r from-cyan-600 to-indigo-800 rounded-full flex items-center justify-center">
                                            <User className="w-5 h-5 text-white" />
                                        </div>
                                    </div>
                                    <div className="flex-1 min-w-0">
                                        <div className="flex items-center justify-between">
                                            <div className="flex items-center space-x-2">
                                                <p className="text-sm font-medium text-gray-900 dark:text-white truncate">
                                                    {getDisplayName(otherParticipant)}
                                                </p>
                                                {getStatusIcon(conversation.status)}
                                            </div>
                                            {lastMessage && (
                                                <span className="text-xs text-gray-500 dark:text-gray-400">
                                                    {formatTime(lastMessage.created_at)}
                                                </span>
                                            )}
                                        </div>
                                        {lastMessage && (
                                            <p className="text-sm text-gray-500 dark:text-gray-400 truncate mt-1">
                                                {lastMessage.sender_id === currentUser.id ? 'You: ' : ''}
                                                {lastMessage.content}
                                            </p>
                                        )}
                                        {conversation.unread_count > 0 && (
                                            <div className="flex items-center justify-between mt-1">
                                                <span className="inline-flex items-center px-2 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                                                    {conversation.unread_count} new
                                                </span>
                                            </div>
                                        )}
                                    </div>
                                </div>
                            </div>
                        );
                    })}
                </div>
            )}
        </div>
    );
} 