import React, { useState, useEffect, useRef, useCallback } from "react";
import { MessageSquare, X, Send, Paperclip, Minus, Maximize2, Plus, Users, CheckCircle, RotateCcw, Archive, AlertTriangle } from "lucide-react";
import { usePage } from "@inertiajs/react";
import axios from "axios";
import ConversationList from "./ConversationList";
import MessageList from "./MessageList";
import MessageInput from "./MessageInput";
import NewConversationModal from "./NewConversationModal";
import ConversationFilters from "./ConversationFilters";
import BulkActions from "./BulkActions";

export default function MessagingWidget() {
    const { auth } = usePage().props;
    const user = auth.user;
    
    const [isOpen, setIsOpen] = useState(false);
    const [isMinimized, setIsMinimized] = useState(false);
    const [currentConversation, setCurrentConversation] = useState(null);
    const [conversations, setConversations] = useState([]);
    const [messages, setMessages] = useState([]);
    const [unreadCount, setUnreadCount] = useState(0);
    const [isLoading, setIsLoading] = useState(false);
    const [error, setError] = useState(null);
    const [showNewConversationModal, setShowNewConversationModal] = useState(false);
    
    // Phase 2: Advanced filtering states
    const [currentFilter, setCurrentFilter] = useState('active');
    const [searchTerm, setSearchTerm] = useState('');
    const [dateRange, setDateRange] = useState(null);
    const [conversationStats, setConversationStats] = useState({});
    const [filteredConversations, setFilteredConversations] = useState([]);
    
    // Bulk selection states
    const [selectedConversations, setSelectedConversations] = useState([]);
    
    const widgetRef = useRef(null);
    const messagesEndRef = useRef(null);

    // Fetch conversations on mount
    useEffect(() => {
        if (isOpen) {
            fetchConversations();
        }
    }, [isOpen]);

    // Fetch messages when conversation changes
    useEffect(() => {
        if (currentConversation) {
            fetchMessages(currentConversation.id);
        }
    }, [currentConversation]);

    // Auto-scroll to bottom when new messages arrive
    useEffect(() => {
        scrollToBottom();
    }, [messages]);

    // Real-time messaging with Laravel Echo
    useEffect(() => {
        if (!window.Echo || !user) return;

        // Listen for new messages in current conversation
        if (currentConversation) {
            const channel = window.Echo.private(`conversation.${currentConversation.id}`);
            
            channel.listen('.message.sent', (e) => {
                // Add new message to the list
                setMessages(prev => [...prev, e.message]);
                
                // Update conversation's last message
                setConversations(prev => 
                    prev.map(conv => 
                        conv.id === currentConversation.id 
                            ? { ...conv, last_message: e.message }
                            : conv
                    )
                );

                // Update unread count
                if (e.message.sender_id !== user.id) {
                    setUnreadCount(prev => prev + 1);
                }
            });

            channel.listen('.message.read', (e) => {
                // Update message read status
                setMessages(prev => 
                    prev.map(msg => 
                        msg.id === e.message_id 
                            ? { ...msg, read_at: e.read_at }
                            : msg
                    )
                );
            });

            return () => {
                window.Echo.leave(`conversation.${currentConversation.id}`);
            };
        }
    }, [currentConversation?.id, user?.id]);
    


    // Poll for unread count (fallback)
    useEffect(() => {
        const interval = setInterval(() => {
            fetchUnreadCount();
        }, 10000); // Every 10 seconds

        return () => clearInterval(interval);
    }, []);

    const scrollToBottom = () => {
        messagesEndRef.current?.scrollIntoView({ behavior: "smooth" });
    };

    const fetchConversations = async (filterStatus = null, search = null, dateFilter = null) => {
        try {
            setIsLoading(true);
            
            // Build query parameters
            const params = new URLSearchParams();
            const status = filterStatus || currentFilter;
            
            if (status && status !== 'all') {
                params.append('status', status);
            }
            
            if (search || searchTerm) {
                params.append('search', search || searchTerm);
            }
            
            if (dateFilter || dateRange) {
                const range = dateFilter || dateRange;
                if (range?.from) params.append('from_date', range.from);
                if (range?.to) params.append('to_date', range.to);
            }
            
            const queryString = params.toString();
            const url = queryString ? `/messaging/conversations?${queryString}` : '/messaging/conversations';
            
            const response = await axios.get(url);
            setConversations(response.data.conversations);
            
            // Update conversation stats for filter tabs
            if (response.data.stats) {
                setConversationStats(response.data.stats);
            }
            
        } catch (error) {
            console.error('Error fetching conversations:', error);
            setError('Failed to load conversations');
        } finally {
            setIsLoading(false);
        }
    };

    const fetchMessages = async (conversationId, reset = true) => {
        try {
            const response = await axios.get(route('messaging.messages', { conversation: conversationId }));
            if (reset) {
                setMessages(response.data.messages);
            } else {
                // Only add new messages
                const newMessages = response.data.messages.filter(
                    newMsg => !messages.some(existingMsg => existingMsg.id === newMsg.id)
                );
                if (newMessages.length > 0) {
                    setMessages(prev => [...prev, ...newMessages]);
                }
            }
        } catch (error) {
            console.error('Error fetching messages:', error);
        }
    };

    const fetchUnreadCount = async () => {
        try {
            const response = await axios.get(route('messaging.unread-count'));
            setUnreadCount(response.data.count);
        } catch (error) {
            console.error('Error fetching unread count:', error);
        }
    };

    const sendMessage = async (content, file = null) => {
        if (!currentConversation || !content.trim()) return;

        try {
            const formData = new FormData();
            formData.append('content', content);
            if (file) {
                formData.append('file', file);
            }

            const response = await axios.post(
                route('messaging.send-message', { conversation: currentConversation.id }),
                formData,
                {
                    headers: {
                        'Content-Type': 'multipart/form-data',
                    },
                }
            );

            // Add the new message to the list immediately for instant feedback
            setMessages(prev => [...prev, response.data.message]);
            
            // Update conversation's last message
            setConversations(prev => 
                prev.map(conv => 
                    conv.id === currentConversation.id 
                        ? { ...conv, last_message: response.data.message }
                        : conv
                )
            );

        } catch (error) {
            console.error('Error sending message:', error);
            setError('Failed to send message');
        }
    };

    const handleConversationSelect = (conversation) => {
        setCurrentConversation(conversation);
        setError(null);
    };

    const handleNewConversation = async (participantId, conversationType) => {
        try {
            // Ensure CSRF token is set
            const token = document.head.querySelector('meta[name="csrf-token"]');
            if (token) {
                // axios.defaults.headers.common['X-CSRF-TOKEN'] = token.content;
            }

            const response = await axios.post(route('messaging.create-conversation'), {
                participant_id: participantId,
                conversation_type: conversationType,
            }, {
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                }
            });

            if (response.data.conversation) {
                // Add the new conversation to the list
                setConversations(prev => [response.data.conversation, ...prev]);
                // Select the new conversation
                setCurrentConversation(response.data.conversation);
                setShowNewConversationModal(false);
            }
        } catch (error) {
            console.error('Error creating conversation:', error);
            if (error.response) {
                console.error('Error response:', error.response.data);
                setError(`Failed to create conversation: ${error.response.data.message || error.response.data.error || 'Unknown error'}`);
            } else {
                setError('Failed to create conversation');
            }
        }
    };

    const toggleWidget = () => {
        setIsOpen(!isOpen);
        if (!isOpen) {
            setIsMinimized(false);
        }
    };

    const toggleMinimize = () => {
        setIsMinimized(!isMinimized);
    };

    const closeWidget = () => {
        setIsOpen(false);
        setIsMinimized(false);
        setCurrentConversation(null);
        setShowNewConversationModal(false);
    };

    const goBackToConversations = () => {
        setCurrentConversation(null);
        setMessages([]);
    };

    // Get display name for participants based on user roles
    const getDisplayName = (participant) => {
        if (!participant) return 'Support';
        
        // If current user is client or writer, show admin/super_admin as "Support"
        if ((user.user_type === 'client' || user.user_type === 'writer') && 
            (participant.user_type === 'admin' || participant.user_type === 'super_admin')) {
            return 'Support';
        }
        
        // Otherwise show the actual name
        return participant.name || 'Support';
    };

    // Status management functions
    const handleResolveConversation = async () => {
        if (!currentConversation) return;
        
        try {
            await axios.patch(`/messaging/conversations/${currentConversation.id}/resolve`);
            
            // Update conversation status locally
            setCurrentConversation(prev => ({
                ...prev,
                status: 'resolved',
                resolved_at: new Date().toISOString(),
                resolved_by: user.id
            }));
            
            // Refresh conversations list
            fetchConversations();
        } catch (error) {
            console.error('Error resolving conversation:', error);
            alert('Failed to resolve conversation');
        }
    };

    const handleReopenConversation = async () => {
        if (!currentConversation) return;
        
        try {
            await axios.patch(`/messaging/conversations/${currentConversation.id}/reopen`);
            
            // Update conversation status locally
            setCurrentConversation(prev => ({
                ...prev,
                status: 'active',
                resolved_at: null,
                resolved_by: null
            }));
            
            // Refresh conversations list
            fetchConversations();
        } catch (error) {
            console.error('Error reopening conversation:', error);
            alert('Failed to reopen conversation');
        }
    };

    const handleArchiveConversation = async () => {
        if (!currentConversation) return;
        
        const confirmed = confirm('Are you sure you want to archive this conversation? This action cannot be undone.');
        if (!confirmed) return;
        
        try {
            await axios.patch(`/messaging/conversations/${currentConversation.id}/archive`);
            
            // Close the conversation and refresh list
            setCurrentConversation(null);
            setMessages([]);
            fetchConversations();
        } catch (error) {
            console.error('Error archiving conversation:', error);
            alert('Failed to archive conversation');
        }
    };

    const handleEscalateConversation = async () => {
        if (!currentConversation) return;
        
        try {
            await axios.patch(`/messaging/conversations/${currentConversation.id}/escalate`, {
                priority_level: 5
            });
            
            // Update conversation status locally
            setCurrentConversation(prev => ({
                ...prev,
                status: 'escalated',
                priority_level: 5
            }));
            
            // Refresh conversations list
            fetchConversations();
        } catch (error) {
            console.error('Error escalating conversation:', error);
            alert('Failed to escalate conversation');
        }
    };

    // Check if user can perform status actions
    const canResolveConversation = () => {
        return currentConversation && 
               currentConversation.status === 'active' && 
               currentConversation.participants.some(p => p.id === user.id);
    };

    const canReopenConversation = () => {
        if (!currentConversation || currentConversation.status !== 'resolved') return false;
        
        // Admins can always reopen
        if (user.user_type === 'admin' || user.user_type === 'super_admin') return true;
        
        // Clients/writers can reopen within 48 hours
        if (currentConversation.resolved_at) {
            const resolvedAt = new Date(currentConversation.resolved_at);
            const hoursSinceResolved = (new Date() - resolvedAt) / (1000 * 60 * 60);
            return hoursSinceResolved <= 48 && currentConversation.participants.some(p => p.id === user.id);
        }
        
        return false;
    };

    const canArchiveConversation = () => {
        return currentConversation && 
               (user.user_type === 'admin' || user.user_type === 'super_admin');
    };

    const canEscalateConversation = () => {
        return currentConversation && 
               currentConversation.status !== 'escalated' &&
               (user.user_type === 'admin' || user.user_type === 'super_admin');
    };

    // Phase 2: Filter handling functions
    const handleFilterChange = (newFilter, newDateRange = null) => {
        setCurrentFilter(newFilter);
        if (newDateRange !== undefined) {
            setDateRange(newDateRange);
        }
        fetchConversations(newFilter, searchTerm, newDateRange);
    };

    const handleSearchChange = (newSearchTerm) => {
        setSearchTerm(newSearchTerm);
        // Debounce search to avoid too many API calls
        const debounceTimeout = setTimeout(() => {
            fetchConversations(currentFilter, newSearchTerm, dateRange);
        }, 300);
        
        return () => clearTimeout(debounceTimeout);
    };

    // Determine if user is admin for advanced features
    const isAdmin = user.user_type === 'admin' || user.user_type === 'super_admin';

    // Handle bulk actions completion
    const handleBulkComplete = () => {
        fetchConversations();
        setSelectedConversations([]);
    };

    // Don't render if user is not authenticated
    if (!user) return null;

    return (
        <>
            {/* Floating Chat Button */}
            <div className="fixed bottom-4 right-4 z-50">
                <button
                    onClick={toggleWidget}
                    className="relative bg-gradient-to-r from-cyan-600 to-indigo-800 hover:from-cyan-700 hover:to-indigo-900 text-white p-4 rounded-full shadow-lg hover:shadow-xl transition-all duration-200 group"
                >
                    <MessageSquare className="w-6 h-6" />
                    {unreadCount > 0 && (
                        <span className="absolute -top-2 -right-2 bg-red-500 text-white text-xs rounded-full h-6 w-6 flex items-center justify-center animate-pulse">
                            {unreadCount > 99 ? '99+' : unreadCount}
                        </span>
                    )}
                    <div className="absolute bottom-full right-0 mb-2 px-2 py-1 bg-gray-800 text-white text-xs rounded opacity-0 group-hover:opacity-100 transition-opacity">
                        Messages
                    </div>
                </button>
            </div>

            {/* Messaging Widget */}
            {isOpen && (
                <div className={`fixed bottom-4 right-4 w-96 ${isMinimized ? 'h-auto' : 'h-[500px]'} bg-white dark:bg-gray-800 rounded-lg shadow-2xl border border-gray-200 dark:border-gray-700 z-50`}>
                    {/* Header */}
                    <div className={`flex items-center justify-between p-4 ${!isMinimized ? 'border-b border-gray-200 dark:border-gray-700' : ''} bg-gradient-to-r from-cyan-600 to-indigo-800 text-white ${isMinimized ? 'rounded-lg' : 'rounded-t-lg'}`}>
                        <div className="flex items-center space-x-2">
                            {currentConversation && (
                                <button
                                    onClick={goBackToConversations}
                                    className="p-1 hover:bg-white/20 rounded transition-colors"
                                >
                                    <X className="w-4 h-4" />
                                </button>
                            )}
                            <MessageSquare className="w-5 h-5" />
                            <h3 className="font-semibold">
                                {currentConversation 
                                    ? `Chat with ${getDisplayName(currentConversation.participants.find(p => p.id !== user.id))}` 
                                    : 'Messages'
                                }
                            </h3>
                        </div>
                        <div className="flex items-center space-x-2">
                            {!currentConversation && (
                                <button
                                    onClick={() => setShowNewConversationModal(true)}
                                    className="p-1 hover:bg-white/20 rounded transition-colors"
                                    title="New Conversation"
                                >
                                    <Plus className="w-4 h-4" />
                                </button>
                            )}
                            
                            {/* Status action buttons for current conversation */}
                            {currentConversation && (
                                <>
                                    {canResolveConversation() && (
                                        <button
                                            onClick={handleResolveConversation}
                                            className="p-1 hover:bg-white/20 rounded transition-colors"
                                            title="Mark as Resolved"
                                        >
                                            <CheckCircle className="w-4 h-4" />
                                        </button>
                                    )}
                                    
                                    {canReopenConversation() && (
                                        <button
                                            onClick={handleReopenConversation}
                                            className="p-1 hover:bg-white/20 rounded transition-colors"
                                            title="Reopen Conversation"
                                        >
                                            <RotateCcw className="w-4 h-4" />
                                        </button>
                                    )}
                                    
                                    {canEscalateConversation() && (
                                        <button
                                            onClick={handleEscalateConversation}
                                            className="p-1 hover:bg-white/20 rounded transition-colors"
                                            title="Escalate Conversation"
                                        >
                                            <AlertTriangle className="w-4 h-4" />
                                        </button>
                                    )}
                                    
                                    {canArchiveConversation() && (
                                        <button
                                            onClick={handleArchiveConversation}
                                            className="p-1 hover:bg-white/20 rounded transition-colors"
                                            title="Archive Conversation"
                                        >
                                            <Archive className="w-4 h-4" />
                                        </button>
                                    )}
                                </>
                            )}
                            
                            <button
                                onClick={toggleMinimize}
                                className="p-1 hover:bg-white/20 rounded transition-colors"
                                title={isMinimized ? "Expand" : "Minimize"}
                            >
                                {isMinimized ? <Maximize2 className="w-4 h-4" /> : <Minus className="w-4 h-4" />}
                            </button>
                            <button
                                onClick={closeWidget}
                                className="p-1 hover:bg-white/20 rounded transition-colors"
                            >
                                <X className="w-4 h-4" />
                            </button>
                        </div>
                    </div>

                    {!isMinimized && (
                        <div className="flex flex-col h-[calc(500px-64px)]">
                            {!currentConversation ? (
                                // Conversation List with Filters
                                <>
                                    <ConversationFilters
                                        currentFilter={currentFilter}
                                        onFilterChange={handleFilterChange}
                                        onSearchChange={handleSearchChange}
                                        searchTerm={searchTerm}
                                        isAdmin={isAdmin}
                                        conversationStats={conversationStats}
                                    />
                                    {isAdmin && (
                                        <BulkActions
                                            conversations={conversations}
                                            isAdmin={isAdmin}
                                            onBulkComplete={handleBulkComplete}
                                            selectedConversations={selectedConversations}
                                            setSelectedConversations={setSelectedConversations}
                                        />
                                    )}
                                    <ConversationList
                                        conversations={conversations}
                                        onSelectConversation={handleConversationSelect}
                                        isLoading={isLoading}
                                        error={error}
                                        currentUser={user}
                                        onNewConversation={() => setShowNewConversationModal(true)}
                                        selectedConversations={selectedConversations}
                                        setSelectedConversations={setSelectedConversations}
                                        isAdmin={isAdmin}
                                    />
                                </>
                            ) : (
                                // Message List and Input
                                <>
                                    <MessageList
                                        messages={messages}
                                        currentUser={user}
                                        messagesEndRef={messagesEndRef}
                                    />
                                    <MessageInput
                                        onSendMessage={sendMessage}
                                        disabled={isLoading}
                                    />
                                </>
                            )}
                        </div>
                    )}
                </div>
            )}

            {/* New Conversation Modal */}
            {showNewConversationModal && (
                <NewConversationModal
                    currentUser={user}
                    onClose={() => setShowNewConversationModal(false)}
                    onConversationCreated={handleNewConversation}
                />
            )}
        </>
    );
} 