import React, { useState, useEffect } from "react";
import { X, Search, User, Users } from "lucide-react";
import axios from "axios";

export default function NewConversationModal({ currentUser, onClose, onConversationCreated }) {
    const [searchQuery, setSearchQuery] = useState("");
    const [users, setUsers] = useState([]);
    const [filteredUsers, setFilteredUsers] = useState([]);
    const [isLoading, setIsLoading] = useState(false);
    const [error, setError] = useState(null);

    const getDisplayName = (user) => {
        // If current user is admin/super_admin, show actual names
        if (currentUser.user_type === 'admin' || currentUser.user_type === 'super_admin') {
            return user.name;
        }
        
        // If the user is admin/super_admin, show "Support" to clients/writers
        if (user.user_type === 'admin' || user.user_type === 'super_admin') {
            return 'Support';
        }
        
        // Otherwise show actual name
        return user.name;
    };

    const getDisplayEmail = (user) => {
        // If current user is admin/super_admin, show actual emails
        if (currentUser.user_type === 'admin' || currentUser.user_type === 'super_admin') {
            return user.email;
        }
        
        // If the user is admin/super_admin, show support email to clients/writers
        if (user.user_type === 'admin' || user.user_type === 'super_admin') {
            return 'support@academicscribe.com';
        }
        
        // Otherwise show actual email
        return user.email;
    };

    useEffect(() => {
        fetchUsers();
    }, []);

    useEffect(() => {
        if (searchQuery.trim()) {
            const filtered = users.filter(user => {
                const displayName = getDisplayName(user);
                const displayEmail = getDisplayEmail(user);
                return displayName.toLowerCase().includes(searchQuery.toLowerCase()) ||
                       displayEmail.toLowerCase().includes(searchQuery.toLowerCase()) ||
                       user.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
                       user.email.toLowerCase().includes(searchQuery.toLowerCase());
            });
            setFilteredUsers(filtered);
        } else {
            setFilteredUsers(users);
        }
    }, [searchQuery, users]);

    const fetchUsers = async () => {
        try {
            setIsLoading(true);
            const response = await axios.get(route('messaging.available-users'));
            console.log('Available users response:', response.data);
            setUsers(response.data.users);
            setFilteredUsers(response.data.users);
        } catch (error) {
            console.error('Error fetching users:', error);
            setError('Failed to load users');
        } finally {
            setIsLoading(false);
        }
    };

    const handleUserSelect = (user) => {
        // Determine conversation type based on current user and selected user
        let conversationType = 'client_admin';
        
        if (currentUser.user_type === 'admin' || currentUser.user_type === 'super_admin') {
            if (user.user_type === 'client') {
                conversationType = 'client_admin';
            } else if (user.user_type === 'writer') {
                conversationType = 'writer_admin';
            } else if (user.user_type === 'admin' || user.user_type === 'super_admin') {
                conversationType = 'admin_admin';
            }
        } else if (currentUser.user_type === 'client') {
            if (user.user_type === 'admin' || user.user_type === 'super_admin') {
                conversationType = 'client_admin';
            }
        } else if (currentUser.user_type === 'writer') {
            if (user.user_type === 'admin' || user.user_type === 'super_admin') {
                conversationType = 'writer_admin';
            }
        }

        onConversationCreated(user.id, conversationType);
    };

    const getConversationTypeLabel = (user) => {
        if (currentUser.user_type === 'admin' || currentUser.user_type === 'super_admin') {
            if (user.user_type === 'client') {
                return 'Client';
            } else if (user.user_type === 'writer') {
                return 'Writer';
            } else if (user.user_type === 'admin' || user.user_type === 'super_admin') {
                return user.user_type === 'super_admin' ? 'Super Admin' : 'Admin';
            }
        } else if (currentUser.user_type === 'client') {
            if (user.user_type === 'admin' || user.user_type === 'super_admin') {
                return user.user_type === 'super_admin' ? 'Super Admin' : 'Admin';
            }
        } else if (currentUser.user_type === 'writer') {
            if (user.user_type === 'admin' || user.user_type === 'super_admin') {
                return user.user_type === 'super_admin' ? 'Super Admin' : 'Admin';
            }
        }
        return user.user_type === 'super_admin' ? 'Super Admin' : user.user_type;
    };

    const canMessageUser = (user) => {
        // Users can't message themselves
        if (user.id === currentUser.id) return false;

        console.log('Checking if can message user:', {
            currentUser: { id: currentUser.id, type: currentUser.user_type },
            targetUser: { id: user.id, type: user.user_type }
        });

        // Clients can only message admins and super_admins
        if (currentUser.user_type === 'client') {
            const canMessage = user.user_type === 'admin' || user.user_type === 'super_admin';
            console.log('Client can message:', canMessage);
            return canMessage;
        }

        // Writers can only message admins and super_admins
        if (currentUser.user_type === 'writer') {
            const canMessage = user.user_type === 'admin' || user.user_type === 'super_admin';
            console.log('Writer can message:', canMessage);
            return canMessage;
        }

        // Admins and super_admins can message anyone
        if (currentUser.user_type === 'admin' || currentUser.user_type === 'super_admin') {
            console.log('Admin/Super Admin can message: true');
            return true;
        }

        console.log('Default case: false');
        return false;
    };

    return (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white dark:bg-gray-800 rounded-lg shadow-xl w-96 max-h-[80vh] overflow-hidden">
                {/* Header */}
                <div className="flex items-center justify-between p-4 border-b border-gray-200 dark:border-gray-700 bg-gradient-to-r from-cyan-600 to-indigo-800 text-white">
                    <h3 className="font-semibold">New Conversation</h3>
                    <button
                        onClick={onClose}
                        className="p-1 hover:bg-white/20 rounded transition-colors"
                    >
                        <X className="w-4 h-4" />
                    </button>
                </div>

                {/* Search */}
                <div className="p-4 border-b border-gray-200 dark:border-gray-700">
                    <div className="relative">
                        <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                        <input
                            type="text"
                            placeholder="Search users..."
                            value={searchQuery}
                            onChange={(e) => setSearchQuery(e.target.value)}
                            className="w-full pl-10 pr-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-cyan-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                        />
                    </div>
                </div>

                {/* Users List */}
                <div className="flex-1 overflow-y-auto max-h-96">
                    {isLoading ? (
                        <div className="flex items-center justify-center p-8">
                            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-cyan-600"></div>
                        </div>
                    ) : error ? (
                        <div className="flex items-center justify-center p-8 text-red-500">
                            <p>{error}</p>
                        </div>
                    ) : filteredUsers.length === 0 ? (
                        <div className="flex flex-col items-center justify-center p-8 text-gray-500">
                            <Users className="w-12 h-12 mb-4 opacity-50" />
                            <p className="text-center">No users found</p>
                            <p className="text-sm text-center mt-2">
                                {searchQuery ? 'Try a different search term' : 'No users available'}
                            </p>
                        </div>
                    ) : (
                        <div className="divide-y divide-gray-200 dark:divide-gray-700">
                            {filteredUsers
                                .filter(canMessageUser)
                                .map((user) => (
                                    <div
                                        key={user.id}
                                        onClick={() => handleUserSelect(user)}
                                        className="p-4 hover:bg-gray-50 dark:hover:bg-gray-700 cursor-pointer transition-colors"
                                    >
                                        <div className="flex items-center space-x-3">
                                            <div className="flex-shrink-0">
                                                <div className="w-10 h-10 bg-gradient-to-r from-cyan-600 to-indigo-800 rounded-full flex items-center justify-center">
                                                    <User className="w-5 h-5 text-white" />
                                                </div>
                                            </div>
                                            <div className="flex-1 min-w-0">
                                                <div className="flex items-center justify-between">
                                                    <p className="text-sm font-medium text-gray-900 dark:text-white truncate">
                                                        {getDisplayName(user)}
                                                    </p>
                                                    <span className="inline-flex items-center px-2 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-200">
                                                        {getConversationTypeLabel(user)}
                                                    </span>
                                                </div>
                                                <p className="text-sm text-gray-500 dark:text-gray-400 truncate">
                                                    {getDisplayEmail(user)}
                                                </p>
                                            </div>
                                        </div>
                                    </div>
                                ))}
                        </div>
                    )}
                </div>

                {/* Footer */}
                <div className="p-4 border-t border-gray-200 dark:border-gray-700">
                    <button
                        onClick={onClose}
                        className="w-full px-4 py-2 bg-gray-300 dark:bg-gray-600 text-gray-700 dark:text-gray-300 rounded-lg hover:bg-gray-400 dark:hover:bg-gray-500 transition-colors"
                    >
                        Cancel
                    </button>
                </div>
            </div>
        </div>
    );
} 