import React, { useState, useEffect, useRef } from "react";
import NotificationItem from "./NotificationItem";
import {
    Bell,
    AlertCircle,
    Loader2,
    ArrowDown,
    RefreshCw,
    Filter,
    Search,
    Clock,
    Calendar,
} from "lucide-react";

export default function NotificationList({
    notifications = [],
    onUpdate,
    getRouteUrl,
    isLoading = false,
    error = null,
    onLoadMore = null,
    hasMore = false,
    onRefresh = null,
    onFilter = null,
    emptyMessage = "No notifications",
    filterOptions = null,
}) {
    const [localNotifications, setLocalNotifications] = useState([]);
    const [isLoadingMore, setIsLoadingMore] = useState(false);
    const [isRefreshing, setIsRefreshing] = useState(false);
    const listRef = useRef(null);
    const observerRef = useRef(null);
    const lastItemRef = useRef(null);

    // Initialize local state with provided notifications
    useEffect(() => {
        if (notifications?.length > 0) {
            setLocalNotifications(notifications);
        } else {
            setLocalNotifications([]);
        }
    }, [notifications]);

    // Set up intersection observer for infinite scrolling if onLoadMore is provided
    useEffect(() => {
        if (!onLoadMore || !hasMore) return;

        if (observerRef.current) {
            observerRef.current.disconnect();
        }

        const handleObserver = (entries) => {
            const [entry] = entries;
            if (
                entry.isIntersecting &&
                hasMore &&
                !isLoadingMore &&
                !isLoading
            ) {
                handleLoadMore();
            }
        };

        const options = {
            root: null,
            rootMargin: "0px",
            threshold: 0.5,
        };

        observerRef.current = new IntersectionObserver(handleObserver, options);

        if (lastItemRef.current) {
            observerRef.current.observe(lastItemRef.current);
        }

        return () => {
            if (observerRef.current) {
                observerRef.current.disconnect();
            }
        };
    }, [onLoadMore, hasMore, isLoadingMore, isLoading, localNotifications]);

    // Handle loading more notifications
    const handleLoadMore = async () => {
        if (!onLoadMore || isLoadingMore || !hasMore) return;

        setIsLoadingMore(true);
        try {
            await onLoadMore();
        } catch (error) {
            console.error("Error loading more notifications:", error);
        } finally {
            setIsLoadingMore(false);
        }
    };

    // Handle refreshing notifications
    const handleRefresh = async () => {
        if (!onRefresh || isRefreshing) return;

        setIsRefreshing(true);
        try {
            await onRefresh();
        } catch (error) {
            console.error("Error refreshing notifications:", error);
        } finally {
            setIsRefreshing(false);
        }
    };

    // Handle notification update from child components
    const handleNotificationUpdate = (updatedNotification) => {
        if (!updatedNotification?.id) return;

        // Update the local state
        setLocalNotifications((prev) =>
            prev.map((notification) =>
                notification.id === updatedNotification.id
                    ? updatedNotification
                    : notification
            )
        );

        // Pass update to parent component
        if (typeof onUpdate === "function") {
            onUpdate(updatedNotification);
        }
    };

    // Scroll to top of list
    const scrollToTop = () => {
        if (listRef.current) {
            listRef.current.scrollTo({
                top: 0,
                behavior: "smooth",
            });
        }
    };

    // Render loading state
    if (isLoading && localNotifications.length === 0) {
        return (
            <div className="flex flex-col items-center justify-center p-8 text-center">
                <Loader2 className="h-8 w-8 text-primary animate-spin mb-3" />
                <p className="text-gray-500 dark:text-gray-400">
                    Loading notifications...
                </p>
            </div>
        );
    }

    // Render error state
    if (error && localNotifications.length === 0) {
        return (
            <div className="flex flex-col items-center justify-center p-8 text-center">
                <AlertCircle className="h-8 w-8 text-red-500 dark:text-red-400 mb-3" />
                <p className="text-red-600 dark:text-red-400 font-medium">
                    {error}
                </p>
                {onRefresh && (
                    <button
                        onClick={handleRefresh}
                        className="mt-4 px-4 py-2 bg-gray-100 dark:bg-gray-700 hover:bg-gray-200 dark:hover:bg-gray-600 rounded-md text-sm text-gray-700 dark:text-gray-300 flex items-center gap-2 transition-colors"
                        disabled={isRefreshing}
                    >
                        {isRefreshing ? (
                            <>
                                <Loader2 className="h-4 w-4 animate-spin" />
                                <span>Trying again...</span>
                            </>
                        ) : (
                            <>
                                <RefreshCw className="h-4 w-4" />
                                <span>Try again</span>
                            </>
                        )}
                    </button>
                )}
            </div>
        );
    }

    // Render empty state
    if (localNotifications.length === 0) {
        return (
            <div className="flex flex-col items-center justify-center p-8 text-center">
                <Bell className="h-10 w-10 text-gray-300 dark:text-gray-600 mb-3" />
                <p className="text-gray-500 dark:text-gray-400">
                    {emptyMessage}
                </p>
                {onRefresh && (
                    <button
                        onClick={handleRefresh}
                        className="mt-4 px-4 py-2 bg-gray-100 dark:bg-gray-700 hover:bg-gray-200 dark:hover:bg-gray-600 rounded-md text-sm text-gray-700 dark:text-gray-300 flex items-center gap-2 transition-colors"
                        disabled={isRefreshing}
                    >
                        {isRefreshing ? (
                            <>
                                <Loader2 className="h-4 w-4 animate-spin" />
                                <span>Refreshing...</span>
                            </>
                        ) : (
                            <>
                                <RefreshCw className="h-4 w-4" />
                                <span>Refresh</span>
                            </>
                        )}
                    </button>
                )}
            </div>
        );
    }

    // Render filter options if provided
    const renderFilterOptions = () => {
        if (!filterOptions || !onFilter) return null;

        return (
            <div className="sticky top-0 z-10 p-2 bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-800 border border-gray-300 dark:border-gray-600 flex flex-wrap gap-2 items-center">
                <div className="flex items-center gap-1 text-gray-700 dark:text-gray-300">
                    <Filter className="h-4 w-4" />
                    <span className="text-xs font-medium">Filters:</span>
                </div>

                {filterOptions.map((option, index) => (
                    <button
                        key={index}
                        onClick={() => onFilter(option.value)}
                        className={`px-2 py-1 rounded-full text-xs ${
                            option.active
                                ? "bg-blue-500 text-white dark:bg-blue-600"
                                : "bg-white dark:bg-gray-700 text-gray-700 dark:text-gray-300"
                        } flex items-center gap-1 transition-colors`}
                    >
                        {option.icon && <option.icon className="h-3 w-3" />}
                        <span>{option.label}</span>
                    </button>
                ))}
            </div>
        );
    };

    return (
        <div className="relative">
            {/* Filter options bar */}
            {renderFilterOptions()}

            {/* Notification list */}
            <div
                className="divide-y divide-gray-200 dark:divide-gray-700 overflow-y-auto max-h-[70vh]"
                ref={listRef}
            >
                {localNotifications.map((notification, index) => {
                    // Determine if this is the last item for infinite scroll
                    const isLastItem = index === localNotifications.length - 1;

                    return (
                        <div
                            key={notification.id || index}
                            ref={isLastItem ? lastItemRef : null}
                        >
                            <NotificationItem
                                notification={notification}
                                onUpdate={handleNotificationUpdate}
                                getRouteUrl={getRouteUrl}
                            />
                        </div>
                    );
                })}

                {/* Loading more indicator */}
                {isLoadingMore && (
                    <div className="flex justify-center items-center p-4 border-t border-gray-200 dark:border-gray-700">
                        <Loader2 className="h-5 w-5 text-primary animate-spin mr-2" />
                        <span className="text-sm text-gray-500 dark:text-gray-400">
                            Loading more...
                        </span>
                    </div>
                )}

                {/* "Load more" button (alternative to infinite scroll) */}
                {hasMore && !isLoadingMore && onLoadMore && (
                    <button
                        onClick={handleLoadMore}
                        className="w-full p-3 text-sm text-center text-gray-500 dark:text-gray-400 hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors flex items-center justify-center gap-2"
                    >
                        <ArrowDown className="h-4 w-4" />
                        <span>Load more</span>
                    </button>
                )}
            </div>

            {/* Scroll to top button - appears when scrolled down */}
            {localNotifications.length > 10 && (
                <button
                    onClick={scrollToTop}
                    className="absolute bottom-4 right-4 bg-white dark:bg-gray-700 shadow-md rounded-full p-2 text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-600 transition-colors"
                    title="Scroll to top"
                >
                    <ArrowDown className="h-4 w-4 transform rotate-180" />
                </button>
            )}

            {/* Global loading overlay */}
            {isLoading && localNotifications.length > 0 && (
                <div className="absolute inset-0 bg-white/60 dark:bg-gray-800/60 flex items-center justify-center">
                    <div className="bg-white dark:bg-gray-700 rounded-lg shadow-lg p-4 flex items-center gap-3">
                        <Loader2 className="h-5 w-5 text-primary animate-spin" />
                        <span className="text-gray-700 dark:text-gray-300">
                            Updating...
                        </span>
                    </div>
                </div>
            )}
        </div>
    );
}
