import React, { useState } from "react";
import { cn } from "@/lib/utils";

/**
 * OptimizedImage Component
 * Automatically serves WebP images with fallbacks for better performance
 */
const OptimizedImage = ({
    src,
    alt,
    className,
    width,
    height,
    loading = "lazy",
    decoding = "async",
    sizes,
    priority = false,
    objectFit = "cover", // New prop to control object-fit behavior
    onLoad,
    onError,
    ...props
}) => {
    const [imageError, setImageError] = useState(false);
    const [imageLoaded, setImageLoaded] = useState(false);

    // Generate WebP and fallback URLs
    const getOptimizedUrls = (originalSrc) => {
        if (!originalSrc) return { webp: null, fallback: originalSrc };

        const pathInfo = originalSrc.split(".");
        const extension = pathInfo.pop();
        const basePath = pathInfo.join(".");

        return {
            webp: `${basePath}.webp`,
            fallback: originalSrc,
        };
    };

    const { webp, fallback } = getOptimizedUrls(src);

    const handleLoad = (event) => {
        setImageLoaded(true);
        if (onLoad) onLoad(event);
    };

    const handleError = (event) => {
        setImageError(true);
        if (onError) onError(event);
    };

    // Determine object-fit class based on prop
    const getObjectFitClass = () => {
        switch (objectFit) {
            case "contain":
                return "object-contain";
            case "fill":
                return "object-fill";
            case "scale-down":
                return "object-scale-down";
            case "none":
                return "object-none";
            default:
                return "object-cover";
        }
    };

    // If there's an error or no WebP support, show fallback
    if (imageError || !webp) {
        return (
            <img
                src={fallback}
                alt={alt}
                className={cn(
                    "transition-opacity duration-300",
                    getObjectFitClass(),
                    imageLoaded ? "opacity-100" : "opacity-0",
                    className
                )}
                width={width}
                height={height}
                loading={priority ? "eager" : loading}
                decoding={decoding}
                sizes={sizes}
                onLoad={handleLoad}
                onError={handleError}
                {...props}
            />
        );
    }

    return (
        <picture className={cn("block", className)}>
            {/* WebP source for modern browsers */}
            <source srcSet={webp} type="image/webp" sizes={sizes} />

            {/* Fallback for browsers that don't support WebP */}
            <img
                src={fallback}
                alt={alt}
                className={cn(
                    "w-full h-full transition-opacity duration-300",
                    getObjectFitClass(),
                    imageLoaded ? "opacity-100" : "opacity-0"
                )}
                width={width}
                height={height}
                loading={priority ? "eager" : loading}
                decoding={decoding}
                sizes={sizes}
                onLoad={handleLoad}
                onError={handleError}
                {...props}
            />
        </picture>
    );
};

export default OptimizedImage;
