import React, { useState, useEffect } from "react";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import {
    faFileWord,
    faFileExcel,
    faFilePowerpoint,
    faFilePdf,
    faFileArchive,
    faDownload,
    faEye,
    faTrash,
    faInfoCircle,
    faClock,
    faUser,
    faTag,
} from "@fortawesome/free-solid-svg-icons";
import axios from "axios";
import clsx from "clsx";

// File type configurations with icons
const FILE_TYPE_ICONS = {
    "application/msword": faFileWord,
    "application/vnd.openxmlformats-officedocument.wordprocessingml.document": faFileWord,
    "application/vnd.ms-excel": faFileExcel,
    "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet": faFileExcel,
    "application/vnd.ms-powerpoint": faFilePowerpoint,
    "application/vnd.openxmlformats-officedocument.presentationml.presentation": faFilePowerpoint,
    "application/pdf": faFilePdf,
    "application/zip": faFileArchive,
    "application/x-zip-compressed": faFileArchive,
};

// File category labels and colors
const FILE_CATEGORIES = {
    sample: { label: 'Sample Work', color: 'bg-blue-100 text-blue-800' },
    instructions: { label: 'Order Instructions', color: 'bg-green-100 text-green-800' },
    draft: { label: 'Draft', color: 'bg-yellow-100 text-yellow-800' },
    complete: { label: 'Complete Work', color: 'bg-purple-100 text-purple-800' },
    admin: { label: 'Admin File', color: 'bg-gray-100 text-gray-800' },
    general: { label: 'General', color: 'bg-gray-100 text-gray-600' },
};

// Uploader type labels and colors
const UPLOADER_TYPES = {
    client: { label: 'Client', color: 'bg-blue-100 text-blue-800' },
    writer: { label: 'Writer', color: 'bg-green-100 text-green-800' },
    admin: { label: 'Admin', color: 'bg-purple-100 text-purple-800' },
};

export default function OrderFileDisplay({ orderId, userType, onFileDeleted }) {
    const [files, setFiles] = useState([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState("");
    const [selectedCategory, setSelectedCategory] = useState('all');

    useEffect(() => {
        fetchFiles();
    }, [orderId]);

    const fetchFiles = async () => {
        try {
            setLoading(true);
            const response = await axios.get(`/order-files/${orderId}`);
            
            if (response.data.success) {
                // Filter files based on user access
                const accessibleFiles = response.data.files.filter(file => 
                    file.canBeAccessedBy && file.canBeAccessedBy.includes(userType)
                );
                setFiles(accessibleFiles);
            } else {
                setError(response.data.message || 'Failed to fetch files');
            }
        } catch (error) {
            console.error('Error fetching files:', error);
            setError('Failed to fetch files');
        } finally {
            setLoading(false);
        }
    };

    const handleDownload = async (fileId, fileName) => {
        try {
            const response = await axios.get(`/files/${fileId}/download`, {
                responseType: 'blob'
            });

            // Create download link
            const url = window.URL.createObjectURL(new Blob([response.data]));
            const link = document.createElement('a');
            link.href = url;
            link.setAttribute('download', fileName);
            document.body.appendChild(link);
            link.click();
            link.remove();
            window.URL.revokeObjectURL(url);
        } catch (error) {
            console.error('Download error:', error);
            alert('Failed to download file');
        }
    };

    const handleDelete = async (fileId) => {
        if (!confirm('Are you sure you want to delete this file?')) {
            return;
        }

        try {
            const response = await axios.delete(`/files/${fileId}`);
            
            if (response.data.success) {
                // Remove file from state
                setFiles(prev => prev.filter(file => file.id !== fileId));
                
                // Call callback if provided
                if (onFileDeleted) {
                    onFileDeleted(fileId);
                }
            } else {
                alert(response.data.message || 'Failed to delete file');
            }
        } catch (error) {
            console.error('Delete error:', error);
            alert('Failed to delete file');
        }
    };

    const formatFileSize = (bytes) => {
        if (bytes === 0) return "0 Bytes";
        const k = 1024;
        const sizes = ["Bytes", "KB", "MB", "GB"];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return `${parseFloat((bytes / Math.pow(k, i)).toFixed(2))} ${sizes[i]}`;
    };

    const formatDate = (dateString) => {
        const date = new Date(dateString);
        return date.toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    const getFileIcon = (fileType) => {
        return FILE_TYPE_ICONS[fileType] || faFileArchive;
    };

    const getFilteredFiles = () => {
        if (selectedCategory === 'all') {
            return files;
        }
        return files.filter(file => file.file_category === selectedCategory);
    };

    const getCategoryCount = (category) => {
        if (category === 'all') {
            return files.length;
        }
        return files.filter(file => file.file_category === category).length;
    };

    if (loading) {
        return (
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                <div className="animate-pulse">
                    <div className="h-4 bg-gray-200 rounded w-1/4 mb-4"></div>
                    <div className="space-y-3">
                        <div className="h-20 bg-gray-200 rounded"></div>
                        <div className="h-20 bg-gray-200 rounded"></div>
                        <div className="h-20 bg-gray-200 rounded"></div>
                    </div>
                </div>
            </div>
        );
    }

    if (error) {
        return (
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                <div className="text-center text-red-600">
                    <FontAwesomeIcon icon={faInfoCircle} className="h-8 w-8 mb-2" />
                    <p>{error}</p>
                </div>
            </div>
        );
    }

    const filteredFiles = getFilteredFiles();

    return (
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <div className="mb-6">
                <h3 className="text-lg font-semibold text-gray-900 mb-2">
                    Order Files
                </h3>
                <p className="text-sm text-gray-600">
                    Files uploaded for this order. Access is controlled based on your role.
                </p>
            </div>

            {/* Category Filter */}
            <div className="mb-6">
                <div className="flex flex-wrap gap-2">
                    {Object.entries(FILE_CATEGORIES).map(([key, category]) => (
                        <button
                            key={key}
                            onClick={() => setSelectedCategory(key)}
                            className={clsx(
                                "px-3 py-1 text-sm font-medium rounded-full transition-colors",
                                selectedCategory === key
                                    ? "bg-blue-600 text-white"
                                    : "bg-gray-100 text-gray-700 hover:bg-gray-200"
                            )}
                        >
                            {category.label} ({getCategoryCount(key)})
                        </button>
                    ))}
                    <button
                        onClick={() => setSelectedCategory('all')}
                        className={clsx(
                            "px-3 py-1 text-sm font-medium rounded-full transition-colors",
                            selectedCategory === 'all'
                                ? "bg-blue-600 text-white"
                                : "bg-gray-100 text-gray-700 hover:bg-gray-200"
                        )}
                    >
                        All Files ({getCategoryCount('all')})
                    </button>
                </div>
            </div>

            {/* Files List */}
            {filteredFiles.length === 0 ? (
                <div className="text-center py-8 text-gray-500">
                    <FontAwesomeIcon icon={faInfoCircle} className="h-12 w-12 mb-4 text-gray-300" />
                    <p>No files found in this category.</p>
                </div>
            ) : (
                <div className="space-y-4">
                    {filteredFiles.map((file) => (
                        <div
                            key={file.id}
                            className="border border-gray-200 rounded-lg p-4 hover:shadow-md transition-shadow"
                        >
                            <div className="flex items-start justify-between">
                                <div className="flex items-start space-x-3 flex-1">
                                    {/* File Icon */}
                                    <div className="flex-shrink-0">
                                        <FontAwesomeIcon
                                            icon={getFileIcon(file.file_type)}
                                            className="h-8 w-8 text-gray-500"
                                        />
                                    </div>

                                    {/* File Info */}
                                    <div className="flex-1 min-w-0">
                                        <div className="flex items-center space-x-2 mb-2">
                                            <h4 className="text-sm font-medium text-gray-900 truncate">
                                                {file.file_name}
                                            </h4>
                                            
                                            {/* File Category Badge */}
                                            <span className={clsx(
                                                "inline-flex items-center px-2 py-1 rounded-full text-xs font-medium",
                                                FILE_CATEGORIES[file.file_category]?.color || FILE_CATEGORIES.general.color
                                            )}>
                                                {FILE_CATEGORIES[file.file_category]?.label || 'General'}
                                            </span>

                                            {/* Uploader Type Badge */}
                                            <span className={clsx(
                                                "inline-flex items-center px-2 py-1 rounded-full text-xs font-medium",
                                                UPLOADER_TYPES[file.uploader_user_type]?.color || 'bg-gray-100 text-gray-800'
                                            )}>
                                                {UPLOADER_TYPES[file.uploader_user_type]?.label || 'Unknown'}
                                            </span>
                                        </div>

                                        {/* File Details */}
                                        <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-xs text-gray-500">
                                            <div className="flex items-center space-x-1">
                                                <FontAwesomeIcon icon={faUser} className="h-3 w-3" />
                                                <span>{file.uploader_name}</span>
                                            </div>
                                            <div className="flex items-center space-x-1">
                                                <FontAwesomeIcon icon={faClock} className="h-3 w-3" />
                                                <span>{formatDate(file.date_uploaded)}</span>
                                            </div>
                                            <div className="flex items-center space-x-1">
                                                <FontAwesomeIcon icon={faTag} className="h-3 w-3" />
                                                <span>{formatFileSize(file.file_size)}</span>
                                            </div>
                                            <div className="flex items-center space-x-1">
                                                <FontAwesomeIcon icon={faInfoCircle} className="h-3 w-3" />
                                                <span className="truncate">
                                                    {file.description || 'No description'}
                                                </span>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                {/* Action Buttons */}
                                <div className="flex items-center space-x-2 ml-4">
                                    <button
                                        onClick={() => handleDownload(file.id, file.file_name)}
                                        className="p-2 text-blue-600 hover:text-blue-800 hover:bg-blue-50 rounded-md transition-colors"
                                        title="Download file"
                                    >
                                        <FontAwesomeIcon icon={faDownload} className="h-4 w-4" />
                                    </button>
                                    
                                    {/* Delete button - only show for files uploaded by current user or admin */}
                                    {(userType === 'admin' || 
                                      (userType === file.uploader_user_type && file.uploader_id === window.authUser?.id)) && (
                                        <button
                                            onClick={() => handleDelete(file.id)}
                                            className="p-2 text-red-600 hover:text-red-800 hover:bg-red-50 rounded-md transition-colors"
                                            title="Delete file"
                                        >
                                            <FontAwesomeIcon icon={faTrash} className="h-4 w-4" />
                                        </button>
                                    )}
                                </div>
                            </div>
                        </div>
                    ))}
                </div>
            )}
        </div>
    );
} 