import React, { useState, useEffect, useRef } from "react";
import {
    Search,
    DollarSign,
    Euro,
    PoundSterling,
    ChevronDown,
} from "lucide-react";
import { FormInput } from "@/Components/ui/FormInput";

const BasicInformation = ({
    data,
    setData,
    errors,
    academicLevels,
    subjects,
    paperTypes,
    currencies,
    orderCalculation,
    onCurrencyChange,
}) => {
    // Refs for dropdown positioning
    const subjectButtonRef = useRef(null);
    const paperTypeButtonRef = useRef(null);

    // States for dropdown open/close
    const [subjectDropdownOpen, setSubjectDropdownOpen] = useState(false);
    const [paperTypeDropdownOpen, setPaperTypeDropdownOpen] = useState(false);

    // States for dropdown positioning
    const [subjectDropdownStyle, setSubjectDropdownStyle] = useState({});
    const [paperTypeDropdownStyle, setPaperTypeDropdownStyle] = useState({});

    // States for searchable options
    const [subjectSearch, setSubjectSearch] = useState("");
    const [paperTypeSearch, setPaperTypeSearch] = useState("");

    // State for selected display names
    const [selectedSubjectName, setSelectedSubjectName] = useState("");
    const [selectedPaperTypeName, setSelectedPaperTypeName] = useState("");

    // Currency icons mapping with more visual variety
    const currencyIcons = {
        USD: <DollarSign className="w-5 h-5" />,
        EUR: <Euro className="w-5 h-5" />,
        GBP: <PoundSterling className="w-5 h-5" />,
        CAD: <DollarSign className="w-5 h-5" />,
        AUD: <DollarSign className="w-5 h-5" />,
        JPY: <span className="text-lg font-bold">¥</span>,
        CNY: <span className="text-lg font-bold">¥</span>,
        INR: <span className="text-lg font-bold">₹</span>,
        CHF: <span className="text-lg font-bold">CHF</span>,
    };

    // Function to calculate dropdown position
    const calculateDropdownPosition = (buttonRef, setDropdownStyle) => {
        if (!buttonRef.current) return;

        const buttonRect = buttonRef.current.getBoundingClientRect();
        const windowHeight = window.innerHeight;
        const spaceBelow = windowHeight - buttonRect.bottom;
        const spaceAbove = buttonRect.top;
        const dropdownHeight = Math.min(320, subjects?.length * 40 || 320); // Estimate dropdown height

        // Check if there's enough space below
        if (spaceBelow >= dropdownHeight || spaceBelow >= spaceAbove) {
            // Position below
            setDropdownStyle({
                position: "absolute",
                top: buttonRect.height + 4,
                width: "100%",
                maxHeight: `${Math.min(320, spaceBelow - 20)}px`,
                overflowY: "auto",
            });
        } else {
            // Position above
            setDropdownStyle({
                position: "absolute",
                bottom: buttonRect.height + 4,
                width: "100%",
                maxHeight: `${Math.min(320, spaceAbove - 20)}px`,
                overflowY: "auto",
            });
        }
    };

    // Update dropdown positions when they open
    useEffect(() => {
        if (subjectDropdownOpen) {
            calculateDropdownPosition(
                subjectButtonRef,
                setSubjectDropdownStyle
            );
        }
    }, [subjectDropdownOpen]);

    useEffect(() => {
        if (paperTypeDropdownOpen) {
            calculateDropdownPosition(
                paperTypeButtonRef,
                setPaperTypeDropdownStyle
            );
        }
    }, [paperTypeDropdownOpen]);

    // Update display names when data changes
    useEffect(() => {
        if (data.subject) {
            const parts = data.subject.split("#");
            if (parts.length > 1) {
                setSelectedSubjectName(parts[1]);
            }
        }

        if (data.papertype) {
            const parts = data.papertype.split("#");
            if (parts.length > 1) {
                setSelectedPaperTypeName(parts[1]);
            }
        }
    }, [data.subject, data.papertype]);

    // Recalculate dropdown positions on window resize
    useEffect(() => {
        const handleResize = () => {
            if (subjectDropdownOpen) {
                calculateDropdownPosition(
                    subjectButtonRef,
                    setSubjectDropdownStyle
                );
            }
            if (paperTypeDropdownOpen) {
                calculateDropdownPosition(
                    paperTypeButtonRef,
                    setPaperTypeDropdownStyle
                );
            }
        };

        window.addEventListener("resize", handleResize);
        return () => {
            window.removeEventListener("resize", handleResize);
        };
    }, [subjectDropdownOpen, paperTypeDropdownOpen]);

    // Filter subjects based on search
    const filteredSubjects =
        subjects?.filter((subject) =>
            subject.name.toLowerCase().includes(subjectSearch.toLowerCase())
        ) || [];

    // Filter paper types based on search
    const filteredPaperTypes =
        paperTypes?.filter((paperType) =>
            paperType.name.toLowerCase().includes(paperTypeSearch.toLowerCase())
        ) || [];

    // Handle academic level selection
    const handleAcademicLevelChange = (value) => {
        setData("aclevel", value);
    };

    // Handle subject selection
    const handleSubjectChange = (value, name) => {
        setData("subject", value);
        setSelectedSubjectName(name);
        setSubjectDropdownOpen(false);
    };

    // Handle paper type selection
    const handlePaperTypeChange = (value, name) => {
        setData("papertype", value);
        setSelectedPaperTypeName(name);
        setPaperTypeDropdownOpen(false);
    };

    // Handle currency selection - fixed implementation
    const handleCurrencySelection = (currency) => {
        if (!currency) return;

        // Update the currency in the data
        setData("currency", currency.currency_code);
        setData("currencyRate", parseFloat(currency.value) || 1);

        // Call the parent handler to update order calculation
        onCurrencyChange(
            currency.currency_code,
            parseFloat(currency.value) || 1
        );
    };

    // Set default currency if not set and currencies are available
    useEffect(() => {
        const activeCurrencies = currencies?.filter(c => c.is_active);
        if (activeCurrencies && activeCurrencies.length > 0 && !data.currency) {
            // Try to find USD as default, otherwise use the first active currency
            const defaultCurrency = activeCurrencies.find(c => c.currency_code === 'USD') || activeCurrencies[0];
            handleCurrencySelection(defaultCurrency);
        }
    }, [currencies]);

    // Click outside to close dropdowns
    useEffect(() => {
        const handleClickOutside = (event) => {
            if (
                subjectDropdownOpen &&
                !event.target.closest("#subject-dropdown")
            ) {
                setSubjectDropdownOpen(false);
            }
            if (
                paperTypeDropdownOpen &&
                !event.target.closest("#papertype-dropdown")
            ) {
                setPaperTypeDropdownOpen(false);
            }
        };

        document.addEventListener("mousedown", handleClickOutside);
        return () => {
            document.removeEventListener("mousedown", handleClickOutside);
        };
    }, [subjectDropdownOpen, paperTypeDropdownOpen]);

    return (
        <div className="space-y-8" aria-labelledby="step1-heading">
            <div className="space-y-6">
                {/* Title input */}
                <div>
                    <label
                        htmlFor="title"
                        className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-2"
                    >
                        Paper Topic/Title{" "}
                        <span className="text-red-500">*</span>
                    </label>
                    <input
                        type="text"
                        id="title"
                        name="title"
                        value={data.title || ""}
                        onChange={(e) => setData("title", e.target.value)}
                        placeholder="Enter the title of your paper"
                        className="w-full bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-lg p-3 focus:outline-none focus:ring-2 focus:ring-blue-500 dark:text-gray-200"
                        required
                    />
                    {errors.title && (
                        <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                            {errors.title}
                        </p>
                    )}
                </div>

                {/* Academic Level selection - pill style */}
                <div>
                    <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-3">
                        Academic Level <span className="text-red-500">*</span>
                    </label>
                    <div
                        className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 gap-3"
                        role="radiogroup"
                        aria-required="true"
                        aria-label="Select academic level"
                        required
                    >
                        {academicLevels?.map((level) => (
                            <div
                                key={level.id}
                                onClick={() =>
                                    handleAcademicLevelChange(
                                        `${level.value}#${level.name}`
                                    )
                                }
                                className={`
                                    relative flex items-center justify-center px-4 py-3 rounded-lg border-2 cursor-pointer transition-all
                                    ${
                                        data.aclevel?.includes(level.name)
                                            ? "border-blue-500 bg-blue-50 dark:border-blue-400 dark:bg-blue-900/30"
                                            : "border-gray-200 bg-white hover:border-gray-300 dark:border-gray-700 dark:bg-gray-800 dark:hover:border-gray-600"
                                    }
                                `}
                                role="radio"
                                aria-checked={data.aclevel?.includes(
                                    level.name
                                )}
                                tabIndex={0}
                                onKeyDown={(e) => {
                                    if (e.key === "Enter" || e.key === " ") {
                                        e.preventDefault();
                                        handleAcademicLevelChange(
                                            `${level.value}#${level.name}`
                                        );
                                    }
                                }}
                            >
                                <span
                                    className={`text-sm font-medium ${
                                        data.aclevel?.includes(level.name)
                                            ? "text-blue-700 dark:text-blue-300"
                                            : "text-gray-700 dark:text-gray-300"
                                    }`}
                                >
                                    {level.name}
                                </span>
                                {data.aclevel?.includes(level.name) && (
                                    <div className="absolute -top-2 -right-2 w-5 h-5 bg-blue-500 dark:bg-blue-400 rounded-full flex items-center justify-center">
                                        <svg
                                            className="w-3 h-3 text-white"
                                            fill="none"
                                            viewBox="0 0 24 24"
                                            stroke="currentColor"
                                        >
                                            <path
                                                strokeLinecap="round"
                                                strokeLinejoin="round"
                                                strokeWidth={2}
                                                d="M5 13l4 4L19 7"
                                            />
                                        </svg>
                                    </div>
                                )}
                            </div>
                        ))}
                    </div>
                    {errors.aclevel && (
                        <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                            {errors.aclevel}
                        </p>
                    )}
                </div>

                {/* Subject selection with dropdown search */}
                <div>
                    <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-3">
                        Subject <span className="text-red-500">*</span>
                    </label>
                    <div className="relative" id="subject-dropdown">
                        <button
                            type="button"
                            ref={subjectButtonRef}
                            onClick={() => {
                                const newState = !subjectDropdownOpen;
                                setSubjectDropdownOpen(newState);
                                if (newState) {
                                    setSubjectSearch("");
                                }
                            }}
                            className="relative w-full bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm pl-3 pr-10 py-2.5 text-left focus:outline-none focus:ring-1 focus:ring-blue-500 focus:border-blue-500 p-3"
                            aria-haspopup="listbox"
                            aria-expanded={subjectDropdownOpen}
                        >
                            <span
                                className={`block truncate ${
                                    selectedSubjectName
                                        ? "text-gray-900 dark:text-white"
                                        : "text-gray-500 dark:text-gray-400"
                                }`}
                            >
                                {selectedSubjectName || "Select a subject"}
                            </span>
                            <span className="absolute inset-y-0 right-0 flex items-center pr-2 pointer-events-none">
                                <ChevronDown
                                    className="h-4 w-4 text-gray-400"
                                    aria-hidden="true"
                                />
                            </span>
                        </button>

                        {subjectDropdownOpen && (
                            <div
                                className="z-50 bg-white dark:bg-gray-800 shadow-lg rounded-md py-1 focus:outline-none"
                                style={subjectDropdownStyle}
                            >
                                <div className="sticky top-0 z-10 bg-white dark:bg-gray-800 p-1">
                                    <div className="relative">
                                        <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                                            <Search className="w-4 h-4 text-gray-500 dark:text-gray-400" />
                                        </div>
                                        <input
                                            type="text"
                                            className="bg-white border border-gray-300 text-gray-900 text-sm rounded-lg focus:ring-blue-500 focus:border-blue-500 block w-full pl-10 p-2 dark:bg-gray-700 dark:border-gray-600 dark:placeholder-gray-400 dark:text-white"
                                            placeholder="Search subjects..."
                                            value={subjectSearch}
                                            onChange={(e) =>
                                                setSubjectSearch(e.target.value)
                                            }
                                            onClick={(e) => e.stopPropagation()}
                                        />
                                    </div>
                                </div>

                                <ul className="py-1" role="listbox">
                                    {filteredSubjects.length === 0 ? (
                                        <li className="text-center py-2 px-4 text-sm text-gray-500 dark:text-gray-400">
                                            No subjects found
                                        </li>
                                    ) : (
                                        filteredSubjects.map((subject) => (
                                            <li
                                                key={subject.id}
                                                className={`
                                                    cursor-pointer select-none relative py-2 pl-3 pr-9 text-sm
                                                    ${
                                                        data.subject?.includes(
                                                            subject.name
                                                        )
                                                            ? "bg-blue-100 text-blue-900 dark:bg-blue-900/50 dark:text-blue-100"
                                                            : "text-gray-900 hover:bg-gray-100 dark:text-white dark:hover:bg-gray-700"
                                                    }
                                                `}
                                                role="option"
                                                aria-selected={data.subject?.includes(
                                                    subject.name
                                                )}
                                                onClick={() =>
                                                    handleSubjectChange(
                                                        `${subject.value}#${subject.name}`,
                                                        subject.name
                                                    )
                                                }
                                            >
                                                <span className="block truncate">
                                                    {subject.name}
                                                </span>
                                                {data.subject?.includes(
                                                    subject.name
                                                ) && (
                                                    <span className="absolute inset-y-0 right-0 flex items-center pr-4">
                                                        <svg
                                                            className="h-4 w-4 text-blue-600 dark:text-blue-400"
                                                            viewBox="0 0 20 20"
                                                            fill="currentColor"
                                                        >
                                                            <path
                                                                fillRule="evenodd"
                                                                d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                                                                clipRule="evenodd"
                                                            />
                                                        </svg>
                                                    </span>
                                                )}
                                            </li>
                                        ))
                                    )}
                                </ul>
                            </div>
                        )}
                    </div>
                    {errors.subject && (
                        <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                            {errors.subject}
                        </p>
                    )}
                </div>

                {/* Paper Type selection with dropdown search */}
                <div>
                    <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-3">
                        Type of Paper <span className="text-red-500">*</span>
                    </label>
                    <div className="relative" id="papertype-dropdown">
                        <button
                            type="button"
                            ref={paperTypeButtonRef}
                            onClick={() => {
                                const newState = !paperTypeDropdownOpen;
                                setPaperTypeDropdownOpen(newState);
                                if (newState) {
                                    setPaperTypeSearch("");
                                }
                            }}
                            className="relative w-full bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm pl-3 pr-10 py-2.5 text-left focus:outline-none focus:ring-1 focus:ring-blue-500 focus:border-blue-500 p-3"
                            aria-haspopup="listbox"
                            aria-expanded={paperTypeDropdownOpen}
                        >
                            <span
                                className={`block truncate ${
                                    selectedPaperTypeName
                                        ? "text-gray-900 dark:text-white"
                                        : "text-gray-500 dark:text-gray-400"
                                }`}
                            >
                                {selectedPaperTypeName || "Select paper type"}
                            </span>
                            <span className="absolute inset-y-0 right-0 flex items-center pr-2 pointer-events-none">
                                <ChevronDown
                                    className="h-4 w-4 text-gray-400"
                                    aria-hidden="true"
                                />
                            </span>
                        </button>

                        {paperTypeDropdownOpen && (
                            <div
                                className="z-50 bg-white dark:bg-gray-800 shadow-lg rounded-md py-1 focus:outline-none"
                                style={paperTypeDropdownStyle}
                            >
                                <div className="sticky top-0 z-10 bg-white dark:bg-gray-800 p-1">
                                    <div className="relative">
                                        <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                                            <Search className="w-4 h-4 text-gray-500 dark:text-gray-400" />
                                        </div>
                                        <input
                                            type="text"
                                            className="bg-white border border-gray-300 text-gray-900 text-sm rounded-lg focus:ring-blue-500 focus:border-blue-500 block w-full pl-10 p-2 dark:bg-gray-700 dark:border-gray-600 dark:placeholder-gray-400 dark:text-white"
                                            placeholder="Search paper types..."
                                            value={paperTypeSearch}
                                            onChange={(e) =>
                                                setPaperTypeSearch(
                                                    e.target.value
                                                )
                                            }
                                            onClick={(e) => e.stopPropagation()}
                                        />
                                    </div>
                                </div>

                                <ul className="py-1" role="listbox">
                                    {filteredPaperTypes.length === 0 ? (
                                        <li className="text-center py-2 px-4 text-sm text-gray-500 dark:text-gray-400">
                                            No paper types found
                                        </li>
                                    ) : (
                                        filteredPaperTypes.map((type) => (
                                            <li
                                                key={type.id}
                                                className={`
                                                    cursor-pointer select-none relative py-2 pl-3 pr-9 text-sm
                                                    ${
                                                        data.papertype?.includes(
                                                            type.name
                                                        )
                                                            ? "bg-blue-100 text-blue-900 dark:bg-blue-900/50 dark:text-blue-100"
                                                            : "text-gray-900 hover:bg-gray-100 dark:text-white dark:hover:bg-gray-700"
                                                    }
                                                `}
                                                role="option"
                                                aria-selected={data.papertype?.includes(
                                                    type.name
                                                )}
                                                onClick={() =>
                                                    handlePaperTypeChange(
                                                        `${type.value}#${type.name}`,
                                                        type.name
                                                    )
                                                }
                                            >
                                                <span className="block truncate">
                                                    {type.name}
                                                </span>
                                                {data.papertype?.includes(
                                                    type.name
                                                ) && (
                                                    <span className="absolute inset-y-0 right-0 flex items-center pr-4">
                                                        <svg
                                                            className="h-4 w-4 text-blue-600 dark:text-blue-400"
                                                            viewBox="0 0 20 20"
                                                            fill="currentColor"
                                                        >
                                                            <path
                                                                fillRule="evenodd"
                                                                d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                                                                clipRule="evenodd"
                                                            />
                                                        </svg>
                                                    </span>
                                                )}
                                            </li>
                                        ))
                                    )}
                                </ul>
                            </div>
                        )}
                    </div>
                    {errors.papertype && (
                        <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                            {errors.papertype}
                        </p>
                    )}
                </div>

                {/* Currency selector */}
                <div>
                    <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-3">
                        Select Currency <span className="text-red-500">*</span>
                    </label>
                    <p className="text-xs text-gray-500 dark:text-gray-400 mb-3">
                        Choose your preferred currency for pricing
                    </p>
                    {currencies?.filter(currency => currency.is_active).length === 0 ? (
                        <div className="p-6 bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg">
                            <div className="flex items-center">
                                <svg className="w-5 h-5 text-yellow-600 dark:text-yellow-400 mr-2" fill="currentColor" viewBox="0 0 20 20">
                                    <path fillRule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                </svg>
                                <div>
                                    <p className="text-sm font-medium text-yellow-800 dark:text-yellow-300">
                                        No currencies available
                                    </p>
                                    <p className="text-xs text-yellow-700 dark:text-yellow-400 mt-1">
                                        Please contact support or check back later. The admin needs to activate currencies.
                                    </p>
                                </div>
                            </div>
                        </div>
                    ) : (
                    <div className="grid grid-cols-2 sm:grid-cols-3 lg:grid-cols-4 xl:grid-cols-5 gap-3">
                        {currencies?.filter(currency => currency.is_active).map((currency) => (
                            <div
                                key={currency.id}
                                onClick={() =>
                                    handleCurrencySelection(currency)
                                }
                                className={`
                                    group relative flex flex-col items-center justify-center p-4 rounded-xl border-2 cursor-pointer transition-all duration-200 hover:shadow-md
                                    ${
                                        data.currency === currency.currency_code
                                            ? "border-blue-500 bg-gradient-to-br from-blue-50 to-blue-100 dark:from-blue-900/40 dark:to-blue-800/40 dark:border-blue-400 shadow-md scale-105"
                                            : "border-gray-200 bg-white hover:border-blue-300 hover:bg-gray-50 dark:border-gray-700 dark:bg-gray-800 dark:hover:border-gray-600 dark:hover:bg-gray-750"
                                    }
                                `}
                                role="button"
                                aria-pressed={
                                    data.currency === currency.currency_code
                                }
                                tabIndex={0}
                                onKeyDown={(e) => {
                                    if (e.key === "Enter" || e.key === " ") {
                                        e.preventDefault();
                                        handleCurrencySelection(currency);
                                    }
                                }}
                            >
                                {/* Selected indicator */}
                                {data.currency === currency.currency_code && (
                                    <div className="absolute -top-2 -right-2 w-6 h-6 bg-blue-500 dark:bg-blue-400 rounded-full flex items-center justify-center shadow-lg">
                                        <svg
                                            className="w-4 h-4 text-white"
                                            fill="none"
                                            viewBox="0 0 24 24"
                                            stroke="currentColor"
                                        >
                                            <path
                                                strokeLinecap="round"
                                                strokeLinejoin="round"
                                                strokeWidth={3}
                                                d="M5 13l4 4L19 7"
                                            />
                                        </svg>
                                    </div>
                                )}
                                
                                {/* Currency icon */}
                                <div className={`mb-2 transition-colors ${
                                    data.currency === currency.currency_code
                                        ? "text-blue-600 dark:text-blue-300"
                                        : "text-gray-600 dark:text-gray-400 group-hover:text-blue-500"
                                }`}>
                                    {currencyIcons[currency.currency_code] || <DollarSign className="w-5 h-5" />}
                                </div>
                                
                                {/* Currency code */}
                                <span className={`text-sm font-bold mb-1 transition-colors ${
                                    data.currency === currency.currency_code
                                        ? "text-blue-700 dark:text-blue-200"
                                        : "text-gray-900 dark:text-gray-100 group-hover:text-blue-600"
                                }`}>
                                    {currency.currency_code}
                                </span>
                                
                                {/* Currency name */}
                                <span className={`text-xs text-center leading-tight transition-colors ${
                                    data.currency === currency.currency_code
                                        ? "text-blue-600 dark:text-blue-300"
                                        : "text-gray-500 dark:text-gray-400 group-hover:text-gray-700"
                                }`}>
                                    {currency.name}
                                </span>
                                
                                {/* Exchange rate indicator */}
                                {currency.value !== "1.00" && currency.value !== 1 && (
                                    <span className="text-xs text-gray-400 dark:text-gray-500 mt-1">
                                        {parseFloat(currency.value).toFixed(2)}
                                    </span>
                                )}
                            </div>
                        ))}
                    </div>
                    )}
                    {errors.currency && (
                        <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                            {errors.currency}
                        </p>
                    )}
                    
                    {/* Selected currency info */}
                    {data.currency && (
                        <div className="mt-4 p-3 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-200 dark:border-blue-800">
                            <div className="flex items-center justify-between text-sm">
                                <span className="text-gray-700 dark:text-gray-300">
                                    Selected:
                                </span>
                                <span className="font-semibold text-blue-700 dark:text-blue-300">
                                    {currencies?.find(c => c.currency_code === data.currency)?.name || data.currency}
                                </span>
                            </div>
                            {data.currency !== "USD" && data.currencyRate && (
                                <div className="flex items-center justify-between text-xs text-gray-600 dark:text-gray-400 mt-1">
                                    <span>Exchange rate:</span>
                                    <span>1 USD = {parseFloat(data.currencyRate).toFixed(2)} {data.currency}</span>
                                </div>
                            )}
                        </div>
                    )}
                </div>
            </div>

            {/* Price preview */}
            {/* {data.aclevel && data.subject && data.papertype && (
                <div className="mt-8 p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-100 dark:border-blue-800">
                    <h3 className="text-sm font-semibold text-blue-800 dark:text-blue-300 mb-2">
                        Preliminary Price Estimate
                    </h3>
                    <div className="flex flex-col sm:flex-row sm:items-center justify-between">
                        <div>
                            <p className="text-sm text-gray-600 dark:text-gray-300">
                                Base price per page:
                            </p>
                            <p className="text-lg font-bold text-gray-900 dark:text-white">
                                {orderCalculation.currencySymbol}
                                {orderCalculation.costPerPage}
                            </p>
                        </div>

                        <div className="mt-2 sm:mt-0">
                            <p className="text-sm text-gray-600 dark:text-gray-300">
                                Total (estimated):
                            </p>
                            <p className="text-lg font-bold text-blue-600 dark:text-blue-300">
                                {orderCalculation.currencySymbol}
                                {orderCalculation.totalAmount}
                            </p>
                            {data.currency !== "USD" && (
                                <p className="text-xs text-gray-500 dark:text-gray-400">
                                    ($
                                    {(
                                        parseFloat(
                                            orderCalculation.totalAmount
                                        ) / (data.currencyRate || 1)
                                    ).toFixed(2)}{" "}
                                    USD)
                                </p>
                            )}
                        </div>
                    </div>
                    <p className="mt-2 text-xs text-gray-500 dark:text-gray-400">
                        This price will be refined as you complete your order
                        details.
                    </p>
                </div>
            )} */}
        </div>
    );
};

export default BasicInformation;
