import React, { useState } from "react";
import {
    BookOpen,
    Book,
    FileText,
    Globe,
    Upload,
    FileUp,
    X,
    AlertCircle,
} from "lucide-react";
import { FormTextarea } from "@/Components/ui/FormTextarea";

const WritingPreferences = ({ data, setData, errors, orderCalculation }) => {
    // Format file size for display
    const formatFileSize = (bytes) => {
        if (bytes === 0) return "0 Bytes";
        const k = 1024;
        const sizes = ["Bytes", "KB", "MB", "GB"];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + " " + sizes[i];
    };

    // Handle file upload
    const handleFileUpload = (e) => {
        setData("files", Array.from(e.target.files));
    };

    // Remove file from upload list
    const removeFile = (index) => {
        const newFiles = [...data.files];
        newFiles.splice(index, 1);
        setData("files", newFiles);
    };

    // Handle citation style selection
    const handleStyleChange = (value) => {
        setData("styles", value);
    };

    // Handle language preference selection
    const handleLanguageChange = (value) => {
        setData("language", value);
    };

    return (
        <div className="space-y-8" aria-labelledby="step3-heading">
            {/* Citation Style selection */}
            <div>
                <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-3">
                    Citation Style <span className="text-red-500">*</span>
                </label>
                <div
                    className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 gap-3"
                    role="radiogroup"
                    aria-required="true"
                    aria-label="Select citation style"
                >
                    {[
                        {
                            id: 1,
                            value: "1",
                            name: "APA",
                            icon: <BookOpen className="w-5 h-5" />,
                        },
                        {
                            id: 2,
                            value: "2",
                            name: "MLA",
                            icon: <Book className="w-5 h-5" />,
                        },
                        {
                            id: 3,
                            value: "3",
                            name: "Chicago",
                            icon: <FileText className="w-5 h-5" />,
                        },
                        {
                            id: 4,
                            value: "4",
                            name: "Harvard",
                            icon: <BookOpen className="w-5 h-5" />,
                        },
                        {
                            id: 5,
                            value: "5",
                            name: "Turabian",
                            icon: <Book className="w-5 h-5" />,
                        },
                        {
                            id: 6,
                            value: "6",
                            name: "Oxford",
                            icon: <FileText className="w-5 h-5" />,
                        },
                        {
                            id: 7,
                            value: "7",
                            name: "Vancouver",
                            icon: <BookOpen className="w-5 h-5" />,
                        },
                    ].map((style) => (
                        <div
                            key={style.id}
                            onClick={() => handleStyleChange(style.value)}
                            className={`
                                flex items-center px-4 py-3 rounded-lg border-2 cursor-pointer transition-all
                                ${
                                    data.styles === style.value
                                        ? "border-blue-500 bg-blue-50 dark:border-blue-400 dark:bg-blue-900/30"
                                        : "border-gray-200 bg-white hover:border-gray-300 dark:border-gray-700 dark:bg-gray-800 dark:hover:border-gray-600"
                                }
                            `}
                            role="radio"
                            aria-checked={data.styles === style.value}
                            tabIndex={0}
                            onKeyDown={(e) => {
                                if (e.key === "Enter" || e.key === " ") {
                                    e.preventDefault();
                                    handleStyleChange(style.value);
                                }
                            }}
                        >
                            <div
                                className={`mr-3 ${
                                    data.styles === style.value
                                        ? "text-blue-600 dark:text-blue-400"
                                        : "text-gray-500 dark:text-gray-400"
                                }`}
                            >
                                {style.icon}
                            </div>
                            <span
                                className={`text-sm font-medium ${
                                    data.styles === style.value
                                        ? "text-blue-700 dark:text-blue-300"
                                        : "text-gray-700 dark:text-gray-300"
                                }`}
                            >
                                {style.name}
                            </span>
                        </div>
                    ))}
                </div>
                {errors.styles && (
                    <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                        {errors.styles}
                    </p>
                )}
            </div>

            {/* Language Preference selection */}
            <div>
                <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-3">
                    Language Preference <span className="text-red-500">*</span>
                </label>
                <div
                    className="grid grid-cols-1 sm:grid-cols-2 gap-3"
                    role="radiogroup"
                    aria-required="true"
                    aria-label="Select language preference"
                >
                    {[
                        {
                            id: 1,
                            value: "1",
                            name: "American English",
                            flag: "🇺🇸",
                        },
                        {
                            id: 2,
                            value: "2",
                            name: "British English",
                            flag: "🇬🇧",
                        },
                        {
                            id: 3,
                            value: "3",
                            name: "Australian English",
                            flag: "🇦🇺",
                        },
                        {
                            id: 4,
                            value: "4",
                            name: "Canadian English",
                            flag: "🇨🇦",
                        },
                    ].map((lang) => (
                        <div
                            key={lang.id}
                            onClick={() => handleLanguageChange(lang.value)}
                            className={`
                                flex items-center px-4 py-3 rounded-lg border-2 cursor-pointer transition-all
                                ${
                                    data.language === lang.value
                                        ? "border-blue-500 bg-blue-50 dark:border-blue-400 dark:bg-blue-900/30"
                                        : "border-gray-200 bg-white hover:border-gray-300 dark:border-gray-700 dark:bg-gray-800 dark:hover:border-gray-600"
                                }
                            `}
                            role="radio"
                            aria-checked={data.language === lang.value}
                            tabIndex={0}
                            onKeyDown={(e) => {
                                if (e.key === "Enter" || e.key === " ") {
                                    e.preventDefault();
                                    handleLanguageChange(lang.value);
                                }
                            }}
                        >
                            <span className="text-xl mr-3" aria-hidden="true">
                                {lang.flag}
                            </span>
                            <span
                                className={`text-sm font-medium ${
                                    data.language === lang.value
                                        ? "text-blue-700 dark:text-blue-300"
                                        : "text-gray-700 dark:text-gray-300"
                                }`}
                            >
                                {lang.name}
                            </span>
                        </div>
                    ))}
                </div>
                {errors.language && (
                    <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                        {errors.language}
                    </p>
                )}
            </div>

            {/* Instructions textarea */}
            <div>
                <FormTextarea
                    id="instructions"
                    name="instructions"
                    label="Instructions"
                    value={data.instructions || ""}
                    setData={setData}
                    error={errors.instructions}
                    placeholder="Provide detailed instructions for your assignment. "
                    className="dark:text-gray-200 min-h-40"
                    required={true}
                    rows={6}
                />
                <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                    Be as detailed as possible to ensure your writer understands
                    your requirements exactly.
                </p>
            </div>

            {/* File Upload */}
            <div>
                <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-3">
                    Upload Files (Optional)
                </label>
                <div className="mt-1 flex justify-center px-6 pt-5 pb-6 border-2 border-dashed rounded-lg border-gray-300 dark:border-gray-600">
                    <div className="space-y-3 text-center">
                        <FileUp
                            className="mx-auto h-12 w-12 text-gray-400"
                            aria-hidden="true"
                        />
                        <div className="flex text-sm text-gray-600 dark:text-gray-400">
                            <label
                                htmlFor="file-upload"
                                className="relative cursor-pointer  dark:bg-gray-800 rounded-md font-medium text-blue-600 dark:text-blue-400 hover:text-blue-500 dark:hover:text-blue-300 focus:outline-none p-2"
                            >
                                <span className="text-xs bg-white dark:bg-gray-800 rounded-md px-2 py-1 border border-gray-300 dark:border-gray-600">
                                    Click to Upload{" "}
                                </span>
                                <input
                                    id="file-upload"
                                    name="file-upload"
                                    type="file"
                                    className="sr-only"
                                    onChange={handleFileUpload}
                                    multiple
                                    accept=".pdf,.doc,.docx,.txt,.rtf,.jpg,.jpeg,.png,.xlsx,.pptx"
                                />
                            </label>
                        </div>
                        <p className="text-xs text-gray-500 dark:text-gray-400">
                            PDF, Word, Excel, PowerPoint, images up to 10MB
                            each. (For Large files, Upload in your order
                            management page)
                        </p>
                    </div>
                </div>

                {/* Files list */}
                {data.files?.length > 0 && (
                    <div className="mt-4">
                        <h4 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Selected Files ({data.files.length})
                        </h4>
                        <ul className="divide-y divide-gray-200 dark:divide-gray-700 border border-gray-200 dark:border-gray-700 rounded-lg overflow-hidden">
                            {Array.from(data.files).map((file, index) => (
                                <li
                                    key={index}
                                    className="px-4 py-3 flex items-center justify-between bg-white dark:bg-gray-800"
                                >
                                    <div className="flex items-center">
                                        <FileText className="w-5 h-5 text-gray-500 dark:text-gray-400 mr-3" />
                                        <div className="flex flex-col">
                                            <span
                                                className="text-sm font-medium text-gray-700 dark:text-gray-300 truncate"
                                                style={{ maxWidth: "200px" }}
                                            >
                                                {file.name}
                                            </span>
                                            <span className="text-xs text-gray-500 dark:text-gray-400">
                                                {formatFileSize(file.size)}
                                            </span>
                                        </div>
                                    </div>
                                    <button
                                        type="button"
                                        onClick={() => removeFile(index)}
                                        className="text-red-600 dark:text-red-400 hover:text-red-800 dark:hover:text-red-300"
                                        aria-label={`Remove file ${file.name}`}
                                    >
                                        <X className="w-5 h-5" />
                                    </button>
                                </li>
                            ))}
                        </ul>
                    </div>
                )}

                {errors.files && (
                    <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                        {errors.files}
                    </p>
                )}
            </div>

            {/* Final price display */}
            <div className="mt-8 p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-100 dark:border-blue-800">
                <h3 className="text-sm font-semibold text-blue-800 dark:text-blue-300 mb-2">
                    Order Summary
                </h3>
                <div className="flex justify-between items-center">
                    <span className="text-sm text-gray-700 dark:text-gray-300">
                        Total Amount:
                    </span>
                    <span className="text-xl font-bold text-blue-600 dark:text-blue-300">
                        {orderCalculation.currencySymbol}
                        {orderCalculation.totalAmount}
                    </span>
                </div>
                {data.currency !== "USD" && (
                    <div className="text-right mt-1">
                        <span className="text-xs text-gray-500 dark:text-gray-400">
                            ($
                            {(
                                parseFloat(orderCalculation.totalAmount) /
                                (data.currencyRate || 1)
                            ).toFixed(2)}{" "}
                            USD)
                        </span>
                    </div>
                )}
                <p className="mt-2 text-xs text-gray-500 dark:text-gray-400">
                    Review your order on the next step before submitting.
                </p>
            </div>
        </div>
    );
};

export default WritingPreferences;
