import { Link } from "@inertiajs/react";
import { motion } from "framer-motion";
import {
    FileText,
    BookOpen,
    GraduationCap,
    ClipboardList,
    BarChart3,
    Search,
    Target,
    User,
    Code,
    ArrowRight,
    CheckCircle,
    Star,
    Clock,
    Shield,
} from "lucide-react";

export default function OurServices() {
    // Animation variants
    const fadeIn = {
        hidden: { opacity: 0, y: 20 },
        visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
    };

    const staggerContainer = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: {
                staggerChildren: 0.1,
            },
        },
    };

    const services = [
        {
            name: "Essay Writing",
            description: "Professional essay writing service with expert writers specializing in all academic levels and subjects.",
            icon: <FileText className="h-8 w-8" />,
            href: "/services/essay-writing",
            features: ["All Academic Levels", "24/7 Support", "Plagiarism-Free"],
            color: "blue",
            gradient: "from-blue-500 to-blue-600",
            bgGradient: "from-blue-50 to-blue-100",
            darkBgGradient: "from-blue-900/20 to-blue-800/20",
        },
        {
            name: "Research Papers",
            description: "Comprehensive research paper writing service with thorough analysis and proper citations.",
            icon: <Search className="h-8 w-8" />,
            href: "/services/research-papers",
            features: ["In-depth Research", "Proper Citations", "Quality Assurance"],
            color: "green",
            gradient: "from-green-500 to-green-600",
            bgGradient: "from-green-50 to-green-100",
            darkBgGradient: "from-green-900/20 to-green-800/20",
        },
        {
            name: "Dissertation Writing",
            description: "Expert dissertation writing help from PhD-qualified writers with extensive research experience.",
            icon: <GraduationCap className="h-8 w-8" />,
            href: "/services/dissertation-writing",
            features: ["PhD Writers", "Original Research", "Timely Delivery"],
            color: "purple",
            gradient: "from-purple-500 to-purple-600",
            bgGradient: "from-purple-50 to-purple-100",
            darkBgGradient: "from-purple-900/20 to-purple-800/20",
        },
        {
            name: "Assignments",
            description: "Custom assignment writing service covering all subjects and academic requirements.",
            icon: <ClipboardList className="h-8 w-8" />,
            href: "/services/assignments",
            features: ["All Subjects", "Custom Solutions", "Fast Turnaround"],
            color: "orange",
            gradient: "from-orange-500 to-orange-600",
            bgGradient: "from-orange-50 to-orange-100",
            darkBgGradient: "from-orange-900/20 to-orange-800/20",
        },
        {
            name: "Case Studies",
            description: "Professional case study analysis and writing service with detailed insights and solutions.",
            icon: <BarChart3 className="h-8 w-8" />,
            href: "/services/case-studies",
            features: ["Detailed Analysis", "Real Examples", "Expert Insights"],
            color: "indigo",
            gradient: "from-indigo-500 to-indigo-600",
            bgGradient: "from-indigo-50 to-indigo-100",
            darkBgGradient: "from-indigo-900/20 to-indigo-800/20",
        },
        {
            name: "Literature Reviews",
            description: "Comprehensive literature review writing service with systematic analysis and synthesis.",
            icon: <BookOpen className="h-8 w-8" />,
            href: "/services/literature-reviews",
            features: ["Systematic Review", "Critical Analysis", "Academic Standards"],
            color: "teal",
            gradient: "from-teal-500 to-teal-600",
            bgGradient: "from-teal-50 to-teal-100",
            darkBgGradient: "from-teal-900/20 to-teal-800/20",
        },
        {
            name: "Thesis Proposals",
            description: "Expert thesis proposal writing service with clear methodology and research objectives.",
            icon: <Target className="h-8 w-8" />,
            href: "/services/thesis-proposals",
            features: ["Clear Methodology", "Research Objectives", "Academic Format"],
            color: "pink",
            gradient: "from-pink-500 to-pink-600",
            bgGradient: "from-pink-50 to-pink-100",
            darkBgGradient: "from-pink-900/20 to-pink-800/20",
        },
        {
            name: "Personal Statements",
            description: "Compelling personal statement writing service for university applications and scholarships.",
            icon: <User className="h-8 w-8" />,
            href: "/services/personal-statements",
            features: ["Compelling Narrative", "Application Focus", "Success Stories"],
            color: "rose",
            gradient: "from-rose-500 to-rose-600",
            bgGradient: "from-rose-50 to-rose-100",
            darkBgGradient: "from-rose-900/20 to-rose-800/20",
        },
        {
            name: "Programming Assignments",
            description: "Expert programming assignment help with clean code, documentation, and testing.",
            icon: <Code className="h-8 w-8" />,
            href: "/services/programming-assignments",
            features: ["Clean Code", "Documentation", "Testing Included"],
            color: "cyan",
            gradient: "from-cyan-500 to-cyan-600",
            bgGradient: "from-cyan-50 to-cyan-100",
            darkBgGradient: "from-cyan-900/20 to-cyan-800/20",
        },
    ];

    const getColorClasses = (color) => {
        const colorMap = {
            blue: {
                text: "text-blue-600 dark:text-blue-400",
                bg: "bg-blue-100 dark:bg-blue-900/30",
                border: "border-blue-200 dark:border-blue-700",
                hover: "hover:bg-blue-50 dark:hover:bg-blue-900/50",
            },
            green: {
                text: "text-green-600 dark:text-green-400",
                bg: "bg-green-100 dark:bg-green-900/30",
                border: "border-green-200 dark:border-green-700",
                hover: "hover:bg-green-50 dark:hover:bg-green-900/50",
            },
            purple: {
                text: "text-purple-600 dark:text-purple-400",
                bg: "bg-purple-100 dark:bg-purple-900/30",
                border: "border-purple-200 dark:border-purple-700",
                hover: "hover:bg-purple-50 dark:hover:bg-purple-900/50",
            },
            orange: {
                text: "text-orange-600 dark:text-orange-400",
                bg: "bg-orange-100 dark:bg-orange-900/30",
                border: "border-orange-200 dark:border-orange-700",
                hover: "hover:bg-orange-50 dark:hover:bg-orange-900/50",
            },
            indigo: {
                text: "text-indigo-600 dark:text-indigo-400",
                bg: "bg-indigo-100 dark:bg-indigo-900/30",
                border: "border-indigo-200 dark:border-indigo-700",
                hover: "hover:bg-indigo-50 dark:hover:bg-indigo-900/50",
            },
            teal: {
                text: "text-teal-600 dark:text-teal-400",
                bg: "bg-teal-100 dark:bg-teal-900/30",
                border: "border-teal-200 dark:border-teal-700",
                hover: "hover:bg-teal-50 dark:hover:bg-teal-900/50",
            },
            pink: {
                text: "text-pink-600 dark:text-pink-400",
                bg: "bg-pink-100 dark:bg-pink-900/30",
                border: "border-pink-200 dark:border-pink-700",
                hover: "hover:bg-pink-50 dark:hover:bg-pink-900/50",
            },
            rose: {
                text: "text-rose-600 dark:text-rose-400",
                bg: "bg-rose-100 dark:bg-rose-900/30",
                border: "border-rose-200 dark:border-rose-700",
                hover: "hover:bg-rose-50 dark:hover:bg-rose-900/50",
            },
            cyan: {
                text: "text-cyan-600 dark:text-cyan-400",
                bg: "bg-cyan-100 dark:bg-cyan-900/30",
                border: "border-cyan-200 dark:border-cyan-700",
                hover: "hover:bg-cyan-50 dark:hover:bg-cyan-900/50",
            },
        };
        return colorMap[color] || colorMap.blue;
    };

    return (
        <section className="py-20 bg-gradient-to-br from-gray-50 via-white to-blue-50 dark:from-gray-900 dark:via-gray-800 dark:to-blue-900/20">
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                {/* Section Header */}
                <motion.div
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true }}
                    variants={fadeIn}
                    className="text-center mb-16"
                >
                    <h2 className="text-4xl md:text-5xl font-bold text-gray-900 dark:text-white mb-6">
                        Our{" "}
                        <span className="bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">
                            Academic Writing Services
                        </span>
                    </h2>
                    <div className="mt-4 h-1 w-24 bg-gradient-to-r from-blue-600 to-purple-600 mx-auto rounded-full"></div>
                    <p className="mt-6 max-w-3xl mx-auto text-xl text-gray-600 dark:text-gray-300 leading-relaxed">
                        Professional academic writing services tailored to meet your specific needs. 
                        Our expert writers deliver high-quality, original content across all academic disciplines.
                    </p>
                </motion.div>

                {/* Services Grid */}
                <motion.div
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true }}
                    variants={staggerContainer}
                    className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8"
                >
                    {services.map((service, index) => {
                        const colorClasses = getColorClasses(service.color);
                        return (
                            <motion.div
                                key={service.name}
                                variants={fadeIn}
                                className="group"
                            >
                                <Link
                                    href={service.href}
                                    className={`block p-6 rounded-2xl border-2 transition-all duration-300 transform hover:scale-105 hover:shadow-xl ${colorClasses.bg} ${colorClasses.border} ${colorClasses.hover}`}
                                >
                                    {/* Service Icon */}
                                    <div className={`inline-flex p-3 rounded-xl ${colorClasses.bg} ${colorClasses.text} mb-4 group-hover:scale-110 transition-transform duration-300`}>
                                        {service.icon}
                                    </div>

                                    {/* Service Title */}
                                    <h3 className={`text-xl font-bold ${colorClasses.text} mb-3 group-hover:text-opacity-80 transition-colors duration-300`}>
                                        {service.name}
                                    </h3>

                                    {/* Service Description */}
                                    <p className="text-gray-600 dark:text-gray-300 mb-4 leading-relaxed">
                                        {service.description}
                                    </p>

                                    {/* Service Features */}
                                    <div className="space-y-2 mb-6">
                                        {service.features.map((feature, featureIndex) => (
                                            <div key={featureIndex} className="flex items-center space-x-2">
                                                <CheckCircle className={`h-4 w-4 ${colorClasses.text} flex-shrink-0`} />
                                                <span className="text-sm text-gray-600 dark:text-gray-300">
                                                    {feature}
                                                </span>
                                            </div>
                                        ))}
                                    </div>

                                    {/* CTA Button */}
                                    <div className="flex items-center justify-between">
                                        <span className={`text-sm font-semibold ${colorClasses.text} group-hover:text-opacity-80 transition-colors duration-300`}>
                                            Learn More
                                        </span>
                                        <ArrowRight className={`h-5 w-5 ${colorClasses.text} group-hover:translate-x-1 transition-transform duration-300`} />
                                    </div>
                                </Link>
                            </motion.div>
                        );
                    })}
                </motion.div>

                {/* Bottom CTA Section */}
                <motion.div
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true }}
                    variants={fadeIn}
                    className="mt-16 text-center"
                >
                    <div className="bg-gradient-to-r from-blue-600 to-purple-600 rounded-2xl p-8 md:p-12 text-white">
                        <div className="max-w-4xl mx-auto">
                            <h3 className="text-3xl md:text-4xl font-bold mb-4">
                                Ready to Get Started?
                            </h3>
                            <p className="text-xl mb-8 opacity-90">
                                Choose from our comprehensive range of academic writing services. 
                                All services come with our quality guarantee and 24/7 support.
                            </p>
                            
                            {/* Service Stats */}
                            <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
                                <div className="flex items-center justify-center space-x-3">
                                    <Star className="h-6 w-6 text-yellow-300 fill-current" />
                                    <div className="text-left">
                                        <div className="text-2xl font-bold">4.9/5</div>
                                        <div className="text-sm opacity-90">Average Rating</div>
                                    </div>
                                </div>
                                <div className="flex items-center justify-center space-x-3">
                                    <Clock className="h-6 w-6 text-green-300" />
                                    <div className="text-left">
                                        <div className="text-2xl font-bold">24/7</div>
                                        <div className="text-sm opacity-90">Support Available</div>
                                    </div>
                                </div>
                                <div className="flex items-center justify-center space-x-3">
                                    <Shield className="h-6 w-6 text-blue-300" />
                                    <div className="text-left">
                                        <div className="text-2xl font-bold">100%</div>
                                        <div className="text-sm opacity-90">Plagiarism-Free</div>
                                    </div>
                                </div>
                            </div>

                            <div className="flex flex-col sm:flex-row gap-4 justify-center">
                                <Link
                                    href="/services"
                                    className="px-8 py-4 bg-white text-blue-600 font-semibold rounded-lg hover:bg-gray-100 transition-colors duration-300 shadow-lg"
                                >
                                    View All Services
                                </Link>
                                <Link
                                    href="/place-order"
                                    className="px-8 py-4 bg-orange-500 hover:bg-orange-600 font-semibold rounded-lg transition-colors duration-300 shadow-lg"
                                >
                                    Place Order Now
                                </Link>
                            </div>
                        </div>
                    </div>
                </motion.div>
            </div>
        </section>
    );
}

