import React, { useEffect, useRef, useState } from "react";
import axios from "axios";
import { Loader2, AlertCircle } from "lucide-react";

const PayPalButton = ({ order, onSuccess, onError }) => {
    const paypalButtonRef = useRef(null);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);
    const [scriptLoaded, setScriptLoaded] = useState(false);
    const [paymentInProcess, setPaymentInProcess] = useState(false);

    // Load the PayPal SDK
    useEffect(() => {
        // Reset states
        setLoading(true);
        setError(null);

        const loadPayPalScript = async () => {
            try {
                // First check if script is already loaded
                if (
                    document.querySelector(
                        'script[src*="www.paypal.com/sdk/js"]'
                    )
                ) {
                    console.log("PayPal script already loaded");
                    setScriptLoaded(true);
                    setLoading(false);
                    return;
                }

                // Get client ID from backend
                const response = await axios.get(route("paypal.client-id"));
                const { clientId } = response.data;

                if (!clientId) {
                    console.error("PayPal client ID is missing");
                    setError(
                        "PayPal configuration is missing. Please contact support."
                    );
                    setLoading(false);
                    return;
                }

                // Load PayPal script
                const script = document.createElement("script");
                script.src = `https://www.paypal.com/sdk/js?client-id=${clientId}&currency=USD`;
                script.async = true;
                script.id = "paypal-sdk-script";

                script.onload = () => {
                    console.log("PayPal script loaded successfully");
                    setScriptLoaded(true);
                    setLoading(false);
                };

                script.onerror = (e) => {
                    console.error("Failed to load PayPal SDK", e);
                    setError("Failed to load PayPal SDK");
                    setLoading(false);
                };

                document.body.appendChild(script);
            } catch (err) {
                console.error("Error initializing PayPal:", err);
                setError("Failed to initialize PayPal");
                setLoading(false);
            }
        };

        loadPayPalScript();
    }, []);

    // Initialize the PayPal buttons
    useEffect(() => {
        if (!scriptLoaded || !window.paypal || !paypalButtonRef.current) {
            return;
        }

        // Clear existing buttons
        paypalButtonRef.current.innerHTML = "";

        try {
            window.paypal
                .Buttons({
                    // Server-side order creation
                    createOrder: async () => {
                        try {
                            setPaymentInProcess(true);

                            // Use our new payment architecture
                            const { data } = await axios.post(
                                route("orders.initialize-payment", order.id),
                                { payment_method: "paypal" }
                            );

                            if (!data.success) {
                                throw new Error(
                                    data.error || "Failed to initialize payment"
                                );
                            }

                            return data.paypal_order_id;
                        } catch (error) {
                            console.error("PayPal create order error:", error);
                            setError(
                                error.response?.data?.message ||
                                    "Failed to create PayPal order"
                            );
                            setPaymentInProcess(false);
                            throw error;
                        }
                    },

                    onApprove: (data, actions) => {
                        return actions.order
                            .capture()
                            .then(async function (details) {
                                try {
                                    // Process the payment with our new API
                                    const response = await axios.post(
                                        route(
                                            "orders.process-payment",
                                            order.id
                                        ),
                                        {
                                            payment_method: "paypal",
                                            payment_data: {
                                                paymentId: data.orderID,
                                                payerId: details.payer.payer_id,
                                                details: details,
                                            },
                                        }
                                    );

                                    if (response.data.success) {
                                        if (onSuccess) {
                                            onSuccess(response.data);
                                        } else {
                                            // Default behavior: redirect
                                            window.location.href =
                                                response.data.redirect ||
                                                route("orders.show", order.id);
                                        }
                                    } else {
                                        throw new Error(
                                            response.data.message ||
                                                "Payment processing failed"
                                        );
                                    }
                                } catch (error) {
                                    console.error(
                                        "PayPal payment processing error:",
                                        error
                                    );
                                    setError(
                                        error.response?.data?.message ||
                                            "Payment processing failed"
                                    );

                                    if (onError) {
                                        onError(error);
                                    }
                                } finally {
                                    setPaymentInProcess(false);
                                }
                            });
                    },

                    onCancel: () => {
                        console.log("Payment cancelled");
                        setPaymentInProcess(false);
                    },

                    onError: (err) => {
                        console.error("PayPal Button Error:", err);
                        setError("Payment failed. Please try again.");
                        setPaymentInProcess(false);

                        if (onError) {
                            onError(err);
                        }
                    },
                })
                .render(paypalButtonRef.current);
        } catch (error) {
            console.error("Error rendering PayPal buttons:", error);
            setError(
                "Error displaying payment options. Please refresh the page."
            );
        }
    }, [scriptLoaded, order.id]);

    if (loading && !scriptLoaded) {
        return (
            <div className="w-full p-4 bg-gray-100 dark:bg-gray-800 rounded-lg text-center">
                <div className="flex items-center justify-center space-x-2 text-gray-600 dark:text-gray-300">
                    <Loader2 className="w-5 h-5 animate-spin" />
                    <span>Loading payment options...</span>
                </div>
            </div>
        );
    }

    if (error) {
        return (
            <div className="w-full p-4 bg-red-100 dark:bg-red-900 text-red-700 dark:text-red-300 rounded-lg flex items-start space-x-2">
                <AlertCircle className="w-5 h-5 mt-0.5 flex-shrink-0" />
                <span>{error}</span>
            </div>
        );
    }

    return (
        <div className="space-y-4">
            <h3 className="text-lg font-bold mb-4 text-gray-800 dark:text-gray-100">
                Pay with PayPal
            </h3>

            <div className="relative">
                {paymentInProcess && (
                    <div className="absolute inset-0 bg-white/70 dark:bg-gray-900/70 flex items-center justify-center z-10 rounded-lg">
                        <div className="flex items-center justify-center space-x-2 text-blue-600 dark:text-blue-400">
                            <Loader2 className="w-5 h-5 animate-spin" />
                            <span>Processing payment...</span>
                        </div>
                    </div>
                )}
                <div ref={paypalButtonRef} className="w-full"></div>
            </div>

            <p className="text-sm text-gray-500 dark:text-gray-400 mt-2">
                You'll be redirected to PayPal to complete your payment
                securely.
            </p>
        </div>
    );
};

export default PayPalButton;
