import React, { useState, useEffect } from "react";
import axios from "axios";
import {
    Loader2,
    AlertCircle,
    CheckCircle,
    CreditCard,
    Lock,
} from "lucide-react";

const PayPalDirectForm = ({ order, onSuccess, onError }) => {
    const [formData, setFormData] = useState(null);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);
    const [submitting, setSubmitting] = useState(false);

    // Helper function to get currency symbol
    const getCurrencySymbol = (currencyCode) => {
        switch (currencyCode) {
            case "EUR":
                return "€";
            case "GBP":
                return "£";
            case "CAD":
                return "C$";
            case "AUD":
                return "A$";
            default:
                return "$";
        }
    };

    useEffect(() => {
        // Fetch form data from backend
        const fetchFormData = async () => {
            try {
                setLoading(true);
                setError(null);

                const { data } = await axios.post(
                    route("orders.initialize-payment", order.id),
                    { payment_method: "paypal_direct" }
                );

                if (!data.success) {
                    throw new Error(data.error || "Failed to initialize payment");
                }

                setFormData(data.form_data);
            } catch (err) {
                console.error("Error fetching PayPal Direct form data:", err);
                const errorMessage = err.response?.data?.message || err.message || "Failed to load payment form";
                setError(errorMessage);
                onError?.({ message: errorMessage });
            } finally {
                setLoading(false);
            }
        };

        fetchFormData();
    }, [order.id, onError]);

    const handleFormSubmit = () => {
        setSubmitting(true);
        // The form will submit to PayPal, so we just need to track the submission
        // The actual payment processing happens via IPN
    };

    if (loading) {
        return (
            <div className="p-6 bg-gray-50 dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 text-center">
                <Loader2 className="h-8 w-8 animate-spin mx-auto mb-4 text-blue-600 dark:text-blue-400" />
                <p className="text-gray-600 dark:text-gray-400">
                    Loading PayPal payment form...
                </p>
            </div>
        );
    }

    if (error) {
        return (
            <div className="p-6 bg-red-50 dark:bg-red-900 rounded-lg border border-red-200 dark:border-red-700">
                <div className="flex items-start space-x-2">
                    <AlertCircle className="h-5 w-5 mt-0.5 text-red-500 dark:text-red-400 flex-shrink-0" />
                    <div>
                        <h3 className="text-sm font-medium text-red-800 dark:text-red-200">
                            Payment Form Error
                        </h3>
                        <p className="mt-1 text-sm text-red-700 dark:text-red-300">
                            {error}
                        </p>
                        <button
                            onClick={() => window.location.reload()}
                            className="mt-3 text-sm text-red-600 dark:text-red-400 hover:text-red-500 dark:hover:text-red-300 underline"
                        >
                            Try again
                        </button>
                    </div>
                </div>
            </div>
        );
    }

    if (!formData) {
        return (
            <div className="p-6 bg-yellow-50 dark:bg-yellow-900 rounded-lg border border-yellow-200 dark:border-yellow-700">
                <p className="text-yellow-700 dark:text-yellow-300">
                    Payment form data not available. Please try again.
                </p>
            </div>
        );
    }

    return (
        <div className="space-y-4">
            {/* Payment Summary */}
            <div className="p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-200 dark:border-blue-700">
                <div className="flex items-center space-x-2 mb-2">
                    <CreditCard className="h-5 w-5 text-blue-600 dark:text-blue-400" />
                    <h3 className="text-sm font-medium text-blue-800 dark:text-blue-200">
                        PayPal Direct Payment
                    </h3>
                </div>
                <p className="text-sm text-blue-700 dark:text-blue-300">
                    You will be redirected to PayPal to complete your payment securely.
                </p>
            </div>

            {/* PayPal Form */}
            <form
                action={formData.action}
                method="post"
                onSubmit={handleFormSubmit}
                className="space-y-4"
            >
                {/* Hidden form fields */}
                {Object.entries(formData).map(([key, value]) => 
                    key !== 'action' && (
                        <input
                            key={key}
                            type="hidden"
                            name={key}
                            value={value}
                        />
                    )
                )}

                {/* Submit Button */}
                <button
                    type="submit"
                    disabled={submitting}
                    className={`
                        w-full flex items-center justify-center px-6 py-3 border border-transparent text-base font-medium rounded-md text-white
                        ${submitting 
                            ? 'bg-gray-400 cursor-not-allowed' 
                            : 'bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500'
                        }
                        transition duration-200 ease-in-out transform hover:scale-105
                    `}
                >
                    {submitting ? (
                        <>
                            <Loader2 className="animate-spin h-5 w-5 mr-2" />
                            Redirecting to PayPal...
                        </>
                    ) : (
                        <>
                            <CreditCard className="h-5 w-5 mr-2" />
                            Pay {getCurrencySymbol(order.currency)}{order.net_amount} with PayPal
                        </>
                    )}
                </button>
            </form>

            {/* Security Notice */}
            <div className="p-4 bg-gray-50 dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg">
                <div className="flex items-center space-x-2">
                    <Lock className="h-4 w-4 text-gray-500 dark:text-gray-400" />
                    <span className="text-sm text-gray-600 dark:text-gray-400">
                        Secure Payment Processing — Your payment information is protected with industry-standard encryption.
                    </span>
                </div>
            </div>

            {/* Important Notes */}
            <div className="p-4 bg-amber-50 dark:bg-amber-900/20 rounded-lg border border-amber-200 dark:border-amber-700">
                <h4 className="text-sm font-medium text-amber-800 dark:text-amber-200 mb-2">
                    Important Information
                </h4>
                <ul className="text-sm text-amber-700 dark:text-amber-300 space-y-1">
                    <li>• You will be redirected to PayPal's secure payment page</li>
                    <li>• After payment, you'll be returned to your order</li>
                    <li>• Payment confirmation may take a few minutes</li>
                    <li>• You'll receive an email confirmation once payment is processed</li>
                </ul>
            </div>
        </div>
    );
};

export default PayPalDirectForm; 