import React, { useState, useEffect } from "react";
import { CreditCard, Loader2 } from "lucide-react";
import PaypalIcon from "./Icons/PaypalIcon"; // Adjust the import path as needed

const PaymentMethodSelector = ({
    onSelect,
    selectedMethod,
    orderId,
    paymentMethods = [],
}) => {
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState(null);

    // Select default method if available and none is selected
    useEffect(() => {
        if (paymentMethods.length > 0 && !selectedMethod) {
            onSelect(paymentMethods[0].name);
        }
    }, [paymentMethods, selectedMethod, onSelect]);

    // Handler for payment method selection
    const handleMethodSelect = (methodName) => {
        onSelect(methodName);
    };

    // Helper function to render the appropriate icon
    const renderMethodIcon = (methodName) => {
        switch (methodName) {
            case "paypal":
                return <PaypalIcon className="h-5 w-5" />;
            case "stripe":
                return <CreditCard className="h-5 w-5" />;
            case "paypal_direct":
                return <PaypalIcon className="h-5 w-5" />;
            default:
                return <CreditCard className="h-5 w-5" />;
        }
    };

    if (loading) {
        return (
            <div className="flex items-center justify-center p-4 text-gray-600 dark:text-gray-300">
                <Loader2 className="h-5 w-5 animate-spin mr-2" />
                <span>Loading payment options...</span>
            </div>
        );
    }

    if (error) {
        return (
            <div className="p-4 bg-red-100 dark:bg-red-900 text-red-700 dark:text-red-200 rounded-lg">
                {error}
            </div>
        );
    }

    if (paymentMethods.length === 0) {
        return (
            <div className="p-4 bg-yellow-100 dark:bg-yellow-900 text-yellow-700 dark:text-yellow-200 rounded-lg">
                No payment methods are currently available. Please contact
                support.
            </div>
        );
    }

    return (
        <div className="space-y-4">
            <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                Select Payment Method
            </h3>

            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                {paymentMethods.map((method) => (
                    <div
                        key={method.id}
                        onClick={() => handleMethodSelect(method.name)}
                        className={`
                            flex items-center p-4 border rounded-lg cursor-pointer transition-colors
                            ${
                                selectedMethod === method.name
                                    ? "border-blue-500 bg-blue-50 dark:bg-blue-900 dark:border-blue-700"
                                    : "border-gray-300 hover:border-blue-400 dark:border-gray-700 dark:hover:border-blue-600"
                            }
                        `}
                    >
                        <div
                            className={`
                            flex items-center justify-center w-10 h-10 rounded-full mr-3
                            ${
                                selectedMethod === method.name
                                    ? "bg-blue-500 text-white dark:bg-blue-700"
                                    : "bg-gray-200 text-gray-600 dark:bg-gray-700 dark:text-gray-300"
                            }
                        `}
                        >
                            {renderMethodIcon(method.name)}
                        </div>
                        <div>
                            <div className="font-medium text-gray-900 dark:text-white">
                                {method.display_name}
                            </div>
                            <div className="text-sm text-gray-500 dark:text-gray-400">
                                {method.name === "paypal" &&
                                    "Pay with your PayPal account"}
                                {method.name === "stripe" &&
                                    "Pay with credit or debit card"}
                                {method.name === "paypal_direct" &&
                                    "Pay with PayPal (Direct Form)"}
                            </div>
                        </div>
                    </div>
                ))}
            </div>
        </div>
    );
};

export default PaymentMethodSelector;
