import React, { useState } from 'react';
import { Calculator, Loader2 } from 'lucide-react';
import axios from 'axios';

export default function PriceCalculatorForm({
    formData,
    setFormData,
    academicLevels = [],
    paperTypes = [],
    subjects = [],
    deadlines = [],
    singleSlidePrice = 5,
    isAuthenticated = false,
    onCalculate,
    onClose,
    errors,
    setErrors,
    isCalculating,
    setIsCalculating,
}) {
    const [validationErrors, setValidationErrors] = useState({});

    // Calculate deadline estimate based on selected deadline and user timezone
    const getDeadlineEstimate = () => {
        if (!formData.deadline_id || !deadlines.length) return null;

        const selectedDeadline = deadlines.find(d => d.id === formData.deadline_id);
        if (!selectedDeadline) return null;

        const now = new Date();
        let futureDate = new Date(now);

        // urgency contains the number, duration contains "Hours" or "Days"
        const amount = parseInt(selectedDeadline.urgency);
        const unit = selectedDeadline.duration.toLowerCase();

        if (unit === 'hours') {
            futureDate.setHours(futureDate.getHours() + amount);
        } else if (unit === 'days') {
            futureDate.setDate(futureDate.getDate() + amount);
        }

        // Format the date in user's timezone
        const options = {
            month: 'short',
            day: 'numeric',
            year: 'numeric',
            hour: '2-digit',
            minute: '2-digit',
            timeZoneName: 'short'
        };

        return futureDate.toLocaleString('en-US', options);
    };

    const validateForm = () => {
        const newErrors = {};

        if (!formData.academic_level_id) {
            newErrors.academic_level_id = 'Please select an academic level';
        }
        if (!formData.paper_type_id) {
            newErrors.paper_type_id = 'Please select a paper type';
        }
        if (!formData.subject_id) {
            newErrors.subject_id = 'Please select a subject';
        }
        if (!formData.deadline_id) {
            newErrors.deadline_id = 'Please select a deadline';
        }
        if (!formData.pages || formData.pages < 1) {
            newErrors.pages = 'Pages must be at least 1';
        }

        setValidationErrors(newErrors);
        return Object.keys(newErrors).length === 0;
    };

    const handleCalculate = async (e) => {
        e.preventDefault();

        if (!validateForm()) {
            return;
        }

        setIsCalculating(true);
        setErrors({});

        try {
            const endpoint = isAuthenticated
                ? '/calculate-order-pricing'
                : '/calculate-pricing';

            const requestData = {
                academic_level_id: formData.academic_level_id,
                paper_type_id: formData.paper_type_id,
                deadline_id: formData.deadline_id,
                pages: formData.pages,
                subject_id: formData.subject_id,
                slides: formData.slides || 0,
            };

            // Add spacing and coupon for authenticated users
            if (isAuthenticated) {
                requestData.spacing = formData.spacing || 'double';
                if (formData.coupon_code) {
                    requestData.coupon_code = formData.coupon_code;
                }
            }

            const response = await axios.post(endpoint, requestData, {
                headers: isAuthenticated
                    ? {
                          'X-CSRF-TOKEN': document
                              .querySelector('meta[name="csrf-token"]')
                              ?.getAttribute('content'),
                      }
                    : {},
            });

            if (response.data.success) {
                onCalculate(response.data);
            } else {
                setErrors({
                    general: response.data.message || 'Unable to calculate pricing',
                });
            }
        } catch (error) {
            console.error('Price calculation error:', error);
            setErrors({
                general:
                    error.response?.data?.message ||
                    'Unable to calculate pricing. Please try again.',
            });
        } finally {
            setIsCalculating(false);
        }
    };

    const updateFormData = (field, value) => {
        setFormData((prev) => ({
            ...prev,
            [field]: value,
        }));
        // Clear validation error when user changes field
        if (validationErrors[field]) {
            setValidationErrors((prev) => {
                const newErrors = { ...prev };
                delete newErrors[field];
                return newErrors;
            });
        }
    };

    const incrementPages = () => {
        updateFormData('pages', Math.min((formData.pages || 1) + 1, 100));
    };

    const decrementPages = () => {
        updateFormData('pages', Math.max((formData.pages || 1) - 1, 1));
    };

    return (
        <form onSubmit={handleCalculate} className="space-y-6">
            {/* Error Message */}
            {errors.general && (
                <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg p-4">
                    <p className="text-sm text-red-600 dark:text-red-400">
                        {errors.general}
                    </p>
                </div>
            )}

            {/* Academic Level */}
            <div>
                <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-2">
                    Academic Level <span className="text-red-500">*</span>
                </label>
                <select
                    value={formData.academic_level_id || ''}
                    onChange={(e) =>
                        updateFormData('academic_level_id', parseInt(e.target.value))
                    }
                    className="w-full bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-lg p-3 focus:outline-none focus:ring-2 focus:ring-blue-500 dark:text-gray-200"
                    required
                >
                    <option value="">Select academic level</option>
                    {academicLevels.map((level) => (
                        <option key={level.id} value={level.id}>
                            {level.name}
                        </option>
                    ))}
                </select>
                {validationErrors.academic_level_id && (
                    <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                        {validationErrors.academic_level_id}
                    </p>
                )}
            </div>

            {/* Paper Type */}
            <div>
                <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-2">
                    Paper Type <span className="text-red-500">*</span>
                </label>
                <select
                    value={formData.paper_type_id || ''}
                    onChange={(e) =>
                        updateFormData('paper_type_id', parseInt(e.target.value))
                    }
                    className="w-full bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-lg p-3 focus:outline-none focus:ring-2 focus:ring-blue-500 dark:text-gray-200"
                    required
                >
                    <option value="">Select paper type</option>
                    {paperTypes.map((type) => (
                        <option key={type.id} value={type.id}>
                            {type.name}
                        </option>
                    ))}
                </select>
                {validationErrors.paper_type_id && (
                    <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                        {validationErrors.paper_type_id}
                    </p>
                )}
            </div>

            {/* Subject */}
            <div>
                <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-2">
                    Subject <span className="text-red-500">*</span>
                </label>
                <select
                    value={formData.subject_id || ''}
                    onChange={(e) =>
                        updateFormData('subject_id', parseInt(e.target.value))
                    }
                    className="w-full bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-lg p-3 focus:outline-none focus:ring-2 focus:ring-blue-500 dark:text-gray-200"
                    required
                >
                    <option value="">Select subject</option>
                    {subjects.map((subject) => (
                        <option key={subject.id} value={subject.id}>
                            {subject.name}
                        </option>
                    ))}
                </select>
                {validationErrors.subject_id && (
                    <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                        {validationErrors.subject_id}
                    </p>
                )}
            </div>

            {/* Pages */}
            <div>
                <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-2">
                    Number of Pages <span className="text-red-500">*</span>
                </label>
                <div className="flex items-center gap-2">
                    <button
                        type="button"
                        onClick={decrementPages}
                        className="px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-gray-50 dark:bg-gray-700 hover:bg-gray-100 dark:hover:bg-gray-600 text-gray-700 dark:text-gray-200"
                        aria-label="Decrease pages"
                    >
                        −
                    </button>
                    <input
                        type="number"
                        value={formData.pages || 1}
                        onChange={(e) => {
                            const value = Math.max(1, Math.min(100, parseInt(e.target.value) || 1));
                            updateFormData('pages', value);
                        }}
                        min="1"
                        max="100"
                        className="flex-1 text-center bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-lg p-3 focus:outline-none focus:ring-2 focus:ring-blue-500 dark:text-gray-200"
                        required
                    />
                    <button
                        type="button"
                        onClick={incrementPages}
                        className="px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-gray-50 dark:bg-gray-700 hover:bg-gray-100 dark:hover:bg-gray-600 text-gray-700 dark:text-gray-200"
                        aria-label="Increase pages"
                    >
                        +
                    </button>
                </div>
                <p className="mt-2 text-xs text-gray-500 dark:text-gray-400">
                    Approximately {((formData.pages || 1) * 275).toLocaleString()} - {((formData.pages || 1) * 300).toLocaleString()} words
                </p>
                {validationErrors.pages && (
                    <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                        {validationErrors.pages}
                    </p>
                )}
            </div>

            {/* Deadline */}
            <div>
                <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-2">
                    Deadline <span className="text-red-500">*</span>
                </label>
                <select
                    value={formData.deadline_id || ''}
                    onChange={(e) =>
                        updateFormData('deadline_id', parseInt(e.target.value))
                    }
                    className="w-full bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-lg p-3 focus:outline-none focus:ring-2 focus:ring-blue-500 dark:text-gray-200"
                    required
                >
                    <option value="">Select deadline</option>
                    {deadlines.map((deadline) => (
                        <option key={deadline.id} value={deadline.id}>
                            {deadline.urgency} {deadline.duration}
                        </option>
                    ))}
                </select>
                {formData.deadline_id && getDeadlineEstimate() && (
                    <p className="mt-2 text-xs text-gray-500 dark:text-gray-400">
                        Estimated delivery: {getDeadlineEstimate()}
                    </p>
                )}
                {validationErrors.deadline_id && (
                    <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                        {validationErrors.deadline_id}
                    </p>
                )}
            </div>

            {/* Disclaimer and Live Chat Notice */}
            <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4 space-y-2">
                <p className="text-xs text-gray-600 dark:text-gray-400">
                    <strong className="text-gray-900 dark:text-gray-200">Note:</strong> This is an estimate only and does not constitute a commitment. Final pricing may vary based on specific requirements.
                </p>
                <div className="flex items-start gap-2 pt-2 border-t border-blue-200 dark:border-blue-700">
                    <svg className="w-4 h-4 text-blue-600 dark:text-blue-400 mt-0.5 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                        <path fillRule="evenodd" d="M18 10c0 3.866-3.582 7-8 7a8.841 8.841 0 01-4.083-.98L2 17l1.338-3.123C2.493 12.767 2 11.434 2 10c0-3.866 3.582-7 8-7s8 3.134 8 7zM7 9H5v2h2V9zm8 0h-2v2h2V9zM9 9h2v2H9V9z" clipRule="evenodd" />
                    </svg>
                    <p className="text-xs text-gray-700 dark:text-gray-300">
                        Need clarification? <a href="javascript:void(0)" onClick={() => window.Tawk_API?.maximize()} className="text-blue-600 dark:text-blue-400 hover:underline font-medium">Start a live chat</a> with our support team - we're always online!
                    </p>
                </div>
            </div>

            {/* Action Buttons */}
            <div className="flex gap-3 pt-4">
                <button
                    type="button"
                    onClick={onClose}
                    className="flex-1 px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-800 text-gray-700 dark:text-gray-200 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
                >
                    Cancel
                </button>
                <button
                    type="submit"
                    disabled={isCalculating}
                    className="flex-1 px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded-lg font-medium transition-colors disabled:opacity-50 disabled:cursor-not-allowed flex items-center justify-center gap-2"
                >
                    {isCalculating ? (
                        <>
                            <Loader2 className="w-4 h-4 animate-spin" />
                            Calculating...
                        </>
                    ) : (
                        <>
                            <Calculator className="w-4 h-4" />
                            Calculate Price
                        </>
                    )}
                </button>
            </div>
        </form>
    );
}

