import React, { useState } from 'react';

export default function ReviewsGrid({ title, description, reviews, serviceFilters }) {
    // Keep filtering/sorting logic in React (interactive features)
    const [selectedService, setSelectedService] = useState("all");
    const [selectedRating, setSelectedRating] = useState("all");
    const [sortBy, setSortBy] = useState("newest");
    const [currentPage, setCurrentPage] = useState(1);
    const reviewsPerPage = 12;

    const renderStars = (rating) => {
        return [...Array(5)].map((_, index) => (
            <svg
                key={index}
                className={`w-5 h-5 ${
                    index < rating ? "text-yellow-400" : "text-gray-300"
                }`}
                fill="currentColor"
                viewBox="0 0 20 20"
            >
                <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
            </svg>
        ));
    };

    // Filter and sort reviews
    const filteredReviews = reviews.filter((review) => {
        const serviceMatch = selectedService === "all" || review.serviceSlug === selectedService;
        const ratingMatch = selectedRating === "all" || review.rating.toString() === selectedRating;
        return serviceMatch && ratingMatch;
    });

    const sortedReviews = [...filteredReviews].sort((a, b) => {
        switch (sortBy) {
            case "newest":
                return new Date(b.date) - new Date(a.date);
            case "oldest":
                return new Date(a.date) - new Date(b.date);
            case "highest":
                return b.rating - a.rating;
            case "helpful":
                return b.helpful - a.helpful;
            default:
                return 0;
        }
    });

    // Pagination
    const totalPages = Math.ceil(sortedReviews.length / reviewsPerPage);
    const startIndex = (currentPage - 1) * reviewsPerPage;
    const paginatedReviews = sortedReviews.slice(startIndex, startIndex + reviewsPerPage);

    return (
        <section className="py-16 bg-white">
            <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
                {title && (
                    <div className="text-center mb-12">
                        <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
                            {title}
                        </h2>
                        {description && (
                            <div 
                                className="text-xl text-gray-600 content-area"
                                dangerouslySetInnerHTML={{ __html: description }}
                            />
                        )}
                    </div>
                )}

                {/* Filters */}
                <div className="bg-gray-50 rounded-lg p-6 mb-8">
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                Filter by Service
                            </label>
                            <select
                                value={selectedService}
                                onChange={(e) => {
                                    setSelectedService(e.target.value);
                                    setCurrentPage(1);
                                }}
                                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                            >
                                {serviceFilters && serviceFilters.map((filter) => (
                                    <option key={filter.value} value={filter.value}>
                                        {filter.label} ({filter.count})
                                    </option>
                                ))}
                            </select>
                        </div>

                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                Filter by Rating
                            </label>
                            <select
                                value={selectedRating}
                                onChange={(e) => {
                                    setSelectedRating(e.target.value);
                                    setCurrentPage(1);
                                }}
                                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                            >
                                <option value="all">All Ratings</option>
                                <option value="5">5 Stars Only</option>
                                <option value="4">4 Stars & Up</option>
                                <option value="3">3 Stars & Up</option>
                            </select>
                        </div>

                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                Sort By
                            </label>
                            <select
                                value={sortBy}
                                onChange={(e) => setSortBy(e.target.value)}
                                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                            >
                                <option value="newest">Newest First</option>
                                <option value="oldest">Oldest First</option>
                                <option value="highest">Highest Rated</option>
                                <option value="helpful">Most Helpful</option>
                            </select>
                        </div>
                    </div>
                </div>

                {/* Results Count */}
                <div className="mb-6 text-center text-gray-600">
                    Showing {paginatedReviews.length} of {sortedReviews.length} reviews
                </div>

                {/* Review Cards Grid */}
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 mb-12">
                    {paginatedReviews.map((review) => (
                        <div
                            key={review.id}
                            className="bg-white border border-gray-200 rounded-lg p-6 hover:shadow-lg transition-shadow"
                        >
                            {/* Header */}
                            <div className="flex items-center justify-between mb-4">
                                <div className="flex">{renderStars(review.rating)}</div>
                                {review.verified && (
                                    <span className="text-xs bg-green-100 text-green-800 px-2 py-1 rounded">
                                        Verified
                                    </span>
                                )}
                            </div>

                            {/* Title */}
                            <h3 className="text-lg font-bold text-gray-900 mb-2">
                                {review.title}
                            </h3>

                            {/* Review Text */}
                            <p className="text-gray-600 mb-4 line-clamp-4">
                                {review.review}
                            </p>

                            {/* Service Badge */}
                            <div className="mb-4">
                                <a
                                    href={`/services/${review.serviceSlug}`}
                                    className="inline-block bg-blue-50 text-blue-700 px-3 py-1 rounded-full text-sm hover:bg-blue-100 transition-colors"
                                >
                                    {review.service}
                                </a>
                            </div>

                            {/* Author Info */}
                            <div className="border-t pt-4">
                                <div className="flex items-start justify-between">
                                    <div className="flex-1">
                                        <div className="font-semibold text-gray-900">
                                            {review.name}
                                        </div>
                                        {review.course && (
                                            <div className="text-sm text-gray-600">
                                                {review.course}
                                            </div>
                                        )}
                                        {review.university && (
                                            <div className="text-sm text-gray-500">
                                                {review.university}
                                            </div>
                                        )}
                                        {review.grade && (
                                            <div className="text-sm text-green-600 font-semibold mt-1">
                                                Grade: {review.grade}
                                            </div>
                                        )}
                                    </div>
                                </div>

                                {/* Footer */}
                                <div className="flex items-center justify-between mt-4 text-sm text-gray-500">
                                    <span>{new Date(review.date).toLocaleDateString()}</span>
                                    {review.helpful > 0 && (
                                        <span>{review.helpful} found helpful</span>
                                    )}
                                </div>
                            </div>
                        </div>
                    ))}
                </div>

                {/* Pagination */}
                {totalPages > 1 && (
                    <div className="flex justify-center items-center space-x-2">
                        <button
                            onClick={() => setCurrentPage(Math.max(1, currentPage - 1))}
                            disabled={currentPage === 1}
                            className="px-4 py-2 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
                        >
                            Previous
                        </button>

                        {[...Array(totalPages)].map((_, index) => {
                            const page = index + 1;
                            if (
                                page === 1 ||
                                page === totalPages ||
                                (page >= currentPage - 1 && page <= currentPage + 1)
                            ) {
                                return (
                                    <button
                                        key={page}
                                        onClick={() => setCurrentPage(page)}
                                        className={`px-4 py-2 rounded-lg ${
                                            currentPage === page
                                                ? "bg-blue-600 text-white"
                                                : "bg-white border border-gray-300 hover:bg-gray-50"
                                        }`}
                                    >
                                        {page}
                                    </button>
                                );
                            } else if (
                                page === currentPage - 2 ||
                                page === currentPage + 2
                            ) {
                                return <span key={page}>...</span>;
                            }
                            return null;
                        })}

                        <button
                            onClick={() => setCurrentPage(Math.min(totalPages, currentPage + 1))}
                            disabled={currentPage === totalPages}
                            className="px-4 py-2 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
                        >
                            Next
                        </button>
                    </div>
                )}
            </div>
        </section>
    );
}




