import React, { useState, useEffect } from 'react';
import { useForm } from '@inertiajs/react';
import { 
    AlertCircle, 
    Clock, 
    FileText, 
    AlertTriangle,
    CheckCircle,
    XCircle
} from 'lucide-react';

const RevisionRequestForm = ({ 
    order, 
    writers = [], 
    onSubmit, 
    onCancel,
    initialData = null,
    isEditing = false 
}) => {
    const [selectedPriority, setSelectedPriority] = useState(initialData?.revision_priority || 'medium');
    const [selectedType, setSelectedType] = useState(initialData?.revision_type || 'content');
    const [customDeadline, setCustomDeadline] = useState(false);

    const { data, setData, post, put, processing, errors, reset } = useForm({
        order_id: order?.id || '',
        revision_type: selectedType,
        revision_priority: selectedPriority,
        revision_section: initialData?.revision_section || '',
        revision_reason: initialData?.revision_reason || '',
        specific_instructions: initialData?.specific_instructions || '',
        assigned_to: initialData?.assigned_to || '',
        revision_deadline: initialData?.revision_deadline || '',
        revision_admin_notes: initialData?.revision_admin_notes || '',
        is_client_requested: initialData?.is_client_requested || false,
        client_notes: initialData?.client_notes || '',
    });

    const priorityOptions = [
        { value: 'low', label: 'Low Priority', description: '24 hours', color: 'text-green-600', bgColor: 'bg-green-100' },
        { value: 'medium', label: 'Medium Priority', description: '8 hours', color: 'text-yellow-600', bgColor: 'bg-yellow-100' },
        { value: 'high', label: 'High Priority', description: '4 hours', color: 'text-red-600', bgColor: 'bg-red-100' },
    ];

    const typeOptions = [
        { value: 'content', label: 'Content Revision', description: 'Changes to the main content, arguments, or structure', icon: FileText },
        { value: 'formatting', label: 'Formatting Revision', description: 'Changes to layout, spacing, fonts, or visual presentation', icon: FileText },
        { value: 'research', label: 'Research Revision', description: 'Additional research, sources, or data needed', icon: FileText },
        { value: 'citation', label: 'Citation Revision', description: 'Citation format, references, or bibliography updates', icon: FileText },
    ];

    useEffect(() => {
        setData(prev => ({
            ...prev,
            revision_type: selectedType,
            revision_priority: selectedPriority,
        }));
    }, [selectedType, selectedPriority]);

    const handleSubmit = (e) => {
        e.preventDefault();
        
        if (isEditing && initialData) {
            put(`/admin/revisions/${initialData.id}`, {
                onSuccess: () => {
                    if (onSubmit) onSubmit();
                    reset();
                },
            });
        } else {
            post('/admin/revisions', {
                onSuccess: () => {
                    if (onSubmit) onSubmit();
                    reset();
                },
            });
        }
    };

    const getPriorityColor = (priority) => {
        return priorityOptions.find(p => p.value === priority)?.color || 'text-gray-600';
    };

    const getPriorityBgColor = (priority) => {
        return priorityOptions.find(p => p.value === priority)?.bgColor || 'bg-gray-100';
    };

    const getTypeIcon = (type) => {
        const option = typeOptions.find(t => t.value === type);
        return option ? option.icon : FileText;
    };

    const getTypeDescription = (type) => {
        const option = typeOptions.find(t => t.value === type);
        return option ? option.description : '';
    };

    return (
        <div className="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700">
            <div className="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                    {isEditing ? 'Edit Revision Request' : 'Request Revision'}
                </h3>
                {order && (
                    <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                        Order #{order.order_number} - {order.title}
                    </p>
                )}
            </div>

            <form onSubmit={handleSubmit} className="p-6 space-y-6">
                {/* Priority Selection */}
                <div>
                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-3">
                        Revision Priority
                    </label>
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-3">
                        {priorityOptions.map((priority) => (
                            <button
                                key={priority.value}
                                type="button"
                                onClick={() => setSelectedPriority(priority.value)}
                                className={`p-4 rounded-lg border-2 transition-all duration-200 ${
                                    selectedPriority === priority.value
                                        ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20'
                                        : 'border-gray-200 dark:border-gray-600 hover:border-gray-300 dark:hover:border-gray-500'
                                }`}
                            >
                                <div className="text-left">
                                    <div className={`font-medium ${priority.color}`}>
                                        {priority.label}
                                    </div>
                                    <div className="text-sm text-gray-500 dark:text-gray-400 mt-1">
                                        {priority.description}
                                    </div>
                                </div>
                            </button>
                        ))}
                    </div>
                    {errors.revision_priority && (
                        <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                            {errors.revision_priority}
                        </p>
                    )}
                </div>

                {/* Type Selection */}
                <div>
                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-3">
                        Revision Type
                    </label>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                        {typeOptions.map((type) => {
                            const Icon = type.icon;
                            return (
                                <button
                                    key={type.value}
                                    type="button"
                                    onClick={() => setSelectedType(type.value)}
                                    className={`p-4 rounded-lg border-2 transition-all duration-200 ${
                                        selectedType === type.value
                                            ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20'
                                            : 'border-gray-200 dark:border-gray-600 hover:border-gray-300 dark:hover:border-gray-500'
                                    }`}
                                >
                                    <div className="text-left">
                                        <div className="flex items-center">
                                            <Icon className="w-5 h-5 text-gray-500 dark:text-gray-400 mr-2" />
                                            <span className="font-medium text-gray-900 dark:text-white">
                                                {type.label}
                                            </span>
                                        </div>
                                        <div className="text-sm text-gray-500 dark:text-gray-400 mt-2">
                                            {type.description}
                                        </div>
                                    </div>
                                </button>
                            );
                        })}
                    </div>
                    {errors.revision_type && (
                        <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                            {errors.revision_type}
                        </p>
                    )}
                </div>

                {/* Section-specific feedback */}
                <div>
                    <label htmlFor="revision_section" className="block text-sm font-medium text-gray-700 dark:text-gray-300">
                        Specific Section (Optional)
                    </label>
                    <input
                        type="text"
                        id="revision_section"
                        value={data.revision_section}
                        onChange={(e) => setData('revision_section', e.target.value)}
                        placeholder="e.g., Introduction, Conclusion, Methodology, etc."
                        className="mt-1 block w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm"
                    />
                    {errors.revision_section && (
                        <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                            {errors.revision_section}
                        </p>
                    )}
                </div>

                {/* Revision Reason */}
                <div>
                    <label htmlFor="revision_reason" className="block text-sm font-medium text-gray-700 dark:text-gray-300">
                        Revision Reason <span className="text-red-500">*</span>
                    </label>
                    <textarea
                        id="revision_reason"
                        rows={3}
                        value={data.revision_reason}
                        onChange={(e) => setData('revision_reason', e.target.value)}
                        placeholder="Please explain why this revision is needed..."
                        className="mt-1 block w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm"
                        required
                    />
                    {errors.revision_reason && (
                        <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                            {errors.revision_reason}
                        </p>
                    )}
                </div>

                {/* Specific Instructions */}
                <div>
                    <label htmlFor="specific_instructions" className="block text-sm font-medium text-gray-700 dark:text-gray-300">
                        Specific Instructions (Optional)
                    </label>
                    <textarea
                        id="specific_instructions"
                        rows={4}
                        value={data.specific_instructions}
                        onChange={(e) => setData('specific_instructions', e.target.value)}
                        placeholder="Provide detailed instructions for the writer on how to complete this revision..."
                        className="mt-1 block w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm"
                    />
                    {errors.specific_instructions && (
                        <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                            {errors.specific_instructions}
                        </p>
                    )}
                </div>

                {/* Writer Assignment */}
                {writers.length > 0 && (
                    <div>
                        <label htmlFor="assigned_to" className="block text-sm font-medium text-gray-700 dark:text-gray-300">
                            Assign to Writer (Optional)
                        </label>
                        <select
                            id="assigned_to"
                            value={data.assigned_to}
                            onChange={(e) => setData('assigned_to', e.target.value)}
                            className="mt-1 block w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm"
                        >
                            <option value="">Select a writer (or leave blank for auto-assignment)</option>
                            {writers.map((writer) => (
                                <option key={writer.id} value={writer.id}>
                                    {writer.name} ({writer.email})
                                </option>
                            ))}
                        </select>
                        {errors.assigned_to && (
                            <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                                {errors.assigned_to}
                            </p>
                        )}
                    </div>
                )}

                {/* Deadline */}
                <div>
                    <div className="flex items-center justify-between">
                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">
                            Revision Deadline
                        </label>
                        <div className="flex items-center space-x-2">
                            <input
                                type="checkbox"
                                id="custom_deadline"
                                checked={customDeadline}
                                onChange={(e) => setCustomDeadline(e.target.checked)}
                                className="rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                            />
                            <label htmlFor="custom_deadline" className="text-sm text-gray-500 dark:text-gray-400">
                                Set custom deadline
                            </label>
                        </div>
                    </div>
                    
                    {customDeadline ? (
                        <input
                            type="datetime-local"
                            value={data.revision_deadline}
                            onChange={(e) => setData('revision_deadline', e.target.value)}
                            className="mt-1 block w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm"
                        />
                    ) : (
                        <div className="mt-1 p-3 bg-gray-50 dark:bg-gray-700 rounded-md">
                            <div className="flex items-center">
                                <Clock className="w-4 h-4 text-gray-400 mr-2" />
                                <span className="text-sm text-gray-600 dark:text-gray-300">
                                    Auto-calculated deadline: {(() => {
                                        const hours = selectedPriority === 'high' ? 4 : selectedPriority === 'medium' ? 8 : 24;
                                        const deadline = new Date();
                                        deadline.setHours(deadline.getHours() + hours);
                                        return deadline.toLocaleString();
                                    })()}
                                </span>
                            </div>
                        </div>
                    )}
                    {errors.revision_deadline && (
                        <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                            {errors.revision_deadline}
                        </p>
                    )}
                </div>

                {/* Admin Notes */}
                <div>
                    <label htmlFor="revision_admin_notes" className="block text-sm font-medium text-gray-700 dark:text-gray-300">
                        Admin Notes (Optional)
                    </label>
                    <textarea
                        id="revision_admin_notes"
                        rows={3}
                        value={data.revision_admin_notes}
                        onChange={(e) => setData('revision_admin_notes', e.target.value)}
                        placeholder="Internal notes for admin team..."
                        className="mt-1 block w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm"
                    />
                    {errors.revision_admin_notes && (
                        <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                            {errors.revision_admin_notes}
                        </p>
                    )}
                </div>

                {/* Client Requested */}
                <div className="flex items-center space-x-3">
                    <input
                        type="checkbox"
                        id="is_client_requested"
                        checked={data.is_client_requested}
                        onChange={(e) => setData('is_client_requested', e.target.checked)}
                        className="rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                    />
                    <label htmlFor="is_client_requested" className="text-sm font-medium text-gray-700 dark:text-gray-300">
                        This revision was requested by the client
                    </label>
                </div>

                {/* Client Notes */}
                {data.is_client_requested && (
                    <div>
                        <label htmlFor="client_notes" className="block text-sm font-medium text-gray-700 dark:text-gray-300">
                            Client Feedback
                        </label>
                        <textarea
                            id="client_notes"
                            rows={3}
                            value={data.client_notes}
                            onChange={(e) => setData('client_notes', e.target.value)}
                            placeholder="Client's specific feedback or requirements..."
                            className="mt-1 block w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm"
                        />
                        {errors.client_notes && (
                            <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                                {errors.client_notes}
                            </p>
                        )}
                    </div>
                )}

                {/* Priority Warning */}
                {selectedPriority === 'high' && (
                    <div className="p-4 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-md">
                        <div className="flex items-start">
                            <AlertTriangle className="w-5 h-5 text-red-400 mt-0.5 mr-2 flex-shrink-0" />
                            <div>
                                <h4 className="text-sm font-medium text-red-800 dark:text-red-200">
                                    High Priority Revision
                                </h4>
                                <p className="text-sm text-red-700 dark:text-red-300 mt-1">
                                    This revision will be marked as high priority and assigned a 4-hour deadline. 
                                    Please ensure this is truly urgent to avoid disrupting other work.
                                </p>
                            </div>
                        </div>
                    </div>
                )}

                {/* Form Actions */}
                <div className="flex items-center justify-end space-x-3 pt-4 border-t border-gray-200 dark:border-gray-700">
                    {onCancel && (
                        <button
                            type="button"
                            onClick={onCancel}
                            className="px-4 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-md hover:bg-gray-50 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                        >
                            Cancel
                        </button>
                    )}
                    <button
                        type="submit"
                        disabled={processing}
                        className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 disabled:opacity-50 disabled:cursor-not-allowed"
                    >
                        {processing ? (
                            <>
                                <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                                {isEditing ? 'Updating...' : 'Creating...'}
                            </>
                        ) : (
                            <>
                                {isEditing ? (
                                    <>
                                        <CheckCircle className="w-4 h-4 mr-2" />
                                        Update Revision
                                    </>
                                ) : (
                                    <>
                                        <AlertCircle className="w-4 h-4 mr-2" />
                                        Request Revision
                                    </>
                                )}
                            </>
                        )}
                    </button>
                </div>
            </form>
        </div>
    );
};

export default RevisionRequestForm; 