import { motion } from "framer-motion";
import { Link } from "@inertiajs/react";
import {
    Clock,
    Award,
    BookOpen,
    MessageSquare,
    GraduationCap,
    FileText,
    PenTool,
    Users,
    CheckCircle,
    Star,
    ArrowRight,
} from "lucide-react";

export default function Services({ fadeIn, staggerContainer }) {
    const handleServiceClick = (serviceName, serviceCategory) => {
        if (typeof window.trackServiceView === "function") {
            window.trackServiceView(serviceName, serviceCategory);
        }
    };

    const handleCTAClick = (ctaText, serviceName) => {
        if (typeof window.trackCTAClick === "function") {
            window.trackCTAClick(ctaText, "services_section", serviceName);
        }
    };

    const mainServices = [
        {
            title: "Dissertation Writing Help",
            shortTitle: "Dissertation Writing",
            description:
                "Comprehensive PhD dissertation writing assistance with expert research methodology, literature review, and data analysis. Our professional essay writers specialize in doctoral-level academic writing services.",
            icon: <GraduationCap className="h-8 w-8" />,
            features: [
                "PhD-level research and writing",
                "Comprehensive literature review",
                "Statistical analysis support",
                "Multiple revision rounds",
                "Plagiarism-free guarantee",
            ],
            pricing: "Starting from $11/page",
            deliveryTime: "2-30 days",
            link: "/dissertation-writing",
            color: "bg-blue-600",
            hoverColor: "hover:bg-blue-700",
            popular: true,
        },
        {
            title: "Research Paper Writing Service",
            shortTitle: "Research Papers",
            description:
                "Professional research paper writing service with in-depth analysis, proper citations, and academic formatting. Our custom academic papers meet the highest scholarly standards.",
            icon: <FileText className="h-8 w-8" />,
            features: [
                "Original research and analysis",
                "Proper academic citations",
                "Multiple formatting styles",
                "Peer-reviewed sources",
                "Quality assurance review",
            ],
            pricing: "Starting from $11/page",
            deliveryTime: "1-14 days",
            link: "/research-papers",
            color: "bg-green-600",
            hoverColor: "hover:bg-green-700",
            popular: false,
        },
        {
            title: "Professional Essay Writers",
            shortTitle: "Essay Writing",
            description:
                "Expert essay writing services by professional essay writers with advanced degrees. We provide custom academic papers for all educational levels and subjects.",
            icon: <PenTool className="h-8 w-8" />,
            features: [
                "All academic levels covered",
                "Subject matter experts",
                "Custom essay writing",
                "Timely delivery guaranteed",
                "Direct writer communication",
            ],
            pricing: "Starting from $11/page",
            deliveryTime: "3 hours - 7 days",
            link: "/essay-writing",
            color: "bg-purple-600",
            hoverColor: "hover:bg-purple-700",
            popular: false,
        },
        {
            title: "Custom Academic Papers",
            shortTitle: "Custom Papers",
            description:
                "Tailored academic writing services for specific requirements. Our dissertation writing help extends to thesis proposals, case studies, and specialized academic documents.",
            icon: <BookOpen className="h-8 w-8" />,
            features: [
                "Customized to requirements",
                "Various paper types",
                "Flexible deadlines",
                "Expert consultation",
                "Unlimited revisions",
            ],
            pricing: "Starting from $15/page",
            deliveryTime: "6 hours - 30 days",
            link: "/custom-papers",
            color: "bg-orange-600",
            hoverColor: "hover:bg-orange-700",
            popular: false,
        },
    ];

    const additionalServices = [
        {
            title: "Urgent Academic Writing",
            description:
                "Quick turnaround solutions for time-sensitive academic projects without compromising on quality.",
            icon: <Clock className="h-6 w-6" />,
            link: "/urgent-writing",
        },
        {
            title: "Thesis Proposal Writing",
            description:
                "Expert assistance with thesis and dissertation proposals that get approved by academic committees.",
            icon: <Award className="h-6 w-6" />,
            link: "/thesis-proposals",
        },
        {
            title: "Literature Review Service",
            description:
                "Comprehensive literature reviews with critical analysis and synthesis of academic sources.",
            icon: <BookOpen className="h-6 w-6" />,
            link: "/literature-reviews",
        },
        {
            title: "Academic Editing & Proofreading",
            description:
                "Professional editing and proofreading services to polish your academic writing to perfection.",
            icon: <MessageSquare className="h-6 w-6" />,
            link: "/services/editing-proofreading",
        },
    ];

    return (
        <>
            <section
                id="services"
                className="py-20 bg-gray-50 dark:bg-gray-900 w-full"
            >
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    {/* Section Header */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true, margin: "-100px" }}
                        variants={fadeIn}
                        className="text-center mb-16"
                    >
                        <h2 className="text-4xl md:text-5xl font-bold text-gray-900 dark:text-white mb-4">
                            Our Academic Writing Services
                        </h2>
                        <div className="mt-4 h-1 w-24 bg-blue-600 mx-auto rounded-full"></div>
                        <p className="mt-6 max-w-3xl mx-auto text-xl text-gray-600 dark:text-gray-300 leading-relaxed">
                            We provide comprehensive{" "}
                            <strong>academic writing services</strong> across
                            all disciplines. Our team of{" "}
                            <strong>professional essay writers</strong> delivers
                            exceptional
                            <strong> dissertation writing help</strong>,{" "}
                            <strong>research paper writing service</strong>, and{" "}
                            <strong>custom academic papers</strong> tailored to
                            your specific needs.
                        </p>

                        {/* Trust Indicators */}
                        <div className="mt-8 flex flex-wrap justify-center items-center space-x-8 text-sm text-gray-500 dark:text-gray-400">
                            <div className="flex items-center">
                                <Users className="w-4 h-4 mr-1" />
                                <span>180+ Expert Writers</span>
                            </div>
                            <div className="flex items-center">
                                <CheckCircle className="w-4 h-4 mr-1" />
                                <span>98% Success Rate</span>
                            </div>
                            <div className="flex items-center">
                                <Star className="w-4 h-4 mr-1" />
                                <span>4.9/5 Average Rating</span>
                            </div>
                        </div>
                    </motion.div>

                    {/* Main Services Grid */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true, margin: "-50px" }}
                        variants={staggerContainer}
                        className="grid md:grid-cols-2 lg:grid-cols-2 gap-8 mb-16"
                    >
                        {mainServices.map((service, index) => (
                            <motion.div
                                key={index}
                                variants={fadeIn}
                                className="bg-white dark:bg-gray-800 rounded-xl shadow-lg hover:shadow-2xl transition-all duration-300 transform hover:-translate-y-2 border border-gray-100 dark:border-gray-700 hover:border-blue-300 dark:hover:border-blue-700 overflow-hidden relative"
                                onClick={() =>
                                    handleServiceClick(
                                        service.title,
                                        "main_academic_services"
                                    )
                                }
                            >
                                {/* Popular Badge */}
                                {service.popular && (
                                    <div className="absolute top-4 right-4 bg-red-500 text-white text-xs font-bold px-3 py-1 rounded-full">
                                        Most Popular
                                    </div>
                                )}

                                <div className="p-8">
                                    {/* Service Icon */}
                                    <div
                                        className={`flex items-center justify-center mx-auto h-16 w-16 rounded-full ${service.color} text-white mb-6 ${service.hoverColor} transition-colors duration-300`}
                                    >
                                        {service.icon}
                                    </div>

                                    {/* Service Title */}
                                    <h3 className="text-2xl font-bold text-gray-900 dark:text-white mb-4 text-center">
                                        {service.shortTitle}
                                    </h3>

                                    {/* Service Description */}
                                    <p className="text-gray-600 dark:text-gray-300 text-center mb-6 leading-relaxed">
                                        {service.description}
                                    </p>

                                    {/* Pricing and Delivery */}
                                    <div className="flex justify-between items-center mb-6 text-sm">
                                        <div className="text-center">
                                            <p className="text-gray-500 dark:text-gray-400">
                                                Starting Price
                                            </p>
                                            <p className="font-semibold text-blue-600 dark:text-blue-400">
                                                {service.pricing}
                                            </p>
                                        </div>
                                        <div className="text-center">
                                            <p className="text-gray-500 dark:text-gray-400">
                                                Delivery Time
                                            </p>
                                            <p className="font-semibold text-green-600 dark:text-green-400">
                                                {service.deliveryTime}
                                            </p>
                                        </div>
                                    </div>

                                    {/* Service Features */}
                                    <div className="mb-6">
                                        <h4 className="font-semibold text-gray-900 dark:text-white mb-3">
                                            What's Included:
                                        </h4>
                                        <ul className="space-y-2">
                                            {service.features.map(
                                                (feature, idx) => (
                                                    <li
                                                        key={idx}
                                                        className="flex items-center text-sm text-gray-600 dark:text-gray-300"
                                                    >
                                                        <CheckCircle className="w-4 h-4 text-green-500 mr-2 flex-shrink-0" />
                                                        {feature}
                                                    </li>
                                                )
                                            )}
                                        </ul>
                                    </div>

                                    {/* CTA Button */}
                                    <div className="text-center">
                                        <Link
                                            href={service.link}
                                            className={`inline-flex items-center justify-center w-full ${service.color} ${service.hoverColor} text-white py-3 px-6 rounded-lg font-semibold transition-all duration-300 transform hover:scale-105 shadow-md hover:shadow-lg`}
                                            onClick={() =>
                                                handleCTAClick(
                                                    `Order ${service.shortTitle}`,
                                                    service.title
                                                )
                                            }
                                        >
                                            Order {service.shortTitle}
                                            <ArrowRight className="ml-2 w-4 h-4" />
                                        </Link>
                                    </div>
                                </div>
                            </motion.div>
                        ))}
                    </motion.div>

                    {/* Additional Services */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true, margin: "-50px" }}
                        variants={fadeIn}
                        className="mb-12"
                    >
                        <h3 className="text-3xl font-bold text-gray-900 dark:text-white text-center mb-8">
                            Additional Academic Writing Services
                        </h3>
                        <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
                            {additionalServices.map((service, index) => (
                                <motion.div
                                    key={index}
                                    variants={fadeIn}
                                    className="bg-white dark:bg-gray-800 rounded-lg p-6 shadow-md hover:shadow-lg transition-all duration-300 transform hover:-translate-y-1 border border-gray-100 dark:border-gray-700 hover:border-blue-300 dark:hover:border-blue-700"
                                    onClick={() =>
                                        handleServiceClick(
                                            service.title,
                                            "additional_academic_services"
                                        )
                                    }
                                >
                                    <Link href={service.link} className="block">
                                        <div className="flex items-center justify-center mx-auto h-12 w-12 rounded-full bg-gray-100 dark:bg-gray-700 text-blue-600 dark:text-blue-400 mb-4">
                                            {service.icon}
                                        </div>
                                        <h4 className="text-lg font-semibold text-gray-900 dark:text-white mb-3 text-center">
                                            {service.title}
                                        </h4>
                                        <p className="text-gray-600 dark:text-gray-300 text-sm text-center leading-relaxed">
                                            {service.description}
                                        </p>
                                        <div className="mt-4 text-center">
                                            <span className="inline-flex items-center text-blue-600 dark:text-blue-400 text-sm font-medium hover:text-blue-700 dark:hover:text-blue-300 transition-colors">
                                                Learn More
                                                <ArrowRight className="ml-1 w-3 h-3" />
                                            </span>
                                        </div>
                                    </Link>
                                </motion.div>
                            ))}
                        </div>
                    </motion.div>

                    {/* Call to Action Section */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="bg-gradient-to-r from-blue-600 to-purple-600 rounded-2xl p-8 md:p-12 text-center text-white"
                    >
                        <h3 className="text-3xl md:text-4xl font-bold mb-4">
                            Need Help with Your Academic Writing?
                        </h3>
                        <p className="text-xl mb-8 text-blue-100 max-w-3xl mx-auto">
                            Our <strong>professional essay writers</strong> are
                            ready to assist you with any academic writing
                            project. Get expert{" "}
                            <strong>dissertation writing help</strong> and{" "}
                            <strong>research paper writing service</strong>{" "}
                            today.
                        </p>
                        <div className="flex flex-col sm:flex-row justify-center space-y-4 sm:space-y-0 sm:space-x-6">
                            <Link
                                href="/place-order"
                                className="bg-white text-blue-600 hover:bg-gray-100 px-8 py-4 rounded-lg font-semibold transition-all duration-300 transform hover:scale-105 shadow-lg"
                                onClick={() =>
                                    handleCTAClick(
                                        "Start Your Order",
                                        "services_cta_section"
                                    )
                                }
                            >
                                Start Your Order Now
                            </Link>
                            <Link
                                href="/consultation"
                                className="border-2 border-white text-white hover:bg-white hover:text-blue-600 px-8 py-4 rounded-lg font-semibold transition-all duration-300 transform hover:scale-105"
                                onClick={() =>
                                    handleCTAClick(
                                        "Free Consultation",
                                        "services_cta_section"
                                    )
                                }
                            >
                                Get Free Consultation
                            </Link>
                        </div>

                        {/* Service Guarantees */}
                        <div className="mt-8 grid grid-cols-1 md:grid-cols-3 gap-6 text-center">
                            <div className="flex flex-col items-center">
                                <CheckCircle className="w-8 h-8 text-green-300 mb-2" />
                                <h4 className="font-semibold mb-1">
                                    100% Original Content
                                </h4>
                                <p className="text-blue-100 text-sm">
                                    Plagiarism-free guarantee with every order
                                </p>
                            </div>
                            <div className="flex flex-col items-center">
                                <Clock className="w-8 h-8 text-yellow-300 mb-2" />
                                <h4 className="font-semibold mb-1">
                                    On-Time Delivery
                                </h4>
                                <p className="text-blue-100 text-sm">
                                    Never miss a deadline with our reliable
                                    service
                                </p>
                            </div>
                            <div className="flex flex-col items-center">
                                <Award className="w-8 h-8 text-orange-300 mb-2" />
                                <h4 className="font-semibold mb-1">
                                    Money-Back Guarantee
                                </h4>
                                <p className="text-blue-100 text-sm">
                                    Full refund if not satisfied with quality
                                </p>
                            </div>
                        </div>
                    </motion.div>

                    {/* Academic Disciplines */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="mt-16 text-center"
                    >
                        <h3 className="text-2xl font-bold text-gray-900 dark:text-white mb-6">
                            Academic Disciplines We Cover
                        </h3>
                        <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 gap-4 text-sm">
                            {[
                                "Business & Management",
                                "Psychology",
                                "Nursing & Healthcare",
                                "Education",
                                "Engineering",
                                "Literature & English",
                                "History",
                                "Political Science",
                                "Sociology",
                                "Economics",
                                "Computer Science",
                                "Law & Legal Studies",
                                "Philosophy",
                                "Biology & Life Sciences",
                                "Chemistry",
                                "Physics",
                                "Mathematics",
                                "Art & Design",
                                "Communications",
                                "Environmental Science",
                                "Marketing",
                                "Finance",
                                "International Relations",
                                "Anthropology",
                            ].map((discipline, index) => (
                                <div
                                    key={index}
                                    className="bg-white dark:bg-gray-800 rounded-lg p-3 shadow-sm border border-gray-200 dark:border-gray-700 hover:border-blue-300 dark:hover:border-blue-600 transition-colors duration-200"
                                >
                                    <span className="text-gray-700 dark:text-gray-300 font-medium">
                                        {discipline}
                                    </span>
                                </div>
                            ))}
                        </div>
                        <p className="mt-6 text-gray-600 dark:text-gray-400">
                            Don't see your subject?{" "}
                            <Link
                                href="/contact"
                                className="text-blue-600 dark:text-blue-400 hover:underline"
                            >
                                Contact us
                            </Link>{" "}
                            - we cover 35+ academic disciplines!
                        </p>
                    </motion.div>

                    {/* Process Overview */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={staggerContainer}
                        className="mt-16"
                    >
                        <h3 className="text-3xl font-bold text-gray-900 dark:text-white text-center mb-12">
                            How Our Academic Writing Service Works
                        </h3>
                        <div className="grid md:grid-cols-4 gap-8">
                            {[
                                {
                                    step: "01",
                                    title: "Place Your Order",
                                    description:
                                        "Submit your academic writing requirements and choose your preferred writer level",
                                    icon: <FileText className="w-6 h-6" />,
                                },
                                {
                                    step: "02",
                                    title: "Writer Assignment",
                                    description:
                                        "We match you with a qualified professional essay writer in your subject area",
                                    icon: <Users className="w-6 h-6" />,
                                },
                                {
                                    step: "03",
                                    title: "Writing Process",
                                    description:
                                        "Your writer conducts research and crafts your custom academic paper with regular updates",
                                    icon: <PenTool className="w-6 h-6" />,
                                },
                                {
                                    step: "04",
                                    title: "Quality Delivery",
                                    description:
                                        "Receive your completed paper with plagiarism report and unlimited revisions",
                                    icon: <Award className="w-6 h-6" />,
                                },
                            ].map((process, index) => (
                                <motion.div
                                    key={index}
                                    variants={fadeIn}
                                    className="text-center relative"
                                >
                                    {/* Step Number */}
                                    <div className="bg-blue-600 text-white w-12 h-12 rounded-full flex items-center justify-center font-bold text-lg mx-auto mb-4">
                                        {process.step}
                                    </div>

                                    {/* Connector Line (except for last item) */}
                                    {index < 3 && (
                                        <div className="hidden md:block absolute top-6 left-1/2 w-full h-0.5 bg-gray-300 dark:bg-gray-600 transform translate-x-6"></div>
                                    )}

                                    {/* Process Icon */}
                                    <div className="bg-gray-100 dark:bg-gray-800 w-16 h-16 rounded-full flex items-center justify-center mx-auto mb-4 text-blue-600 dark:text-blue-400">
                                        {process.icon}
                                    </div>

                                    {/* Process Title */}
                                    <h4 className="text-xl font-semibold text-gray-900 dark:text-white mb-3">
                                        {process.title}
                                    </h4>

                                    {/* Process Description */}
                                    <p className="text-gray-600 dark:text-gray-300 leading-relaxed">
                                        {process.description}
                                    </p>
                                </motion.div>
                            ))}
                        </div>
                    </motion.div>
                </div>
            </section>
        </>
    );
}
