import { motion } from "framer-motion";
import { useState } from "react";
import { ChevronDown } from "lucide-react";
import PropTypes from "prop-types";

export default function ServiceFAQs({
    title = "Frequently Asked Questions",
    description = "Find answers to common questions about our service",
    faqs = [],
}) {
    const [activeFaq, setActiveFaq] = useState(null);

    // Default FAQs if none provided
    const defaultFaqs = [
        {
            question: "How do you ensure academic integrity?",
            answer: "We maintain strict academic integrity by providing original, plagiarism-free content. Every paper is written from scratch, properly cited using your preferred style (APA, MLA, Chicago, etc.), and checked through professional plagiarism detection software before delivery.",
        },
        {
            question: "What qualifications do your writers have?",
            answer: "Our writers hold advanced degrees (Master's and PhDs) from accredited universities across various disciplines. We match your project with a writer who has relevant subject-matter expertise and academic background in your specific field.",
        },
        {
            question: "How confidential is your service?",
            answer: "We guarantee complete confidentiality. Your personal information is protected by advanced encryption, and we never share your details with third parties. Your transactions and communications remain private, and we operate with a strict non-disclosure policy.",
        },
        {
            question: "Do you offer revisions?",
            answer: "Yes, we provide free revisions within 14 days of delivery. If any aspects of your paper don't meet the requirements you specified, our writers will revise the work until you're completely satisfied, at no additional cost.",
        },
        {
            question: "What are your pricing policies?",
            answer: "Our pricing is transparent and based on several factors: academic level, deadline, page count, and complexity. We offer competitive rates with no hidden fees. Longer deadlines generally mean lower prices, and we frequently provide special discounts for returning customers.",
        },
    ];

    // Use provided FAQs or default ones
    const displayFaqs = faqs.length > 0 ? faqs : defaultFaqs;

    // Animation variants
    const fadeIn = {
        hidden: { opacity: 0, y: 20 },
        visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
    };

    const staggerContainer = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: {
                staggerChildren: 0.2,
            },
        },
    };

    return (
        <section
            id="service-faqs"
            className="py-20 bg-gray-50 dark:bg-gray-900 w-full border-t border-gray-200 dark:border-gray-700"
        >
            <div className="max-w-3xl mx-auto px-4 sm:px-6 lg:px-8">
                <motion.div
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true }}
                    variants={fadeIn}
                    className="text-center mb-16"
                >
                    <h2 className="text-3xl font-bold text-gray-900 dark:text-white">
                        {title}
                    </h2>
                    <div className="mt-4 h-1 w-24 bg-blue-600 dark:bg-blue-500 mx-auto rounded-full"></div>
                    <div 
                        className="mt-6 max-w-2xl mx-auto text-lg text-gray-600 dark:text-gray-300 content-area"
                        dangerouslySetInnerHTML={{ __html: description }}
                    />
                </motion.div>

                <motion.div
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true }}
                    variants={staggerContainer}
                    className="space-y-4"
                >
                    {displayFaqs.map((faq, index) => (
                        <motion.div
                            key={index}
                            variants={fadeIn}
                            className="border border-gray-200 dark:border-gray-700 rounded-lg overflow-hidden shadow-sm"
                        >
                            <button
                                className="flex justify-between items-center w-full p-4 text-left bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-800 hover:bg-blue-50 dark:hover:bg-gray-700 transition-colors duration-200 border-0"
                                onClick={() =>
                                    setActiveFaq(
                                        activeFaq === index ? null : index
                                    )
                                }
                                aria-expanded={activeFaq === index}
                                aria-controls={`faq-answer-${index}`}
                            >
                                <span className="font-medium text-gray-900 dark:text-white">
                                    {faq.question}
                                </span>
                                <ChevronDown
                                    className={`h-5 w-5 text-gray-500 dark:text-gray-400 transition-transform duration-200 ${
                                        activeFaq === index
                                            ? "transform rotate-180"
                                            : ""
                                    }`}
                                />
                            </button>
                            {activeFaq === index && (
                                <motion.div
                                    initial={{ opacity: 0, height: 0 }}
                                    animate={{ opacity: 1, height: "auto" }}
                                    exit={{ opacity: 0, height: 0 }}
                                    transition={{ duration: 0.3 }}
                                    className="p-4 bg-white dark:bg-gray-800/50 text-gray-600 dark:text-gray-300 border-t border-gray-200 dark:border-gray-700 content-area"
                                    id={`faq-answer-${index}`}
                                    dangerouslySetInnerHTML={{ __html: faq.answer }}
                                />
                            )}
                        </motion.div>
                    ))}
                </motion.div>

                <motion.div
                    initial={{ opacity: 0 }}
                    whileInView={{ opacity: 1 }}
                    transition={{ delay: 0.4 }}
                    viewport={{ once: true }}
                    className="mt-12 text-center"
                >
                    <p className="text-gray-600 dark:text-gray-300 mb-4">
                        Still have questions? We're here to help.
                    </p>
                    <a
                        href="/contact"
                        className="inline-flex items-center px-6 py-3 border border-transparent text-base font-medium rounded-md shadow-sm text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                    >
                        Contact Us
                    </a>
                </motion.div>
            </div>
        </section>
    );
}

ServiceFAQs.propTypes = {
    title: PropTypes.string,
    description: PropTypes.string,
    faqs: PropTypes.arrayOf(
        PropTypes.shape({
            question: PropTypes.string.isRequired,
            answer: PropTypes.string.isRequired,
        })
    ),
};
