import { motion } from "framer-motion";
import { ArrowRight, CheckCircle, Shield, Star, Users } from "lucide-react";
import { Fragment, useEffect, useRef } from "react";
import PropTypes from "prop-types";

export default function ServiceHero({
    title,
    description,
    keyPoints = [],
    ctaText = "Get Started",
    ctaLink = "#contact",
    trustSignals = [],
}) {
    const descriptionRef = useRef(null);

    // Force link visibility in hero sections by directly styling links after render
    useEffect(() => {
        if (descriptionRef.current) {
            const links = descriptionRef.current.querySelectorAll('a[href]:not([class*="btn"]):not(.inline-flex):not(.no-link-style)');
            links.forEach(link => {
                link.style.color = 'rgb(217 119 6)'; // amber-600
                link.style.textDecoration = 'underline';
                link.style.textDecorationThickness = '2px';
                link.style.textUnderlineOffset = '2px';
                link.style.fontWeight = '500';
                
                // Add hover effect
                link.addEventListener('mouseenter', () => {
                    link.style.color = 'rgb(180 83 9)'; // amber-700
                });
                link.addEventListener('mouseleave', () => {
                    link.style.color = 'rgb(217 119 6)'; // amber-600
                });
            });
        }
    }, [description]);

    const fadeIn = {
        hidden: { opacity: 0, y: 20 },
        visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
    };

    const defaultTrust = [
        { icon: Star, label: "4.8/5 Rating" },
        { icon: Users, label: "150,000+ Students" },
        { icon: Shield, label: "100% Plagiarism-Free" },
    ];

    const formattedTrustSignals = (trustSignals || []).map((signal) => {
        if (typeof signal === "string") {
            return { label: signal };
        }

        if (signal && typeof signal === "object") {
            return {
                label: signal.label || signal.value || "",
                value: signal.value || null,
            };
        }

        return { label: "" };
    });

    const renderRichText = (content, { inline = false, className = "" } = {}) => {
        if (!content) {
            return null;
        }

        const text =
            typeof content === "object" && content !== null
                ? content.text || content.value || content.label || ""
                : content;

        if (!text) {
            return null;
        }

        const isHtml =
            typeof text === "string" && /<([a-z][^>]*)>/i.test(text || "");

        if (isHtml) {
            const htmlClass = ["content-area", inline ? "inline-block" : "", className]
                .filter(Boolean)
                .join(" ");

            return inline ? (
                <span
                    className={htmlClass}
                    dangerouslySetInnerHTML={{ __html: text }}
                />
            ) : (
                <div
                    className={htmlClass}
                    dangerouslySetInnerHTML={{ __html: text }}
                />
            );
        }

        if (inline) {
            return <span className={className}>{text}</span>;
        }

        return <p className={className}>{text}</p>;
    };

    const headlineStats = formattedTrustSignals.length > 0
        ? formattedTrustSignals.map((stat, index) => ({
              ...stat,
              icon: defaultTrust[index]?.icon || CheckCircle,
          }))
        : defaultTrust;

    return (
        <section className="w-full bg-gradient-to-r from-blue-900 to-cyan-500 text-white dark:from-gray-900 dark:to-gray-800 pt-28 pb-20">
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <motion.div
                    initial="hidden"
                    animate="visible"
                    variants={fadeIn}
                    className="max-w-4xl mx-auto text-center"
                >
                    <div className="flex flex-wrap items-center justify-center gap-6 mb-6 text-blue-100 text-sm">
                        {headlineStats.map((stat, index) => {
                            const IconComponent = stat.icon || CheckCircle;
                            const labelContent = stat.label || stat.value || "";

                            return (
                                <Fragment key={`${labelContent}-${index}`}>
                                    <div className="flex items-center gap-2 text-left">
                                        <IconComponent className="h-5 w-5 text-blue-200" />
                                        {renderRichText(labelContent, {
                                            inline: true,
                                            className: "font-medium text-blue-100 dark:text-gray-300",
                                        }) || (
                                            <span className="font-medium text-blue-100 dark:text-gray-300">
                                                {labelContent}
                                            </span>
                                        )}
                                    </div>
                                    {index !== headlineStats.length - 1 && (
                                        <div className="hidden sm:block w-px h-5 bg-blue-300/40" />
                                    )}
                                </Fragment>
                            );
                        })}
                    </div>

                    <h1 className="text-4xl md:text-5xl font-bold leading-tight mb-6">
                        {title || "Professional Academic Service"}
                    </h1>
                    <div
                        ref={descriptionRef}
                        className="text-lg md:text-xl mb-8 prose prose-lg dark:prose-invert max-w-none prose-headings:!text-blue-100 prose-headings:dark:!text-gray-300 prose-p:!text-blue-100 prose-p:dark:!text-gray-300 prose-strong:!text-blue-100 prose-strong:dark:!text-gray-300 prose-em:!text-blue-100 prose-em:dark:!text-gray-300 prose-u:!text-blue-100 prose-u:dark:!text-gray-300 prose-a:!text-amber-600 prose-a:dark:!text-amber-400 prose-a:!underline prose-a:!decoration-2 prose-a:!underline-offset-2 prose-a:!font-medium hover:prose-a:!text-amber-700 dark:hover:prose-a:!text-amber-300 content-area"
                        dangerouslySetInnerHTML={{
                            __html:
                                description ||
                                "Expert assistance with your academic needs. Our team of PhD-level writers delivers excellence with every project.",
                        }}
                    />

                    {keyPoints.length > 0 && (
                        <div className="space-y-3 mb-10 max-w-3xl mx-auto text-left">
                            {keyPoints.map((point, index) => (
                                <div key={index} className="flex items-start gap-2">
                                    <CheckCircle className="h-6 w-6 text-blue-200 flex-shrink-0 mt-1" />
                                    <div className="flex-1">
                                        {renderRichText(point, {
                                            className: "text-blue-100 dark:text-gray-300",
                                        })}
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}

                    <motion.div
                        className="flex flex-col sm:flex-row justify-center space-y-4 sm:space-y-0 sm:space-x-4 mt-8"
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ delay: 0.4, duration: 0.6 }}
                    >
                        <a
                            href={ctaLink}
                            className="bg-white text-blue-800 hover:bg-gray-100 dark:bg-gray-700 dark:text-white dark:hover:bg-gray-600 px-8 py-4 rounded-md font-medium shadow-lg transition-colors duration-200 flex items-center justify-center"
                        >
                            {ctaText} <ArrowRight className="ml-2 h-5 w-5" />
                        </a>
                        <a
                            href="/place-order"
                            className="border border-white text-white hover:bg-white hover:text-blue-800 dark:border-gray-500 dark:text-gray-300 dark:hover:bg-gray-700 dark:hover:text-white px-8 py-4 rounded-md font-medium transition-colors duration-200"
                        >
                            Place Order
                        </a>
                    </motion.div>
                </motion.div>
            </div>
        </section>
    );
}

ServiceHero.propTypes = {
    title: PropTypes.string,
    description: PropTypes.string,
    keyPoints: PropTypes.arrayOf(
        PropTypes.oneOfType([
            PropTypes.string,
            PropTypes.shape({
                text: PropTypes.string,
                value: PropTypes.string,
                label: PropTypes.string,
            }),
        ])
    ),
    ctaText: PropTypes.string,
    ctaLink: PropTypes.string,
    trustSignals: PropTypes.arrayOf(
        PropTypes.oneOfType([
            PropTypes.string,
            PropTypes.shape({
                label: PropTypes.string,
                value: PropTypes.string,
            })
        ])
    ),
};
