import { motion } from "framer-motion";
import {
    BookOpen,
    Award,
    Clock,
    CheckCircle,
    User,
    FileText,
} from "lucide-react";
import PropTypes from "prop-types";
import { useEffect, useRef } from "react";

export default function ServiceOverview({
    title = "Our Service Overview",
    description = "We provide comprehensive academic assistance tailored to your specific needs.",
    features = [],
}) {
    const descriptionRef = useRef(null);
    const featureRefs = useRef([]);

    // Force link visibility in overview description
    useEffect(() => {
        if (descriptionRef.current) {
            const links = descriptionRef.current.querySelectorAll('a[href]:not([class*="btn"]):not(.inline-flex):not(.no-link-style)');
            links.forEach(link => {
                link.style.color = 'rgb(217 119 6)'; // amber-600
                link.style.textDecoration = 'underline';
                link.style.textDecorationThickness = '2px';
                link.style.textUnderlineOffset = '2px';
                link.style.fontWeight = '500';
                
                link.addEventListener('mouseenter', () => {
                    link.style.color = 'rgb(180 83 9)'; // amber-700
                });
                link.addEventListener('mouseleave', () => {
                    link.style.color = 'rgb(217 119 6)'; // amber-600
                });
            });
        }
    }, [description]);

    // Force link visibility in feature descriptions
    useEffect(() => {
        featureRefs.current.forEach(ref => {
            if (ref) {
                const links = ref.querySelectorAll('a[href]:not([class*="btn"]):not(.inline-flex):not(.no-link-style)');
                links.forEach(link => {
                    link.style.color = 'rgb(217 119 6)'; // amber-600
                    link.style.textDecoration = 'underline';
                    link.style.textDecorationThickness = '2px';
                    link.style.textUnderlineOffset = '2px';
                    link.style.fontWeight = '500';
                    
                    link.addEventListener('mouseenter', () => {
                        link.style.color = 'rgb(180 83 9)'; // amber-700
                    });
                    link.addEventListener('mouseleave', () => {
                        link.style.color = 'rgb(217 119 6)'; // amber-600
                    });
                });
            }
        });
    }, [features]);

    // Default features if none provided
    const defaultFeatures = [
        {
            title: "Expert Writers",
            description:
                "PhD-level academics with expertise in your specific field",
            icon: "User",
        },
        {
            title: "Original Content",
            description: "100% plagiarism-free papers written from scratch",
            icon: "FileText",
        },
        {
            title: "Quality Assured",
            description: "Rigorous quality control and editing process",
            icon: "Award",
        },
        {
            title: "Timely Delivery",
            description: "On-time completion with fast turnaround options",
            icon: "Clock",
        },
    ];

    // Use provided features or default ones
    const displayFeatures = features.length > 0 ? features : defaultFeatures;

    // Map icon names to actual components
    const iconMap = {
        BookOpen: BookOpen,
        Award: Award,
        Clock: Clock,
        CheckCircle: CheckCircle,
        User: User,
        FileText: FileText,
    };

    // Animation variants
    const fadeIn = {
        hidden: { opacity: 0, y: 20 },
        visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
    };

    const staggerContainer = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: {
                staggerChildren: 0.2,
            },
        },
    };

    return (
        <section
            id="service-overview"
            className="py-20 bg-gray-50 dark:bg-gray-900 w-full"
        >
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <motion.div
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true, margin: "-100px" }}
                    variants={fadeIn}
                    className="text-center mb-16"
                >
                    <h2 className="text-3xl font-bold text-gray-900 dark:text-white">
                        {title}
                    </h2>
                    <div className="mt-4 h-1 w-24 bg-blue-600 mx-auto rounded-full"></div>
                    <div 
                        ref={descriptionRef}
                        className="mt-6 max-w-2xl mx-auto text-xl prose prose-lg dark:prose-invert max-w-none prose-headings:!text-gray-900 prose-headings:dark:!text-white prose-p:!text-gray-600 prose-p:dark:!text-gray-300 prose-strong:!text-gray-900 prose-strong:dark:!text-white prose-em:!text-gray-600 prose-em:dark:!text-gray-300 prose-u:!text-gray-600 prose-u:dark:!text-gray-300 prose-a:!text-amber-600 prose-a:dark:!text-amber-400 prose-a:!underline prose-a:!decoration-2 prose-a:!underline-offset-2 prose-a:!font-medium hover:prose-a:!text-amber-700 dark:hover:prose-a:!text-amber-300 content-area"
                        dangerouslySetInnerHTML={{ __html: description }}
                    />
                </motion.div>

                <motion.div
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true, margin: "-50px" }}
                    variants={staggerContainer}
                    className="grid md:grid-cols-2 lg:grid-cols-4 gap-8"
                >
                    {displayFeatures.map((feature, index) => {
                        // Get the icon component or default to BookOpen
                        const IconComponent = iconMap[feature.icon] || BookOpen;

                        return (
                            <motion.div
                                key={index}
                                variants={fadeIn}
                                className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 dark:bg-gray-800 rounded-lg p-6
                                           transition-all duration-300 ease-in-out
                                           hover:shadow-xl hover:transform hover:-translate-y-2
                                           flex flex-col h-full"
                            >
                                <div
                                    className="flex items-center justify-center mx-auto h-16 w-16 rounded-full bg-blue-600 text-white mb-6
                                              hover:bg-blue-700 transition-colors duration-300"
                                >
                                    <IconComponent className="h-6 w-6" />
                                </div>
                                <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-3 text-center">
                                    {feature.title}
                                </h3>
                                <div 
                                    ref={el => featureRefs.current[index] = el}
                                    className="text-center mb-6 prose prose-sm dark:prose-invert max-w-none prose-headings:!text-gray-900 prose-headings:dark:!text-white prose-p:!text-gray-600 prose-p:dark:!text-gray-300 prose-strong:!text-gray-900 prose-strong:dark:!text-white prose-em:!text-gray-600 prose-em:dark:!text-gray-300 prose-u:!text-gray-600 prose-u:dark:!text-gray-300 prose-a:!text-amber-600 prose-a:dark:!text-amber-400 prose-a:!underline prose-a:!decoration-2 prose-a:!underline-offset-2 prose-a:!font-medium hover:prose-a:!text-amber-700 dark:hover:prose-a:!text-amber-300 content-area"
                                    dangerouslySetInnerHTML={{ __html: feature.description }}
                                />
                            </motion.div>
                        );
                    })}
                </motion.div>
            </div>
        </section>
    );
}

ServiceOverview.propTypes = {
    title: PropTypes.string,
    description: PropTypes.string,
    features: PropTypes.arrayOf(
        PropTypes.shape({
            title: PropTypes.string.isRequired,
            description: PropTypes.string.isRequired,
            icon: PropTypes.string,
        })
    ),
};
