import { motion } from "framer-motion";
import { Star } from "lucide-react";
import PropTypes from "prop-types";
import { useState } from "react";

export default function ServiceTestimonials({
    title = "What Our Clients Say",
    description = "Read testimonials from students who've used our service",
    testimonials = [],
}) {
    const [activeIndex, setActiveIndex] = useState(0);

    // Default testimonials if none provided
    const defaultTestimonials = [
        {
            text: "The quality of research and writing exceeded my expectations. My paper received excellent feedback from my professor.",
            author: "Sarah J.",
            role: "Undergraduate Student",
            rating: 5,
        },
        {
            text: "Their expertise helped me complete my dissertation on time. The writer understood exactly what I needed and delivered exceptional work.",
            author: "Michael T.",
            role: "PhD Candidate",
            rating: 5,
        },
        {
            text: "As an international student, I struggled with complex writing assignments. This service helped me improve my academic performance significantly.",
            author: "Li Wei",
            role: "Masters Student",
            rating: 4,
        },
    ];

    // Use provided testimonials or default ones
    const displayTestimonials =
        testimonials.length > 0 ? testimonials : defaultTestimonials;

    // Animation variants
    const fadeIn = {
        hidden: { opacity: 0, y: 20 },
        visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
    };

    const staggerContainer = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: {
                staggerChildren: 0.2,
            },
        },
    };

    // Function to render stars based on rating
    const renderStars = (rating) => {
        return Array.from({ length: 5 }).map((_, index) => (
            <Star
                key={index}
                className={`h-4 w-4 ${
                    index < rating
                        ? "text-yellow-400 fill-yellow-400"
                        : "text-gray-300 dark:text-gray-600"
                }`}
            />
        ));
    };

    // Go to next testimonial
    const nextTestimonial = () => {
        setActiveIndex((prevIndex) =>
            prevIndex === displayTestimonials.length - 1 ? 0 : prevIndex + 1
        );
    };

    // Go to previous testimonial
    const prevTestimonial = () => {
        setActiveIndex((prevIndex) =>
            prevIndex === 0 ? displayTestimonials.length - 1 : prevIndex - 1
        );
    };

    return (
        <section
            id="service-testimonials"
            className="py-20 bg-gray-100 dark:bg-gray-800 w-full border-t border-gray-200 dark:border-gray-700"
        >
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <motion.div
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true }}
                    variants={fadeIn}
                    className="text-center mb-16"
                >
                    <h2 className="text-3xl font-bold text-gray-900 dark:text-white">
                        {title}
                    </h2>
                    <div className="mt-4 h-1 w-24 bg-blue-600 mx-auto rounded-full"></div>
                    <div 
                        className="mt-6 max-w-2xl mx-auto text-xl text-gray-600 dark:text-gray-300 content-area"
                        dangerouslySetInnerHTML={{ __html: description }}
                    />
                </motion.div>

                {/* Desktop Layout - Card Grid */}
                <motion.div
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true }}
                    variants={staggerContainer}
                    className="hidden md:grid md:grid-cols-3 gap-8"
                >
                    {displayTestimonials.map((testimonial, index) => (
                        <motion.div
                            key={index}
                            variants={fadeIn}
                            className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 dark:bg-gray-800 p-8 rounded-lg relative shadow-sm hover:shadow-md transition-shadow duration-300 flex flex-col"
                        >
                            <svg
                                className="absolute top-4 left-4 h-12 w-12 text-blue-200 dark:text-blue-900"
                                fill="currentColor"
                                viewBox="0 0 32 32"
                                aria-hidden="true"
                            >
                                <path d="M9.352 4C4.456 7.456 1 13.12 1 19.36c0 5.088 3.072 8.064 6.624 8.064 3.36 0 5.856-2.688 5.856-5.856 0-3.168-2.208-5.472-5.088-5.472-.576 0-1.344.096-1.536.192.48-3.264 3.552-7.104 6.624-9.024L9.352 4zm16.512 0c-4.8 3.456-8.256 9.12-8.256 15.36 0 5.088 3.072 8.064 6.624 8.064 3.264 0 5.856-2.688 5.856-5.856 0-3.168-2.304-5.472-5.184-5.472-.576 0-1.248.096-1.44.192.48-3.264 3.456-7.104 6.528-9.024L25.864 4z" />
                            </svg>
                            <div className="relative pl-12">
                                <div className="flex mb-4">
                                    {renderStars(testimonial.rating)}
                                </div>
                                <div 
                                    className="text-gray-600 dark:text-gray-300 mb-6 content-area"
                                    dangerouslySetInnerHTML={{ __html: `"${testimonial.text}"` }}
                                />
                                <div className="mt-auto">
                                    <p className="font-medium text-gray-900 dark:text-white">
                                        {testimonial.author}
                                    </p>
                                    <p className="text-sm text-gray-500 dark:text-gray-400">
                                        {testimonial.role}
                                    </p>
                                </div>
                            </div>
                        </motion.div>
                    ))}
                </motion.div>

                {/* Mobile Layout - Carousel */}
                <div className="md:hidden">
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 dark:bg-gray-800 p-8 rounded-lg relative shadow-sm"
                    >
                        <svg
                            className="absolute top-4 left-4 h-12 w-12 text-blue-200 dark:text-blue-900"
                            fill="currentColor"
                            viewBox="0 0 32 32"
                            aria-hidden="true"
                        >
                            <path d="M9.352 4C4.456 7.456 1 13.12 1 19.36c0 5.088 3.072 8.064 6.624 8.064 3.36 0 5.856-2.688 5.856-5.856 0-3.168-2.208-5.472-5.088-5.472-.576 0-1.344.096-1.536.192.48-3.264 3.552-7.104 6.624-9.024L9.352 4zm16.512 0c-4.8 3.456-8.256 9.12-8.256 15.36 0 5.088 3.072 8.064 6.624 8.064 3.264 0 5.856-2.688 5.856-5.856 0-3.168-2.304-5.472-5.184-5.472-.576 0-1.248.096-1.44.192.48-3.264 3.456-7.104 6.528-9.024L25.864 4z" />
                        </svg>
                        <div className="relative pl-12">
                            <div className="flex mb-4">
                                {renderStars(
                                    displayTestimonials[activeIndex].rating
                                )}
                            </div>
                            <p className="text-gray-600 dark:text-gray-300 mb-6">
                                "{displayTestimonials[activeIndex].text}"
                            </p>
                            <div>
                                <p className="font-medium text-gray-900 dark:text-white">
                                    {displayTestimonials[activeIndex].author}
                                </p>
                                <p className="text-sm text-gray-500 dark:text-gray-400">
                                    {displayTestimonials[activeIndex].role}
                                </p>
                            </div>
                        </div>
                    </motion.div>

                    {/* Carousel Controls */}
                    <div className="flex justify-between items-center mt-6">
                        <button
                            onClick={prevTestimonial}
                            className="p-2 rounded-full bg-gray-200 dark:bg-gray-700 text-gray-600 dark:text-gray-300 hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors"
                            aria-label="Previous testimonial"
                        >
                            <svg
                                xmlns="http://www.w3.org/2000/svg"
                                className="h-5 w-5"
                                viewBox="0 0 20 20"
                                fill="currentColor"
                            >
                                <path
                                    fillRule="evenodd"
                                    d="M12.707 5.293a1 1 0 010 1.414L9.414 10l3.293 3.293a1 1 0 01-1.414 1.414l-4-4a1 1 0 010-1.414l4-4a1 1 0 011.414 0z"
                                    clipRule="evenodd"
                                />
                            </svg>
                        </button>

                        <div className="flex space-x-2">
                            {displayTestimonials.map((_, index) => (
                                <button
                                    key={index}
                                    onClick={() => setActiveIndex(index)}
                                    className={`h-2 w-2 rounded-full ${
                                        activeIndex === index
                                            ? "bg-blue-600"
                                            : "bg-gray-300 dark:bg-gray-600"
                                    }`}
                                    aria-label={`Go to testimonial ${
                                        index + 1
                                    }`}
                                />
                            ))}
                        </div>

                        <button
                            onClick={nextTestimonial}
                            className="p-2 rounded-full bg-gray-200 dark:bg-gray-700 text-gray-600 dark:text-gray-300 hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors"
                            aria-label="Next testimonial"
                        >
                            <svg
                                xmlns="http://www.w3.org/2000/svg"
                                className="h-5 w-5"
                                viewBox="0 0 20 20"
                                fill="currentColor"
                            >
                                <path
                                    fillRule="evenodd"
                                    d="M7.293 14.707a1 1 0 010-1.414L10.586 10 7.293 6.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z"
                                    clipRule="evenodd"
                                />
                            </svg>
                        </button>
                    </div>
                </div>
            </div>
        </section>
    );
}

ServiceTestimonials.propTypes = {
    title: PropTypes.string,
    description: PropTypes.string,
    testimonials: PropTypes.arrayOf(
        PropTypes.shape({
            text: PropTypes.string.isRequired,
            author: PropTypes.string.isRequired,
            role: PropTypes.string.isRequired,
            rating: PropTypes.number,
        })
    ),
};
