import React from "react";
import { motion } from "framer-motion";
import { GraduationCap, Target, BookOpen, Award, CheckCircle, Users } from "lucide-react";
import PropTypes from "prop-types";

/**
 * Writer Selection Section Component
 * Displays writer selection criteria and qualifications
 */
export default function WriterSelectionSection({
  title = "How We Select Writers",
  description = "",
  criteria = [],
  qualifications = [],
}) {
  const iconMap = {
    GraduationCap,
    Target,
    BookOpen,
    Award,
    CheckCircle,
    Users,
  };

  const fadeIn = {
    hidden: { opacity: 0, y: 20 },
    visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
  };

  const staggerContainer = {
    hidden: { opacity: 0 },
    visible: {
      opacity: 1,
      transition: {
        staggerChildren: 0.15,
      },
    },
  };

  return (
    <section className="py-20 bg-gray-50 dark:bg-gray-800 w-full">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <motion.div
          initial="hidden"
          whileInView="visible"
          viewport={{ once: true }}
          variants={fadeIn}
          className="text-center mb-16"
        >
          <h2 className="text-3xl font-bold text-gray-900 dark:text-white">
            {title}
          </h2>
          <div className="mt-4 h-1 w-24 bg-blue-600 mx-auto rounded-full"></div>
          {description && (
            <div 
              className="mt-6 max-w-2xl mx-auto text-xl text-gray-600 dark:text-gray-300 content-area"
              dangerouslySetInnerHTML={{ __html: description }}
            />
          )}
        </motion.div>

        {/* Writer Criteria Cards */}
        <motion.div
          initial="hidden"
          whileInView="visible"
          viewport={{ once: true }}
          variants={staggerContainer}
          className="grid md:grid-cols-2 lg:grid-cols-3 gap-8 mb-12"
        >
          {criteria.map((item, index) => {
            const IconComponent = iconMap[item.icon] || GraduationCap;
            return (
              <motion.div
                key={index}
                variants={fadeIn}
                className="bg-white dark:bg-gray-900 rounded-lg p-6 shadow-md border border-gray-200 dark:border-gray-700
                           transition-all duration-300 hover:shadow-xl hover:-translate-y-1"
              >
                <div className="flex items-center justify-center h-14 w-14 rounded-full bg-gradient-to-br from-blue-500 to-indigo-600 text-white mb-4">
                  <IconComponent className="h-7 w-7" />
                </div>
                <h3 className="text-xl font-semibold text-gray-900 dark:text-white mb-3">
                  {item.title}
                </h3>
                <div 
                  className="text-gray-600 dark:text-gray-300 leading-relaxed content-area"
                  dangerouslySetInnerHTML={{ __html: item.description }}
                />
              </motion.div>
            );
          })}
        </motion.div>

        {/* Writer Qualifications List */}
        {qualifications.length > 0 && (
          <motion.div
            initial="hidden"
            whileInView="visible"
            viewport={{ once: true }}
            variants={fadeIn}
            className="max-w-4xl mx-auto bg-white dark:bg-gray-900 rounded-lg shadow-lg p-8"
          >
            <h3 className="text-2xl font-bold text-gray-900 dark:text-white mb-6 text-center">
              Writer Qualifications
            </h3>
            <div className="grid md:grid-cols-2 gap-6">
              {qualifications.map((qual, index) => (
                <div key={index} className="flex items-start">
                  <CheckCircle className="h-6 w-6 text-green-500 mr-3 flex-shrink-0 mt-1" />
                  <div>
                    <h4 className="font-semibold text-gray-900 dark:text-white mb-1">
                      {qual.title}
                    </h4>
                    <p className="text-gray-600 dark:text-gray-300 text-sm">
                      {qual.description}
                    </p>
                  </div>
                </div>
              ))}
            </div>
          </motion.div>
        )}
      </div>
    </section>
  );
}

WriterSelectionSection.propTypes = {
  title: PropTypes.string,
  description: PropTypes.string,
  criteria: PropTypes.arrayOf(
    PropTypes.shape({
      title: PropTypes.string.isRequired,
      description: PropTypes.string.isRequired,
      icon: PropTypes.string,
    })
  ),
  qualifications: PropTypes.arrayOf(
    PropTypes.shape({
      title: PropTypes.string.isRequired,
      description: PropTypes.string.isRequired,
    })
  ),
};









