const HTML_REGEX = /<\/?[a-z][\s\S]*>/i;

const toString = (value, fallback = "") => {
    if (typeof value === "string") {
        return value;
    }
    if (value === null || value === undefined) {
        return fallback;
    }
    if (typeof value === "number" || typeof value === "boolean") {
        return String(value);
    }
    return fallback;
};

export const normalizeRichText = (value) => {
    const raw = value ?? "";

    if (typeof raw === "string") {
        const trimmed = raw.trim();
        if (trimmed === "") {
            return "";
        }
        if (HTML_REGEX.test(trimmed)) {
            return trimmed;
        }
        return `<p>${trimmed
            .replace(/\r\n/g, "\n")
            .replace(/\n{2,}/g, "</p><p>")
            .replace(/\n/g, "<br />")}</p>`;
    }

    if (typeof raw === "object" && raw !== null) {
        if ("html" in raw) {
            return normalizeRichText(raw.html);
        }
        if ("text" in raw) {
            return normalizeRichText(raw.text);
        }
        if (Array.isArray(raw)) {
            return normalizeRichText(
                raw
                    .map((item) => (typeof item === "string" ? item : toString(item?.text)))
                    .filter(Boolean)
                    .join("\n\n")
            );
        }
    }

    return "";
};

const isActive = (record) => (record?.is_active ?? true) !== false;

const filterActive = (items) =>
    Array.isArray(items) ? items.filter((item) => isActive(item)) : [];

const normalizeTextItems = (items = []) =>
    filterActive(items)
        .map((item) => {
            if (typeof item === "string") {
                return item.trim();
            }
            if (item && typeof item === "object") {
                const text = toString(item.text || item.value || item.title || "");
                return text.trim();
            }
            return "";
        })
        .filter(Boolean);

const normalizeHero = (content = {}) => {
    const description = normalizeRichText(content.description_html ?? content.description);
    const keyPoints = normalizeTextItems(content.keyPoints);
    const trustSignals = normalizeTextItems(content.trustSignals);

    if (!content.title && !description && keyPoints.length === 0) {
        return null;
    }

    return {
        eyebrow: toString(content.eyebrow, "") || null,
        title: toString(content.title, ""),
        description,
        keyPoints,
        ctaText: toString(content.ctaText, "Get Started"),
        ctaLink: toString(content.ctaLink, "/contact"),
        trustSignals,
    };
};

const normalizeOverview = (content = {}) => {
    const description = normalizeRichText(content.description_html ?? content.description);
    const features = filterActive(content.features).map((feature) => ({
        icon: toString(feature?.icon, "Award"),
        title: toString(feature?.title, ""),
        description: normalizeRichText(feature?.description_html ?? feature?.description),
    }));

    if (!content.title && !description && features.length === 0) {
        return null;
    }

    return {
        title: toString(content.title, ""),
        description,
        features,
    };
};

const normalizeProcess = (content = {}) => {
    const description = normalizeRichText(content.description_html ?? content.description);
    const steps = filterActive(content.steps).map((step, index) => ({
        number:
            typeof step?.number === "number" && !Number.isNaN(step.number)
                ? step.number
                : index + 1,
        title: toString(step?.title, `Step ${index + 1}`),
        description: normalizeRichText(step?.description_html ?? step?.description),
        icon: toString(step?.icon, "Clock"),
    }));

    if (!content.title && !description && steps.length === 0) {
        return null;
    }

    return {
        title: toString(content.title, ""),
        description,
        steps,
    };
};

const normalizeBenefits = (content = {}) => {
    const description = normalizeRichText(content.description_html ?? content.description);
    const benefits = filterActive(content.benefits).map((benefit) => ({
        icon: toString(benefit?.icon, "Award"),
        title: toString(benefit?.title, ""),
        description: normalizeRichText(
            benefit?.description_html ?? benefit?.description ?? benefit?.text
        ),
        is_active: isActive(benefit),
    }));

    // Normalize guarantees - ensure it's either null, an object with items, or an array
    let guarantees = content.guarantees ?? null;
    if (guarantees !== null && guarantees !== undefined) {
        if (Array.isArray(guarantees)) {
            // If it's an array, keep it as is (valid) but filter out empty arrays
            guarantees = guarantees.length > 0 ? guarantees : null;
        } else if (typeof guarantees === 'object' && guarantees !== null) {
            // Check if it's an empty object
            const keys = Object.keys(guarantees);
            if (keys.length === 0) {
                // Empty object - set to null
                guarantees = null;
            } else if (Array.isArray(guarantees.items)) {
                // Valid object with items array - keep the whole object if items exist
                guarantees = guarantees.items.length > 0 ? {
                    title: guarantees.title || null,
                    description: guarantees.description || null,
                    items: guarantees.items
                } : null;
            } else if (guarantees.title || guarantees.description) {
                // Object with title/description but no items - ensure it has items array
                guarantees = {
                    title: guarantees.title || null,
                    description: guarantees.description || null,
                    items: Array.isArray(guarantees.items) ? guarantees.items : []
                };
            } else {
                // Object without expected structure - set to null
                guarantees = null;
            }
        } else {
            // Invalid type (string, number, etc.) - set to null
            guarantees = null;
        }
    } else {
        guarantees = null;
    }

    return {
        eyebrow: toString(content.eyebrow, "") || null,
        title: toString(content.title, ""),
        description,
        benefits,
        ctaText: toString(content.ctaText, ""),
        ctaLink: toString(content.ctaLink, ""),
        serviceName: toString(content.serviceName, "") || null,
        quickStart: content.quickStart ?? null,
        offer: content.offer ?? null,
        support: content.support ?? null,
        trustSignals: content.trustSignals ?? [],
        guarantees,
    };
};

const normalizeTestimonials = (content = {}) => {
    const description = normalizeRichText(content.description_html ?? content.description);
    const testimonials = filterActive(content.testimonials).map((testimonial) => ({
        text: normalizeRichText(testimonial?.text_html ?? testimonial?.text),
        author: toString(testimonial?.author, ""),
        role: toString(testimonial?.role, ""),
        rating:
            typeof testimonial?.rating === "number" ? testimonial.rating : testimonial?.rating ?? 5,
    }));

    if (!content.title && !description && testimonials.length === 0) {
        return null;
    }

    return {
        title: toString(content.title, ""),
        description,
        testimonials,
        trustMetrics: content.trustMetrics ?? {},
    };
};

const normalizeFaqs = (content = {}) => {
    const description = normalizeRichText(content.description_html ?? content.description);
    const faqs = filterActive(content.faqs).map((faq) => ({
        question: toString(faq?.question, ""),
        answer: normalizeRichText(faq?.answer_html ?? faq?.answer),
    }));

    if (!content.title && !description && faqs.length === 0) {
        return null;
    }

    return {
        title: toString(content.title, ""),
        description,
        faqs,
    };
};

const normalizeCta = (content = {}) => {
    const description = normalizeRichText(content.description_html ?? content.description);
    const benefits = filterActive(content.benefits).map((benefit) => ({
        text: toString(benefit?.text, ""),
        icon: toString(benefit?.icon, ""),
        is_active: isActive(benefit),
    }));

    return {
        title: toString(content.title, ""),
        description,
        benefits,
        primaryButtonText: toString(content.primaryButtonText, "Get Started"),
        primaryButtonLink: toString(content.primaryButtonLink, "/place-order"),
        secondaryButtonText: toString(content.secondaryButtonText, ""),
        secondaryButtonLink: toString(content.secondaryButtonLink, ""),
        bgColor: toString(content.bgColor, "blue"),
        urgencyText: toString(content.urgencyText, ""),
        guarantees: filterActive(content.guarantees),
        trustBadges: filterActive(content.trustBadges),
    };
};

const deriveLinkCategory = (block, defaultKey) => {
    const explicit = toString(block?.content?.linkCategory, "").toLowerCase();
    if (explicit) {
        return explicit;
    }

    const key = toString(block?.key ?? defaultKey, "").toLowerCase();
    if (key.includes("related")) return "related";
    if (key.includes("support")) return "support";
    if (key.includes("resource")) return "resources";
    return "related";
};

const normalizeLinksBlock = (content = {}) => {
    const description = normalizeRichText(content.description_html ?? content.description);
    const links = filterActive(content.links).map((link) => ({
        title: toString(link?.title, ""),
        url: toString(link?.url, "#"),
        description: normalizeRichText(link?.description_html ?? link?.description),
    }));

    if (!content.title && !description && links.length === 0) {
        return null;
    }

    return {
        title: toString(content.title, ""),
        description,
        links,
    };
};

const normalizeScrollableBlocks = (blocks, heroTitle) => {
    const sections = blocks
        .map((block) => ({
            title: toString(block?.content?.title, ""),
            html: normalizeRichText(
                block?.content?.description_html ?? block?.content?.description
            ),
            eyebrow: toString(block?.content?.eyebrow, ""),
            preferredHeight:
                typeof block?.content?.preferredHeight === "number"
                    ? block.content.preferredHeight
                    : 480,
        }))
        .filter((section) => section.html);

    if (sections.length === 0) {
        return {
            eyebrow: null,
            title: null,
            description: null,
            preferredHeight: 480,
            sections: [],
        };
    }

    const primarySection = sections.find((section) => section.eyebrow) ?? sections[0];

    return {
        eyebrow: primarySection.eyebrow || "Deep Dive",
        title: `${heroTitle || "Service"} Deep Dive`,
        description: null,
        preferredHeight: primarySection.preferredHeight,
        sections: sections.map(({ title, html }) => ({ title, html })),
    };
};

export const buildServicePageState = (contentBlocks = []) => {
    const sortedBlocks = [...contentBlocks].sort(
        (a, b) => (a.order ?? 0) - (b.order ?? 0)
    );

    // Helper to find blocks by type, supporting both new (service_*) and old naming conventions
    const getFirstByType = (newType, oldType = null) => {
        const block = sortedBlocks.find(
            (block) => 
                (block.type === newType || (oldType && block.type === oldType)) && 
                isActive(block)
        );
        return block;
    };

    const getAllByType = (newType, oldType = null) =>
        sortedBlocks.filter(
            (block) => 
                (block.type === newType || (oldType && block.type === oldType)) && 
                isActive(block)
        );

    const heroBlock = getFirstByType("service_hero", "hero");
    const overviewBlock = getFirstByType("service_overview", "overview");
    const processBlock = getFirstByType("service_process", "process");
    const benefitsBlock = getFirstByType("service_benefits", "benefits");
    const testimonialsBlock = getFirstByType("service_testimonials", "testimonials");
    const faqsBlock = getFirstByType("service_faqs", "faqs");
    const ctaBlock = getFirstByType("service_cta", "cta");
    const linksBlocks = getAllByType("service_links", "internal_links");
    const scrollableBlocks = getAllByType("service_scrollable");
    const priceCalculatorBlock = getFirstByType("price_calculator");

    const hero = heroBlock ? normalizeHero(heroBlock.content ?? {}) : null;
    const overview = overviewBlock
        ? normalizeOverview(overviewBlock.content ?? {})
        : null;
    const process = processBlock
        ? normalizeProcess(processBlock.content ?? {})
        : null;
    const benefits = benefitsBlock
        ? normalizeBenefits(benefitsBlock.content ?? {})
        : null;
    const testimonials = testimonialsBlock
        ? normalizeTestimonials(testimonialsBlock.content ?? {})
        : null;
    const faqs = faqsBlock ? normalizeFaqs(faqsBlock.content ?? {}) : null;
    const cta = ctaBlock ? normalizeCta(ctaBlock.content ?? {}) : null;
    const priceCalculator = priceCalculatorBlock ? priceCalculatorBlock.content : null;

    const links = linksBlocks.reduce(
        (acc, block) => {
            const category = deriveLinkCategory(block, block?.key);
            const normalized = normalizeLinksBlock(block.content ?? {});

            if (!normalized) {
                return acc;
            }

            if (category === "related") {
                acc.relatedServices = normalized;
            } else if (category === "support") {
                acc.supportPages = normalized;
            } else if (category === "resources") {
                acc.resourceLinks = normalized;
            }

            return acc;
        },
        {
            relatedServices: null,
            supportPages: null,
            resourceLinks: null,
        }
    );

    const scrollable = normalizeScrollableBlocks(scrollableBlocks, hero?.title);

    return {
        hero,
        overview,
        process,
        benefits,
        testimonials,
        faqs,
        cta,
        links,
        scrollable,
        priceCalculator,
    };
};



