import React, { useState, useEffect } from "react";
import { loadStripe } from "@stripe/stripe-js";
import {
    Elements,
    CardElement,
    useStripe,
    useElements,
} from "@stripe/react-stripe-js";
import axios from "axios";
import {
    Loader2,
    AlertCircle,
    CheckCircle,
    CreditCard,
    Lock,
} from "lucide-react";

// Helper function to get currency symbol
const getCurrencySymbol = (currencyCode) => {
    switch (currencyCode) {
        case "EUR":
            return "€";
        case "GBP":
            return "£";
        case "CAD":
            return "C$";
        case "AUD":
            return "A$";
        default:
            return "$";
    }
};

// Stripe form component - this is nested inside the main component
const StripeCheckoutForm = ({ order, onSuccess, onError }) => {
    const stripe = useStripe();
    const elements = useElements();
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState(null);
    const [clientSecret, setClientSecret] = useState("");
    const [processing, setProcessing] = useState(false);
    const [succeeded, setSucceeded] = useState(false);

    useEffect(() => {
        // Create PaymentIntent on component mount
        const createPaymentIntent = async () => {
            try {
                setLoading(true);
                setError(null);

                const { data } = await axios.post(
                    route("orders.initialize-payment", order.id),
                    { payment_method: "stripe" }
                );

                if (!data.success) {
                    throw new Error(
                        data.error || "Failed to initialize payment"
                    );
                }

                setClientSecret(data.client_secret);
            } catch (err) {
                console.error("Error creating Stripe payment intent:", err);
                setError(
                    err.response?.data?.message ||
                        "Failed to initialize payment. Please try again."
                );
            } finally {
                setLoading(false);
            }
        };

        createPaymentIntent();
    }, [order.id]);

    const handleSubmit = async (event) => {
        event.preventDefault();

        if (!stripe || !elements) {
            // Stripe.js hasn't loaded yet
            return;
        }

        // Prevent multiple submissions
        if (processing) {
            return;
        }

        setProcessing(true);
        setError(null);

        try {
            // Confirm the card payment
            const { error: stripeError, paymentIntent } =
                await stripe.confirmCardPayment(clientSecret, {
                    payment_method: {
                        card: elements.getElement(CardElement),
                        billing_details: {
                            name: order.user?.name || "Customer",
                        },
                    },
                });

            if (stripeError) {
                throw new Error(stripeError.message);
            }

            if (paymentIntent.status === "succeeded") {
                // Process the payment with our backend
                const response = await axios.post(
                    route("orders.process-payment", order.id),
                    {
                        payment_method: "stripe",
                        payment_data: {
                            payment_intent_id: paymentIntent.id,
                        },
                    }
                );

                if (response.data.success) {
                    setSucceeded(true);

                    // Add small delay before calling success handler
                    setTimeout(() => {
                        if (onSuccess) {
                            onSuccess(response.data);
                        } else {
                            // Default behavior: redirect
                            window.location.href =
                                response.data.redirect ||
                                route("orders.show", order.id);
                        }
                    }, 1000);
                } else {
                    throw new Error(
                        response.data.message || "Payment processing failed"
                    );
                }
            } else {
                throw new Error("Payment was not completed successfully");
            }
        } catch (err) {
            console.error("Stripe payment error:", err);
            setError(err.message || "Payment failed. Please try again.");

            if (onError) {
                onError(err);
            }
        } finally {
            setProcessing(false);
        }
    };

    const cardElementOptions = {
        style: {
            base: {
                color: "#32325d",
                fontFamily: '"Helvetica Neue", Helvetica, sans-serif',
                fontSmoothing: "antialiased",
                fontSize: "16px",
                "::placeholder": {
                    color: "#aab7c4",
                },
            },
            invalid: {
                color: "#fa755a",
                iconColor: "#fa755a",
            },
        },
        hidePostalCode: true,
    };

    if (loading) {
        return (
            <div className="flex items-center justify-center p-4 text-gray-600 dark:text-gray-300">
                <Loader2 className="h-5 w-5 animate-spin mr-2" />
                <span>Initializing payment...</span>
            </div>
        );
    }

    if (error) {
        return (
            <div className="p-4 bg-red-100 dark:bg-red-900 text-red-700 dark:text-red-300 rounded-lg flex items-start space-x-2">
                <AlertCircle className="w-5 h-5 mt-0.5 flex-shrink-0" />
                <span>{error}</span>
            </div>
        );
    }

    return (
        <form onSubmit={handleSubmit} className="space-y-6">
            <div className="p-4 border border-gray-300 dark:border-gray-700 rounded-lg">
                <div className="mb-4 flex items-center justify-between">
                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">
                        Card Details
                    </label>
                    <div className="flex items-center text-gray-500 dark:text-gray-400 text-sm">
                        <Lock className="h-4 w-4 mr-1" />
                        <span>Secure Payment</span>
                    </div>
                </div>
                <CardElement
                    id="card-element"
                    options={cardElementOptions}
                    className="p-3 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-800"
                />
            </div>

            <div className="mt-2 text-sm text-gray-500 dark:text-gray-400 flex items-center">
                <Lock className="h-4 w-4 mr-1" />
                <span>Your payment information is processed securely.</span>
            </div>

            <button
                type="submit"
                disabled={!stripe || processing || succeeded}
                className={`
                    w-full flex items-center justify-center px-4 py-3 rounded-md text-white font-medium
                    ${
                        processing || succeeded || !stripe
                            ? "bg-blue-400 dark:bg-blue-700 cursor-not-allowed"
                            : "bg-blue-600 hover:bg-blue-700 dark:bg-blue-700 dark:hover:bg-blue-800"
                    }
                    transition-colors duration-200
                `}
            >
                {processing ? (
                    <>
                        <Loader2 className="animate-spin h-5 w-5 mr-2" />
                        Processing...
                    </>
                ) : succeeded ? (
                    <>
                        <CheckCircle className="h-5 w-5 mr-2" />
                        Payment Successful
                    </>
                ) : (
                    <>
                        <CreditCard className="h-5 w-5 mr-2" />
                        Pay {getCurrencySymbol(order.currency)}
                        {order.net_amount}
                    </>
                )}
            </button>
        </form>
    );
};

// Main container component that loads Stripe
const StripePaymentForm = (props) => {
    const [stripePromise, setStripePromise] = useState(null);
    const [error, setError] = useState(null);

    useEffect(() => {
        const loadStripeKey = async () => {
            try {
                const { data } = await axios.get(
                    route("stripe.publishable-key")
                );
                const publishableKey = data.publishableKey;

                if (!publishableKey) {
                    throw new Error("Stripe publishable key is missing");
                }

                setStripePromise(loadStripe(publishableKey));
            } catch (err) {
                console.error("Error loading Stripe:", err);
                setError(
                    "Failed to load Stripe. Please refresh the page or try another payment method."
                );
            }
        };

        loadStripeKey();
    }, []);

    if (error) {
        return (
            <div className="p-4 bg-red-100 dark:bg-red-900 text-red-700 dark:text-red-300 rounded-lg flex items-start space-x-2">
                <AlertCircle className="w-5 h-5 mt-0.5 flex-shrink-0" />
                <span>{error}</span>
            </div>
        );
    }

    if (!stripePromise) {
        return (
            <div className="flex items-center justify-center p-4 text-gray-600 dark:text-gray-300">
                <Loader2 className="h-5 w-5 animate-spin mr-2" />
                <span>Loading payment processor...</span>
            </div>
        );
    }

    return (
        <div className="space-y-4">
            <h3 className="text-lg font-bold mb-4 text-gray-800 dark:text-gray-100">
                Pay with Card
            </h3>

            <Elements stripe={stripePromise}>
                <StripeCheckoutForm {...props} />
            </Elements>
        </div>
    );
};

export default StripePaymentForm;
