import { Head } from "@inertiajs/react";

export default function StructuredData({ pageData = {} }) {
    // Simple destructuring - no transformation
    const {
        title,
        description,
        keywords,
        canonicalUrl,
        ogImage,
        ogImageAlt,
        ogUrl,
        ogType,
        ogSiteName,
        ogLocale,
        twitterCard,
        twitterSite,
        twitterCreator,
        twitterTitle,
        twitterDescription,
        twitterImage,
        twitterImageAlt,
        structuredData,
    } = pageData;

    return (
        <Head>
            {title && <title>{title}</title>}
            {description && <meta name="description" content={description} />}
            {keywords && <meta name="keywords" content={keywords} />}
            {canonicalUrl && <link rel="canonical" href={canonicalUrl} />}
            
            {ogUrl && <meta property="og:url" content={ogUrl} />}
            {ogImage && <meta property="og:image" content={ogImage} />}
            {ogType && <meta property="og:type" content={ogType} />}
            {(twitterTitle || title) && <meta property="og:title" content={twitterTitle || title} />}
            {(twitterDescription || description) && <meta property="og:description" content={twitterDescription || description} />}
            
            {twitterCard && <meta name="twitter:card" content={twitterCard} />}
            {twitterSite && <meta name="twitter:site" content={twitterSite} />}
            {twitterCreator && <meta name="twitter:creator" content={twitterCreator} />}

            {/* Render database structured data - handles multiple formats */}
            {structuredData && typeof structuredData === 'object' && (() => {
                // Handle Schema.org @graph format: {"@context": "...", "@graph": [{...}, {...}]}
                if (structuredData['@context'] && Array.isArray(structuredData['@graph'])) {
                    const context = structuredData['@context'];
                    return structuredData['@graph'].map((schema, index) => {
                        if (!schema || typeof schema !== 'object') return null;
                        try {
                            // Each schema gets its own script tag with the shared context
                            const schemaWithContext = {
                                '@context': context,
                                ...schema
                            };
                            const cleanSchema = JSON.parse(JSON.stringify(schemaWithContext));
                            return (
                                <script
                                    key={`graph-${index}`}
                                    type="application/ld+json"
                                    dangerouslySetInnerHTML={{
                                        __html: JSON.stringify(cleanSchema),
                                    }}
                                />
                            );
                        } catch (e) {
                            console.error('Error rendering @graph schema at index', index, e);
                            return null;
                        }
                    });
                }
                
                // Handle array format: [{...}, {...}, {...}]
                if (Array.isArray(structuredData)) {
                    return structuredData.map((schema, index) => {
                        if (!schema || typeof schema !== 'object') return null;
                        try {
                            const cleanSchema = JSON.parse(JSON.stringify(schema));
                            return (
                                <script
                                    key={`schema-${index}`}
                                    type="application/ld+json"
                                    dangerouslySetInnerHTML={{
                                        __html: JSON.stringify(cleanSchema),
                                    }}
                                />
                            );
                        } catch (e) {
                            console.error('Error rendering schema at index', index, e);
                            return null;
                        }
                    });
                }
                
                // Handle keyed object format: {organization: {...}, webpage: {...}}
                return Object.entries(structuredData).map(([key, schema]) => {
                    if (!schema || typeof schema !== 'object') return null;
                    try {
                        const cleanSchema = JSON.parse(JSON.stringify(schema));
                        return (
                            <script
                                key={key}
                                type="application/ld+json"
                                dangerouslySetInnerHTML={{
                                    __html: JSON.stringify(cleanSchema),
                                }}
                            />
                        );
                    } catch (e) {
                        console.error('Error rendering schema:', key, e);
                        return null;
                    }
                });
            })()}
        </Head>
    );
}
