import { motion } from "framer-motion";
import { useState } from "react";
import { Link } from "@inertiajs/react";
import {
    Star,
    Quote,
    ChevronLeft,
    ChevronRight,
    Award,
    GraduationCap,
    BookOpen,
    Users,
    CheckCircle,
    Heart,
    TrendingUp,
    Globe,
    Calendar,
    User,
} from "lucide-react";

export default function Testimonials({ fadeIn, staggerContainer }) {
    const [activeTab, setActiveTab] = useState(0);
    const [currentTestimonial, setCurrentTestimonial] = useState(0);

    const handleTabClick = (tabIndex, tabName) => {
        setActiveTab(tabIndex);
        if (typeof window.trackTabView === "function") {
            window.trackTabView(tabName, "testimonials_section");
        }
    };

    const handleTestimonialNavigation = (direction) => {
        if (typeof window.trackTestimonialNavigation === "function") {
            window.trackTestimonialNavigation(
                direction,
                "testimonials_carousel"
            );
        }
    };

    const handleCTAClick = (ctaText, section) => {
        if (typeof window.trackCTAClick === "function") {
            window.trackCTAClick(ctaText, "testimonials_section", section);
        }
    };

    const testimonialCategories = [
        {
            name: "Students",
            description: "Undergraduate and graduate students",
            icon: <GraduationCap className="w-5 h-5" />,
            testimonials: [
                {
                    text: "The dissertation writing help I received was exceptional. My committee was impressed with the quality of research and writing. The professional essay writers understood exactly what was needed for my PhD program.",
                    author: "Dr. Sarah Johnson",
                    role: "PhD Graduate - Psychology",
                    university: "Stanford University",
                    grade: "A+",
                    service: "Dissertation Writing Help",
                    rating: 5,
                    date: "March 2024",
                    verified: true,
                    image: "/images/testimonials/sarah-j.jpg",
                },
                {
                    text: "I was struggling with my research paper writing and found their academic writing services through a friend's recommendation. The custom academic paper I received helped me understand proper research methodology and improved my overall academic performance.",
                    author: "Michael Chen",
                    role: "Master's Student - Business Administration",
                    university: "Harvard Business School",
                    grade: "A",
                    service: "Research Paper Writing Service",
                    rating: 5,
                    date: "February 2024",
                    verified: true,
                    image: "/images/testimonials/michael-c.jpg",
                },
                {
                    text: "As an international student, I needed help with academic writing standards. Their professional essay writers not only delivered excellent work but also taught me valuable writing techniques that I use in all my assignments now.",
                    author: "Priya Patel",
                    role: "Undergraduate - Computer Science",
                    university: "MIT",
                    grade: "A-",
                    service: "Custom Academic Papers",
                    rating: 5,
                    date: "January 2024",
                    verified: true,
                    image: "/images/testimonials/priya-p.jpg",
                },
            ],
        },
        {
            name: "Researchers",
            description: "Academic researchers and faculty",
            icon: <BookOpen className="w-5 h-5" />,
            testimonials: [
                {
                    text: "Their research paper writing service helped me publish in a top-tier journal. The writers have deep understanding of academic standards and research methodology. I've recommended their academic writing services to several colleagues.",
                    author: "Prof. David Martinez",
                    role: "Associate Professor - Environmental Science",
                    university: "UC Berkeley",
                    grade: "Published",
                    service: "Research Paper Writing Service",
                    rating: 5,
                    date: "April 2024",
                    verified: true,
                    image: "/images/testimonials/david-m.jpg",
                },
                {
                    text: "The dissertation writing help provided was comprehensive and professional. The writer assigned to my project had extensive knowledge in my field and delivered work that exceeded my expectations. Truly exceptional academic writing services.",
                    author: "Dr. Lisa Thompson",
                    role: "Postdoctoral Researcher - Neuroscience",
                    university: "Johns Hopkins University",
                    grade: "Excellent",
                    service: "Dissertation Writing Help",
                    rating: 5,
                    date: "March 2024",
                    verified: true,
                    image: "/images/testimonials/lisa-t.jpg",
                },
                {
                    text: "Working with their professional essay writers was a game-changer for my research career. The quality of custom academic papers they produce is consistently high, and their understanding of academic requirements is impressive.",
                    author: "Dr. James Wilson",
                    role: "Research Fellow - Economics",
                    university: "Oxford University",
                    grade: "Outstanding",
                    service: "Custom Academic Papers",
                    rating: 5,
                    date: "February 2024",
                    verified: true,
                    image: "/images/testimonials/james-w.jpg",
                },
            ],
        },
        {
            name: "Professionals",
            description: "Working professionals pursuing degrees",
            icon: <Users className="w-5 h-5" />,
            testimonials: [
                {
                    text: "Balancing work and studies was challenging until I found their academic writing services. The research paper writing service helped me maintain my GPA while managing my full-time job. Highly recommend their professional approach.",
                    author: "Amanda Rodriguez",
                    role: "MBA Student & Marketing Director",
                    university: "Wharton School",
                    grade: "A",
                    service: "Research Paper Writing Service",
                    rating: 5,
                    date: "April 2024",
                    verified: true,
                    image: "/images/testimonials/amanda-r.jpg",
                },
                {
                    text: "The dissertation writing help I received was invaluable for completing my executive PhD program. The professional essay writers understood the unique challenges of working professionals and delivered exceptional quality work.",
                    author: "Robert Kim",
                    role: "Executive PhD Student & CEO",
                    university: "Northwestern Kellogg",
                    grade: "A+",
                    service: "Dissertation Writing Help",
                    rating: 5,
                    date: "March 2024",
                    verified: true,
                    image: "/images/testimonials/robert-k.jpg",
                },
                {
                    text: "Their custom academic papers helped me excel in my part-time master's program. The quality and attention to detail in their academic writing services made it possible for me to achieve my educational goals while working full-time.",
                    author: "Jennifer Lee",
                    role: "Master's Student & Financial Analyst",
                    university: "Columbia Business School",
                    grade: "A-",
                    service: "Custom Academic Papers",
                    rating: 5,
                    date: "January 2024",
                    verified: true,
                    image: "/images/testimonials/jennifer-l.jpg",
                },
            ],
        },
    ];

    const overallStats = {
        totalReviews: 1247,
        averageRating: 4.9,
        recommendationRate: 98,
        repeatCustomers: 76,
    };

    const currentTestimonials = testimonialCategories[activeTab].testimonials;

    const nextTestimonial = () => {
        setCurrentTestimonial((prev) =>
            prev === currentTestimonials.length - 1 ? 0 : prev + 1
        );
        handleTestimonialNavigation("next");
    };

    const prevTestimonial = () => {
        setCurrentTestimonial((prev) =>
            prev === 0 ? currentTestimonials.length - 1 : prev - 1
        );
        handleTestimonialNavigation("previous");
    };

    const renderStars = (rating) => {
        return Array.from({ length: 5 }, (_, i) => (
            <Star
                key={i}
                className={`w-5 h-5 ${
                    i < rating
                        ? "text-yellow-400 fill-current"
                        : "text-gray-300 dark:text-gray-600"
                }`}
            />
        ));
    };

    return (
        <>
            {/* Main Testimonials Section */}
            <section
                id="testimonials"
                className="py-20 bg-gradient-to-br from-gray-50 to-blue-50 dark:from-gray-900 dark:to-gray-800 w-full"
            >
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    {/* Section Header */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="text-center mb-16"
                    >
                        <h2 className="text-4xl md:text-5xl font-bold text-gray-900 dark:text-white mb-6">
                            What Our Clients Say About Our Academic Writing
                            Services
                        </h2>
                        <div className="mt-4 h-1 w-24 bg-gradient-to-r from-blue-600 to-purple-600 mx-auto rounded-full"></div>
                        <p className="mt-6 max-w-4xl mx-auto text-xl text-gray-600 dark:text-gray-300 leading-relaxed">
                            Discover why students, researchers, and
                            professionals trust our
                            <strong> professional essay writers</strong> and
                            comprehensive
                            <strong> academic writing services</strong>. From{" "}
                           <a href="https://essaypro.com/custom-dissertation-writing-service"><strong>dissertation writing help</strong></a>  to
                            <strong> research paper writing service</strong>,
                            our <strong>custom academic papers</strong> 
                             have helped thousands achieve academic excellence.
                        </p>
                    </motion.div>

                    {/* Overall Statistics */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="mb-16"
                    >
                        <div className="bg-white dark:bg-gray-800 rounded-2xl shadow-xl p-8 border border-gray-200 dark:border-gray-700">
                            <h3 className="text-2xl font-bold text-gray-900 dark:text-white text-center mb-8">
                                Trusted Academic Writing Services - Client
                                Satisfaction Overview
                            </h3>
                            <div className="grid grid-cols-2 md:grid-cols-4 gap-8">
                                <div className="text-center">
                                    <div className="flex justify-center mb-3">
                                        <Star className="w-12 h-12 text-yellow-400 fill-current" />
                                    </div>
                                    <div className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                                        {overallStats.averageRating}/5
                                    </div>
                                    <div className="text-sm text-gray-600 dark:text-gray-300">
                                        Average Rating
                                    </div>
                                </div>
                                <div className="text-center">
                                    <div className="flex justify-center mb-3">
                                        <Users className="w-12 h-12 text-blue-500" />
                                    </div>
                                    <div className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                                        {overallStats.totalReviews.toLocaleString()}
                                        +
                                    </div>
                                    <div className="text-sm text-gray-600 dark:text-gray-300">
                                        Verified Reviews
                                    </div>
                                </div>
                                <div className="text-center">
                                    <div className="flex justify-center mb-3">
                                        <Heart className="w-12 h-12 text-red-500" />
                                    </div>
                                    <div className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                                        {overallStats.recommendationRate}%
                                    </div>
                                    <div className="text-sm text-gray-600 dark:text-gray-300">
                                        Recommend Us
                                    </div>
                                </div>
                                <div className="text-center">
                                    <div className="flex justify-center mb-3">
                                        <TrendingUp className="w-12 h-12 text-green-500" />
                                    </div>
                                    <div className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                                        {overallStats.repeatCustomers}%
                                    </div>
                                    <div className="text-sm text-gray-600 dark:text-gray-300">
                                        Repeat Clients
                                    </div>
                                </div>
                            </div>
                        </div>
                    </motion.div>

                    {/* Category Tabs */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="flex justify-center mb-12"
                    >
                        <div className="flex rounded-xl bg-white dark:bg-gray-800 p-2 shadow-lg border border-gray-200 dark:border-gray-700">
                            {testimonialCategories.map((category, idx) => (
                                <button
                                    key={idx}
                                    className={`flex items-center px-6 py-3 rounded-lg font-medium transition-all duration-300 ${
                                        activeTab === idx
                                            ? "bg-gradient-to-r from-blue-600 to-purple-600 text-white shadow-md transform scale-105"
                                            : "text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white hover:bg-gray-50 dark:hover:bg-gray-700"
                                    }`}
                                    onClick={() =>
                                        handleTabClick(idx, category.name)
                                    }
                                >
                                    {category.icon}
                                    <span className="ml-2">
                                        {category.name}
                                    </span>
                                </button>
                            ))}
                        </div>
                    </motion.div>

                    {/* Featured Testimonial Carousel */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="mb-16"
                    >
                        <div className="relative bg-white dark:bg-gray-800 rounded-2xl shadow-2xl overflow-hidden border border-gray-200 dark:border-gray-700">
                            <div className="absolute top-0 left-0 w-full h-2 bg-gradient-to-r from-blue-600 to-purple-600"></div>

                            <div className="p-8 md:p-12">
                                <div className="flex items-center justify-between mb-8">
                                    <div className="flex items-center space-x-4">
                                        <div className="flex items-center space-x-1">
                                            {renderStars(
                                                currentTestimonials[
                                                    currentTestimonial
                                                ].rating
                                            )}
                                        </div>
                                        {currentTestimonials[currentTestimonial]
                                            .verified && (
                                            <div className="flex items-center text-green-600 dark:text-green-400">
                                                <CheckCircle className="w-5 h-5 mr-1" />
                                                <span className="text-sm font-medium">
                                                    Verified Review
                                                </span>
                                            </div>
                                        )}
                                    </div>
                                    <div className="flex items-center space-x-2">
                                        <button
                                            onClick={prevTestimonial}
                                            className="p-2 rounded-full bg-gray-100 dark:bg-gray-700 hover:bg-gray-200 dark:hover:bg-gray-600 transition-colors duration-200"
                                            aria-label="Previous testimonial"
                                        >
                                            <ChevronLeft className="w-5 h-5 text-gray-600 dark:text-gray-300" />
                                        </button>
                                        <button
                                            onClick={nextTestimonial}
                                            className="p-2 rounded-full bg-gray-100 dark:bg-gray-700 hover:bg-gray-200 dark:hover:bg-gray-600 transition-colors duration-200"
                                            aria-label="Next testimonial"
                                        >
                                            <ChevronRight className="w-5 h-5 text-gray-600 dark:text-gray-300" />
                                        </button>
                                    </div>
                                </div>

                                <div className="grid md:grid-cols-3 gap-8 items-center">
                                    {/* Testimonial Content */}
                                    <div className="md:col-span-2">
                                        <Quote className="w-12 h-12 text-blue-600 dark:text-blue-400 mb-4" />
                                        <blockquote className="text-xl md:text-2xl text-gray-700 dark:text-gray-300 leading-relaxed mb-6 font-medium">
                                            "
                                            {
                                                currentTestimonials[
                                                    currentTestimonial
                                                ].text
                                            }
                                            "
                                        </blockquote>

                                        <div className="space-y-2">
                                            <div className="flex items-center space-x-3">
                                                <User className="w-5 h-5 text-gray-500 dark:text-gray-400" />
                                                <span className="font-bold text-gray-900 dark:text-white text-lg">
                                                    {
                                                        currentTestimonials[
                                                            currentTestimonial
                                                        ].author
                                                    }
                                                </span>
                                            </div>
                                            <div className="flex items-center space-x-3">
                                                <GraduationCap className="w-5 h-5 text-gray-500 dark:text-gray-400" />
                                                <span className="text-gray-600 dark:text-gray-300">
                                                    {
                                                        currentTestimonials[
                                                            currentTestimonial
                                                        ].role
                                                    }
                                                </span>
                                            </div>
                                            <div className="flex items-center space-x-3">
                                                <Globe className="w-5 h-5 text-gray-500 dark:text-gray-400" />
                                                <span className="text-gray-600 dark:text-gray-300">
                                                    {
                                                        currentTestimonials[
                                                            currentTestimonial
                                                        ].university
                                                    }
                                                </span>
                                            </div>
                                            <div className="flex items-center space-x-3">
                                                <Calendar className="w-5 h-5 text-gray-500 dark:text-gray-400" />
                                                <span className="text-gray-600 dark:text-gray-300">
                                                    {
                                                        currentTestimonials[
                                                            currentTestimonial
                                                        ].date
                                                    }
                                                </span>
                                            </div>
                                        </div>
                                    </div>

                                    {/* Client Info Card */}
                                    <div className="bg-gradient-to-br from-blue-50 to-purple-50 dark:from-gray-700 dark:to-gray-600 rounded-xl p-6">
                                        <div className="text-center">
                                            <div className="w-20 h-20 bg-gradient-to-r from-blue-600 to-purple-600 rounded-full flex items-center justify-center mx-auto mb-4">
                                                <User className="w-10 h-10 text-white" />
                                            </div>
                                            <h4 className="font-bold text-gray-900 dark:text-white mb-2">
                                                Service Used
                                            </h4>
                                            <p className="text-blue-600 dark:text-blue-400 font-medium mb-4">
                                                {
                                                    currentTestimonials[
                                                        currentTestimonial
                                                    ].service
                                                }
                                            </p>

                                            <div className="space-y-3">
                                                <div className="flex justify-between items-center">
                                                    <span className="text-gray-600 dark:text-gray-300">
                                                        Grade Achieved:
                                                    </span>
                                                    <span className="font-bold text-green-600 dark:text-green-400">
                                                        {
                                                            currentTestimonials[
                                                                currentTestimonial
                                                            ].grade
                                                        }
                                                    </span>
                                                </div>
                                                <div className="flex justify-between items-center">
                                                    <span className="text-gray-600 dark:text-gray-300">
                                                        Rating:
                                                    </span>
                                                    <div className="flex items-center space-x-1">
                                                        {renderStars(
                                                            currentTestimonials[
                                                                currentTestimonial
                                                            ].rating
                                                        )}
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                {/* Testimonial Navigation Dots */}
                                <div className="flex justify-center mt-8 space-x-2">
                                    {currentTestimonials.map((_, index) => (
                                        <button
                                            key={index}
                                            onClick={() =>
                                                setCurrentTestimonial(index)
                                            }
                                            className={`w-3 h-3 rounded-full transition-all duration-300 ${
                                                index === currentTestimonial
                                                    ? "bg-blue-600 dark:bg-blue-400 scale-125"
                                                    : "bg-gray-300 dark:bg-gray-600 hover:bg-gray-400 dark:hover:bg-gray-500"
                                            }`}
                                            aria-label={`Go to testimonial ${
                                                index + 1
                                            }`}
                                        />
                                    ))}
                                </div>
                            </div>
                        </div>
                    </motion.div>

                    {/* All Testimonials Grid */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={staggerContainer}
                        className="mb-16"
                    >
                        <h3 className="text-3xl font-bold text-gray-900 dark:text-white text-center mb-12">
                            More Success Stories from Our{" "}
                            {testimonialCategories[activeTab].name}
                        </h3>
                        <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
                            {currentTestimonials.map((testimonial, index) => (
                                <motion.div
                                    key={index}
                                    variants={fadeIn}
                                    className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-lg hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1 border border-gray-200 dark:border-gray-700"
                                >
                                    {/* Header */}
                                    <div className="flex items-center justify-between mb-4">
                                        <div className="flex items-center space-x-1">
                                            {renderStars(testimonial.rating)}
                                        </div>
                                        {testimonial.verified && (
                                            <div className="flex items-center text-green-600 dark:text-green-400">
                                                <CheckCircle className="w-4 h-4 mr-1" />
                                                <span className="text-xs font-medium">
                                                    Verified
                                                </span>
                                            </div>
                                        )}
                                    </div>

                                    {/* Quote */}
                                    <Quote className="w-8 h-8 text-blue-600 dark:text-blue-400 mb-3" />
                                    <blockquote className="text-gray-700 dark:text-gray-300 mb-4 leading-relaxed">
                                        "
                                        {testimonial.text.length > 150
                                            ? testimonial.text.substring(
                                                  0,
                                                  150
                                              ) + "..."
                                            : testimonial.text}
                                        "
                                    </blockquote>

                                    {/* Author Info */}
                                    <div className="border-t border-gray-200 dark:border-gray-700 pt-4">
                                        <div className="flex items-center space-x-3 mb-2">
                                            <div className="w-10 h-10 bg-gradient-to-r from-blue-600 to-purple-600 rounded-full flex items-center justify-center">
                                                <User className="w-5 h-5 text-white" />
                                            </div>
                                            <div>
                                                <h4 className="font-bold text-gray-900 dark:text-white">
                                                    {testimonial.author}
                                                </h4>
                                                <p className="text-sm text-gray-600 dark:text-gray-300">
                                                    {testimonial.role}
                                                </p>
                                            </div>
                                        </div>

                                        <div className="space-y-1 text-sm">
                                            <div className="flex items-center justify-between">
                                                <span className="text-gray-500 dark:text-gray-400">
                                                    Service:
                                                </span>
                                                <span className="text-blue-600 dark:text-blue-400 font-medium">
                                                    {testimonial.service}
                                                </span>
                                            </div>
                                            <div className="flex items-center justify-between">
                                                <span className="text-gray-500 dark:text-gray-400">
                                                    Grade:
                                                </span>
                                                <span className="text-green-600 dark:text-green-400 font-bold">
                                                    {testimonial.grade}
                                                </span>
                                            </div>
                                            <div className="flex items-center justify-between">
                                                <span className="text-gray-500 dark:text-gray-400">
                                                    Date:
                                                </span>
                                                <span className="text-gray-600 dark:text-gray-300">
                                                    {testimonial.date}
                                                </span>
                                            </div>
                                        </div>
                                    </div>
                                </motion.div>
                            ))}
                        </div>
                    </motion.div>

                    {/* Service-Specific Testimonial Highlights */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="mb-16"
                    >
                        <h3 className="text-3xl font-bold text-gray-900 dark:text-white text-center mb-12">
                            Academic Writing Services Success by Category
                        </h3>
                        <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
                            {[
                                {
                                    service: "Dissertation Writing Help",
                                    satisfaction: "97%",
                                    avgGrade: "A+",
                                    completionRate: "99.2%",
                                    icon: (
                                        <GraduationCap className="w-8 h-8 text-purple-600" />
                                    ),
                                    color: "from-purple-500 to-indigo-600",
                                    testimonialCount: 245,
                                },
                                {
                                    service: "Research Paper Writing Service",
                                    satisfaction: "96%",
                                    avgGrade: "A",
                                    completionRate: "98.8%",
                                    icon: (
                                        <BookOpen className="w-8 h-8 text-blue-600" />
                                    ),
                                    color: "from-blue-500 to-cyan-600",
                                    testimonialCount: 387,
                                },
                                {
                                    service: "Custom Academic Papers",
                                    satisfaction: "98%",
                                    avgGrade: "A-",
                                    completionRate: "99.5%",
                                    icon: (
                                        <Award className="w-8 h-8 text-green-600" />
                                    ),
                                    color: "from-green-500 to-emerald-600",
                                    testimonialCount: 615,
                                },
                            ].map((service, index) => (
                                <div
                                    key={index}
                                    className="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-lg border border-gray-200 dark:border-gray-700"
                                >
                                    <div
                                        className={`bg-gradient-to-r ${service.color} p-4 rounded-lg mb-6`}
                                    >
                                        <div className="flex items-center justify-between text-white">
                                            {service.icon}
                                            <span className="text-2xl font-bold">
                                                {service.satisfaction}
                                            </span>
                                        </div>
                                    </div>

                                    <h4 className="text-xl font-bold text-gray-900 dark:text-white mb-4">
                                        {service.service}
                                    </h4>

                                    <div className="space-y-3">
                                        <div className="flex justify-between items-center">
                                            <span className="text-gray-600 dark:text-gray-300">
                                                Client Satisfaction:
                                            </span>
                                            <span className="font-bold text-green-600 dark:text-green-400">
                                                {service.satisfaction}
                                            </span>
                                        </div>
                                        <div className="flex justify-between items-center">
                                            <span className="text-gray-600 dark:text-gray-300">
                                                Average Grade:
                                            </span>
                                            <span className="font-bold text-blue-600 dark:text-blue-400">
                                                {service.avgGrade}
                                            </span>
                                        </div>
                                        <div className="flex justify-between items-center">
                                            <span className="text-gray-600 dark:text-gray-300">
                                                Completion Rate:
                                            </span>
                                            <span className="font-bold text-purple-600 dark:text-purple-400">
                                                {service.completionRate}
                                            </span>
                                        </div>
                                        <div className="flex justify-between items-center">
                                            <span className="text-gray-600 dark:text-gray-300">
                                                Reviews:
                                            </span>
                                            <span className="font-bold text-gray-900 dark:text-white">
                                                {service.testimonialCount}+
                                            </span>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    </motion.div>

                    {/* Trust Indicators */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="mb-16"
                    >
                        <div className="bg-gradient-to-r from-blue-600 to-purple-600 rounded-2xl p-8 text-white">
                            <h3 className="text-3xl font-bold text-center mb-8">
                                Why Students Trust Our Academic Writing Services
                            </h3>
                            <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
                                {[
                                    {
                                        icon: (
                                            <CheckCircle className="w-8 h-8" />
                                        ),
                                        title: "Verified Reviews",
                                        description:
                                            "All testimonials verified through our secure review system",
                                    },
                                    {
                                        icon: <Award className="w-8 h-8" />,
                                        title: "Grade Guarantee",
                                        description:
                                            "92% of students achieve A+ grades with our help",
                                    },
                                    {
                                        icon: <Users className="w-8 h-8" />,
                                        title: "Expert Writers",
                                        description:
                                            "PhD-level professional essay writers in every field",
                                    },
                                    {
                                        icon: <Heart className="w-8 h-8" />,
                                        title: "Student Success",
                                        description:
                                            "98% client satisfaction rate across all services",
                                    },
                                ].map((feature, index) => (
                                    <div key={index} className="text-center">
                                        <div className="flex justify-center mb-4">
                                            {feature.icon}
                                        </div>
                                        <h4 className="font-bold mb-2">
                                            {feature.title}
                                        </h4>
                                        <p className="text-blue-100 text-sm">
                                            {feature.description}
                                        </p>
                                    </div>
                                ))}
                            </div>
                        </div>
                    </motion.div>

                    {/* Call to Action */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="text-center"
                    >
                        <div className="bg-white dark:bg-gray-800 rounded-2xl p-8 md:p-12 shadow-xl border border-gray-200 dark:border-gray-700">
                            <h3 className="text-3xl md:text-4xl font-bold text-gray-900 dark:text-white mb-6">
                                Ready to Join Our Success Stories?
                            </h3>
                            <p className="text-xl text-gray-600 dark:text-gray-300 mb-8 max-w-3xl mx-auto leading-relaxed">
                                Experience the same exceptional results as our
                                satisfied clients. Our{" "}
                                <strong>professional essay writers</strong> and
                                comprehensive
                                <strong> academic writing services</strong> are
                                ready to help you achieve academic excellence.
                            </p>

                            <div className="flex flex-col sm:flex-row justify-center space-y-4 sm:space-y-0 sm:space-x-6 mb-8">
                                <Link
                                    href="/place-order"
                                    className="bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white px-8 py-4 rounded-lg font-bold transition-all duration-300 transform hover:scale-105 shadow-lg"
                                    onClick={() =>
                                        handleCTAClick(
                                            "Start Your Success Story",
                                            "main_cta"
                                        )
                                    }
                                >
                                    Start Your Success Story
                                </Link>
                                <Link
                                    href="/reviews"
                                    className="border-2 border-blue-600 text-blue-600 dark:text-blue-400 hover:bg-blue-600 hover:text-white dark:hover:bg-blue-600 dark:hover:text-white px-8 py-4 rounded-lg font-bold transition-all duration-300 transform hover:scale-105"
                                    onClick={() =>
                                        handleCTAClick(
                                            "View our Reviews",
                                            "reviews_cta"
                                        )
                                    }
                                >
                                    View More reviews
                                </Link>
                            </div>

                            {/* Additional Trust Elements */}
                            <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm text-gray-600 dark:text-gray-300">
                                <div className="flex items-center justify-center">
                                    <CheckCircle className="w-4 h-4 mr-2 text-green-500" />
                                    <span>Money-Back Guarantee</span>
                                </div>
                                <div className="flex items-center justify-center">
                                    <CheckCircle className="w-4 h-4 mr-2 text-green-500" />
                                    <span>100% Original Work</span>
                                </div>
                                <div className="flex items-center justify-center">
                                    <CheckCircle className="w-4 h-4 mr-2 text-green-500" />
                                    <span>24/7 Support</span>
                                </div>
                                <div className="flex items-center justify-center">
                                    <CheckCircle className="w-4 h-4 mr-2 text-green-500" />
                                    <span>Unlimited Revisions</span>
                                </div>
                            </div>
                        </div>
                    </motion.div>
                </div>
            </section>
        </>
    );
}
