import React from 'react';
import { Clock, AlertTriangle, AlertCircle, AlertOctagon, ArrowRight } from 'lucide-react';
import { Link } from '@inertiajs/react';
import PropTypes from 'prop-types';
import StatusBadge from './StatusBadge';

/**
 * Elegant Urgent Orders Summary Component
 * Clean, minimal design with color-coded urgency levels
 */
const UrgentOrdersSummary = ({ urgentOrders, urgentOrdersCount, userType = 'admin' }) => {
    const formatTimeRemaining = (hours) => {
        if (hours === undefined || hours === null || isNaN(hours)) {
            return 'Unknown';
        }
        
        if (hours < 0) {
            const overdueHours = Math.abs(hours);
            if (overdueHours < 24) {
                return `${Math.round(overdueHours)}h overdue`;
            }
            const overdueDays = Math.floor(overdueHours / 24);
            return `${overdueDays}d overdue`;
        }
        if (hours < 1) {
            const minutes = Math.round(hours * 60);
            return `${minutes}m`;
        }
        if (hours < 24) {
            return `${Math.round(hours)}h`;
        }
        const days = Math.floor(hours / 24);
        return `${days}d`;
    };

    // Get top 5 most urgent orders
    const getMostUrgentOrders = () => {
        const allOrders = [];
        
        // Add overdue orders first (highest priority)
        if (urgentOrders.overdue && Array.isArray(urgentOrders.overdue)) {
            urgentOrders.overdue.forEach(order => {
                allOrders.push({ ...order, urgency: 'overdue', priority: 1 });
            });
        }
        
        // Add critical orders
        if (urgentOrders.critical_12h && Array.isArray(urgentOrders.critical_12h)) {
            urgentOrders.critical_12h.forEach(order => {
                allOrders.push({ ...order, urgency: 'critical', priority: 2 });
            });
        }
        
        // Add urgent orders
        if (urgentOrders.urgent_24h && Array.isArray(urgentOrders.urgent_24h)) {
            urgentOrders.urgent_24h.forEach(order => {
                allOrders.push({ ...order, urgency: 'urgent', priority: 3 });
            });
        }
        
        // Sort by priority and get top 5
        return allOrders
            .sort((a, b) => a.priority - b.priority || (a.hours_remaining || 0) - (b.hours_remaining || 0))
            .slice(0, 5);
    };

    const topOrders = getMostUrgentOrders();
    const hasUrgentOrders = topOrders.length > 0;

    // Urgency level config
    const urgencyLevels = [
        {
            key: 'overdue',
            label: 'Overdue',
            icon: AlertOctagon,
            color: 'red',
            borderColor: 'border-red-500',
            bgColor: 'bg-red-50',
            textColor: 'text-red-900',
            iconColor: 'text-red-600',
            count: urgentOrdersCount?.overdue || 0
        },
        {
            key: 'critical',
            label: 'Critical',
            icon: AlertOctagon,
            color: 'orange',
            borderColor: 'border-orange-500',
            bgColor: 'bg-orange-50',
            textColor: 'text-orange-900',
            iconColor: 'text-orange-600',
            count: urgentOrdersCount?.critical_12h || 0
        },
        {
            key: 'urgent',
            label: 'Urgent',
            icon: AlertTriangle,
            color: 'yellow',
            borderColor: 'border-yellow-500',
            bgColor: 'bg-yellow-50',
            textColor: 'text-yellow-900',
            iconColor: 'text-yellow-600',
            count: urgentOrdersCount?.urgent_24h || 0
        },
        {
            key: 'warning',
            label: 'Warning',
            icon: AlertCircle,
            color: 'blue',
            borderColor: 'border-blue-500',
            bgColor: 'bg-blue-50',
            textColor: 'text-blue-900',
            iconColor: 'text-blue-600',
            count: urgentOrdersCount?.warning_48h || 0
        }
    ];

    const getUrgencyDotColor = (urgency) => {
        switch (urgency) {
            case 'overdue': return 'bg-red-500';
            case 'critical': return 'bg-orange-500';
            case 'urgent': return 'bg-yellow-500';
            default: return 'bg-blue-500';
        }
    };

    if (!hasUrgentOrders) {
        return (
            <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700 p-8">
                <div className="text-center text-gray-500 dark:text-gray-400">
                    <Clock className="w-16 h-16 mx-auto mb-4 text-gray-300 dark:text-gray-600" />
                    <h3 className="text-lg font-medium text-gray-700 dark:text-gray-300 mb-2">All caught up!</h3>
                    <p className="text-sm">No urgent orders at the moment</p>
                </div>
            </div>
        );
    }

    return (
        <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700">
            {/* Urgency Summary Cards */}
            <div className="p-6 border-b border-gray-200 dark:border-gray-700">
                <div className="grid grid-cols-2 lg:grid-cols-4 gap-4">
                    {urgencyLevels.map((level) => (
                        <div
                            key={level.key}
                            className={`
                                ${level.bgColor} dark:bg-gray-700
                                border-l-4 ${level.borderColor}
                                rounded-lg p-4
                                hover:shadow-md transition-all duration-200
                                cursor-pointer
                            `}
                        >
                            <div className="flex items-center justify-between mb-2">
                                <level.icon className={`h-5 w-5 ${level.iconColor}`} />
                                <span className={`text-3xl font-bold ${level.textColor} dark:text-white`}>
                                    {level.count}
                                </span>
                            </div>
                            <div className={`text-xs font-medium ${level.textColor} dark:text-gray-300 uppercase tracking-wide`}>
                                {level.label}
                            </div>
                        </div>
                    ))}
                </div>
            </div>

            {/* Top Urgent Orders List */}
            <div className="p-6">
                <div className="flex items-center justify-between mb-4">
                    <h4 className="text-sm font-semibold text-gray-700 dark:text-gray-300 uppercase tracking-wide">
                        Most Urgent ({topOrders.length})
                    </h4>
                    <Link
                        href={userType === 'writer' ? '/writer/orders?filter=urgent' : '/admin/orders?filter=urgent'}
                        className="text-sm text-blue-600 dark:text-blue-400 hover:text-blue-700 dark:hover:text-blue-300 font-medium flex items-center gap-1 group"
                    >
                        View All
                        <ArrowRight className="h-4 w-4 group-hover:translate-x-1 transition-transform" />
                    </Link>
                </div>

                <div className="space-y-2">
                    {topOrders.map((order, index) => (
                        <div
                            key={order.id || index}
                            className="flex items-center justify-between p-3 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors border border-transparent hover:border-gray-200 dark:hover:border-gray-600"
                        >
                            {/* Left: Order Info */}
                            <div className="flex items-center gap-3 flex-1">
                                {/* Urgency Dot */}
                                <div className={`
                                    w-2 h-2 rounded-full ${getUrgencyDotColor(order.urgency)}
                                    animate-pulse
                                `}></div>

                                {/* Order Details */}
                                <div className="flex-1 min-w-0">
                                    <div className="flex items-center gap-2 mb-1">
                                        <Link
                                            href={userType === 'writer' ? `/writer/orders/${order.id}` : `/admin/orders/${order.id}`}
                                            className="text-sm font-semibold text-blue-600 dark:text-blue-400 hover:text-blue-700 dark:hover:text-blue-300 hover:underline"
                                        >
                                            #{order.order_number || 'N/A'}
                                        </Link>
                                        <span className="text-xs text-gray-500 dark:text-gray-400 truncate">
                                            {order.title || 'Untitled'}
                                        </span>
                                    </div>
                                    <div className="text-xs text-gray-500 dark:text-gray-400">
                                        {order.user?.name || 'Unknown Client'}
                                    </div>
                                </div>
                            </div>

                            {/* Right: Time Remaining */}
                            <div className="flex items-center gap-3">
                                <div className="text-right">
                                    <div className={`
                                        text-sm font-bold
                                        ${order.urgency === 'overdue' ? 'text-red-600 dark:text-red-400' : ''}
                                        ${order.urgency === 'critical' ? 'text-orange-600 dark:text-orange-400' : ''}
                                        ${order.urgency === 'urgent' ? 'text-yellow-600 dark:text-yellow-400' : ''}
                                        ${order.urgency === 'warning' ? 'text-blue-600 dark:text-blue-400' : ''}
                                    `}>
                                        {formatTimeRemaining(order.hours_remaining)}
                                    </div>
                                    {order.deadline && (
                                        <div className="text-xs text-gray-400 dark:text-gray-500">
                                            {new Date(order.deadline).toLocaleDateString('en-US', { 
                                                month: 'short', 
                                                day: 'numeric',
                                                hour: '2-digit',
                                                minute: '2-digit'
                                            })}
                                        </div>
                                    )}
                                </div>
                            </div>
                        </div>
                    ))}
                </div>

                {/* No orders message */}
                {topOrders.length === 0 && (
                    <div className="text-center py-8 text-gray-500 dark:text-gray-400">
                        <Clock className="w-12 h-12 mx-auto mb-3 text-gray-300 dark:text-gray-600" />
                        <p className="text-sm">No urgent orders at the moment</p>
                    </div>
                )}
            </div>
        </div>
    );
};

UrgentOrdersSummary.propTypes = {
    urgentOrders: PropTypes.shape({
        overdue: PropTypes.array,
        critical_12h: PropTypes.array,
        urgent_24h: PropTypes.array,
        warning_48h: PropTypes.array,
    }),
    urgentOrdersCount: PropTypes.shape({
        overdue: PropTypes.number,
        critical_12h: PropTypes.number,
        urgent_24h: PropTypes.number,
        warning_48h: PropTypes.number,
    }),
    userType: PropTypes.oneOf(['admin', 'writer']),
};

export default UrgentOrdersSummary;

